
import os
import sys
import random
import subprocess
import time
import glob
import shutil
from dataclasses import dataclass
from typing import List, Dict, Optional, Tuple
from pathlib import Path

####################
# Configuration
####################
@dataclass
class PlannerConfig:
    """Configuration for the Fast Downward planner"""
    domain_file: str
    downward_path: str = 'submodule/downward/fast-downward.py'
    timeout: int = 300
    planner_alias: str = 'seq-sat-lama-2011'

####################
# Planner Interface
####################
def generate_plan(config: PlannerConfig,
                 problem_file: str,
                 output_file: str) -> bool:
    """
    Generates a plan using Fast Downward planner.
    
    Args:
        config: PlannerConfig object containing planner settings
        problem_file: Path to the PDDL problem file
        output_file: Desired path for the solution file
    
    Returns:
        bool: True if plan generation succeeded, False otherwise
    
    The function:
    1. Uses --plan-file to generate intermediate files (e.g., "problem_sas_plan")
    2. Moves the first solution to output_file and cleans up other plans
    """
    problem_prefix = Path(problem_file).stem
    plan_base = f"{problem_prefix}_sas_plan"

    print(f"\n[generate_plan] Solving: {problem_file}")
    print(f"  Domain file: {config.domain_file}")
    print(f"  Planner script: {config.downward_path}")
    print(f"  Intermediate plan prefix: {plan_base}")
    print(f"  Final output: {output_file}")

    # Cleanup old plan files
    for old_file in glob.glob(f"{plan_base}*"):
        os.remove(old_file)

    # Construct planner command
    cmd = [
        "python3", config.downward_path,
        "--alias", config.planner_alias,
        "--plan-file", plan_base,
        config.domain_file,
        problem_file
    ]

    start_time = time.time()
    print("  Fast Downward command:", " ".join(cmd))

    try:
        # Run planner process
        with subprocess.Popen(cmd, 
                            stdout=subprocess.PIPE, 
                            stderr=subprocess.PIPE, 
                            text=True) as process:
            while True:
                retcode = process.poll()
                if line := process.stdout.readline():
                    print("FD>", line.rstrip())

                if retcode is not None:
                    print(f"[FD] Process finished with return code: {retcode}")
                    break

                if time.time() - start_time > config.timeout:
                    process.terminate()
                    print("Planner timed out")
                    return False

        # Handle generated plans
        if new_plans := glob.glob(f"{plan_base}*"):
            best_plan = new_plans[0]
            os.makedirs(os.path.dirname(output_file), exist_ok=True)
            shutil.move(best_plan, output_file)
            
            # Cleanup additional plans
            for other_plan in new_plans[1:]:
                os.remove(other_plan)
            
            print(f"Solution saved to: {output_file}")
            return True
        
        print(f"No solution files generated: {plan_base}")
        return False

    except Exception as e:
        print(f"Planner error: {e}")
        return False



class FloorTileProblem:
    """
    Handles the generation of PDDL problem components for the floor-tile domain.
    Encapsulates the logic for objects, initial state, and goal generation.
    """
    def __init__(self, num_rows: int, num_columns: int, num_robots: int):
        self.num_rows = num_rows
        self.num_columns = num_columns
        self.num_robots = num_robots
        self.robot_locations = []

    def get_objects(self) -> str:
        """Generates the :objects section of the PDDL problem"""
        # Generate tile objects
        tiles = []
        for row in range(self.num_rows + 1):
            row_tiles = [f"tile_{row}-{col+1}" for col in range(self.num_columns)]
            tiles.append(" ".join(row_tiles))
        tiles_str = "\n           ".join(tiles) + " - tile\n"

        # Generate robot objects
        robots_str = "           " + " ".join(f"robot{i+1}" for i in range(self.num_robots)) + " - robot\n"

        # Colors are fixed
        colors_str = "           white black - color\n"

        return tiles_str + robots_str + colors_str

    def get_init(self, mode_flag: str) -> str:
        """
        Generates the :init section of the PDDL problem
        
        Args:
            mode_flag: Determines cost metric ("time" or "seq")
        """
        init_facts = []
        self.robot_locations = []

        # Add cost/time metric
        if mode_flag != "time":
            init_facts.append("   (= (total-cost) 0)")

        # Place robots and assign colors
        used_cols = list(range(1, self.num_columns + 1))
        for robot in range(self.num_robots):
            posx = random.randint(0, self.num_rows)
            posy = random.choice(used_cols) if used_cols else 1
            used_cols.remove(posy) if posy in used_cols else None

            robot_pos = f"tile_{posx}-{posy}"
            self.robot_locations.append(robot_pos)
            
            init_facts.extend([
                f"   (robot-at robot{robot+1} {robot_pos})",
                f"   (robot-has robot{robot+1} {'white' if robot % 2 == 0 else 'black'})"
            ])

        # Add color availability
        init_facts.extend([
            "   (available-color white)",
            "   (available-color black)"
        ])

        # Add clear tiles
        for row in range(self.num_rows + 1):
            for col in range(self.num_columns):
                tile = f"tile_{row}-{col+1}"
                if tile not in self.robot_locations:
                    init_facts.append(f"   (clear {tile})")

        # Add adjacency relationships
        init_facts.extend(self._generate_adjacency_facts())

        return "\n".join(init_facts)

    def _generate_adjacency_facts(self) -> List[str]:
        """Generates all adjacency relationships (up/down/left/right)"""
        facts = []
        
        # Up/Down relationships
        for row in range(self.num_rows):
            for col in range(self.num_columns):
                above = f"tile_{row+1}-{col+1}"
                below = f"tile_{row}-{col+1}"
                facts.extend([
                    f"   (up {above} {below})",
                    f"   (down {below} {above})"
                ])

        # Left/Right relationships
        for row in range(self.num_rows + 1):
            for col in range(self.num_columns - 1):
                left = f"tile_{row}-{col+1}"
                right = f"tile_{row}-{col+2}"
                facts.extend([
                    f"   (right {right} {left})",
                    f"   (left {left} {right})"
                ])

        return facts

    def get_goals(self) -> str:
        """Generates the :goal section defining the desired tile coloring pattern"""
        goals = ["(and"]
        for r in range(self.num_rows):
            for c in range(self.num_columns):
                color = "white" if ((r + c) % 2 == 0) else "black"
                goals.append(f"    (painted tile_{r+1}-{c+1} {color})")
        goals.append(")")
        return "\n".join(goals)

@dataclass
class GenerationConfig:
    """Configuration for problem generation"""
    rows_range: List[int]
    cols_range: List[int]
    robots_range: List[int]
    mode_range: List[str]
    seeds: List[int]
    problems_dir: Path
    solutions_dir: Path
    domain_file: Path
    downward_path: Path

def generate_problems(config: GenerationConfig) -> None:
    """
    Main problem generation function that creates and solves multiple problem instances
    
    Args:
        config: Generation parameters and paths
    """
    os.makedirs(config.problems_dir, exist_ok=True)
    os.makedirs(config.solutions_dir, exist_ok=True)

    planner_config = PlannerConfig(
        domain_file=str(config.domain_file),
        downward_path=str(config.downward_path)
    )

    problem_prefix = "floor"
    total_problems = (len(config.rows_range) * len(config.cols_range) * 
                     len(config.robots_range) * len(config.mode_range) * 
                     len(config.seeds))
    
    print(f"Generating {total_problems} problems...")

    for r in config.rows_range:
        for c in config.cols_range:
            for rob in config.robots_range:
                for mode in config.mode_range:
                    for seed in config.seeds:
                        problem_name = f"{problem_prefix}_r{r}_c{c}_rob{rob}_{mode}_seed{seed}"
                        
                        # Generate problem instance
                        problem = FloorTileProblem(r, c, rob)
                        pddl_content = (
                            f"(define (problem {problem_name})\n"
                            f" (:domain floor-tile)\n"
                            f" (:objects {problem.get_objects()})\n"
                            f" (:init {problem.get_init(mode)})\n"
                            f" (:goal {problem.get_goals()})\n"
                            f" (:metric minimize ({('total-time' if mode == 'time' else 'total-cost')}))\n"
                            f")\n"
                        )

                        # Save problem file
                        problem_path = config.problems_dir / f"{problem_name}.pddl"
                        solution_path = config.solutions_dir / f"{problem_name}.sol"
                        
                        with open(problem_path, "w", encoding="utf-8") as f:
                            f.write(pddl_content)

                        print(f"[+] Generated problem: {problem_path}")

                        # Generate solution
                        if generate_plan(planner_config, str(problem_path), str(solution_path)):
                            print(f"    => Solution saved to: {solution_path}")
                        else:
                            print(f"    => Failed to generate solution for {problem_path}")
# The default path run this generate_problems file from the root directory of the project. 
def main():
    """Entry point for the problem generator"""
    config = GenerationConfig(
        rows_range=[2, 3, 4],
        cols_range=[2, 3, 4],
        robots_range=[1],  # 1 or 2 
        mode_range=["seq"],
        seeds=list(range(1, 11)),
        problems_dir=Path("final_datasets/floortile/spatial_instances_instances/problems"),
        solutions_dir=Path("final_datasets/floortile/spatial_instances_instances/solutions"),
        domain_file=Path("final_datasets/floortile/domain.pddl"),
        downward_path=Path("submodule/downward/fast-downward.py")
    )
    
    generate_problems(config)

if __name__ == "__main__":
    main()