
import torch
from torch import nn, einsum
import torch.nn.functional as F
import numpy as np

from einops import rearrange, repeat
from einops.layers.torch import Rearrange
#from positional_embedding import PositionalEncoding
import math

# helpers

class Identity(torch.autograd.Function):
  @staticmethod
  def forward(ctx, input):
    return input * 1.0
  def backward(ctx, grad_output):
    print(grad_output)
    print(grad_output.shape)
    print('------')
    return grad_output * 1.0

def pair(t):
    return t if isinstance(t, tuple) else (t, t)

# classes

class PreNorm(nn.Module):
    def __init__(self, dim, fn):
        super().__init__()
        self.norm = nn.LayerNorm(dim)
        self.fn = fn
    def forward(self, x, **kwargs):
        x, _, _ = self.fn(x, **kwargs)
        return self.norm(x)

class PreNormCross(nn.Module):
    def __init__(self, dim, fn):
        super().__init__()
        self.norm_1 = nn.LayerNorm(dim)
        self.norm_2 = nn.LayerNorm(dim)
        self.fn = fn
    def forward(self, x, k, **kwargs):
        return self.fn(x, k)

class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim, dropout = 0.):
        super().__init__()
        self.net = nn.Sequential(
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(hidden_dim, dim),
            nn.Dropout(dropout)
        )
    def forward(self, x):
        return self.net(x)

class CrossAttention(nn.Module):
    def __init__(self, dim, heads = 8, dim_head = 64, dropout = 0.):
        super().__init__()
        inner_dim = dim_head *  heads
        project_out = not (heads == 1 and dim_head == dim)

        self.heads = heads
        self.scale = dim_head ** -0.5

        self.attend = nn.Softmax(dim = -1)

        self.to_q = nn.Linear(dim, inner_dim , bias = True)
        self.to_k = nn.Linear(dim, inner_dim , bias = True)
        self.to_v = nn.Linear(dim, inner_dim , bias = True)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            #nn.Dropout(dropout)
        ) if project_out else nn.Identity()

    def forward(self, q, k):


        b, n_q, d, h = *q.shape, self.heads
        b, n_k, d = k.shape 

        q = self.to_q(q)#.chunk(3, dim = -1)
        k = self.to_k(k)
        v = self.to_v(k)

        #q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h = h), qkv)

        #q_ = q_.reshape(b, n_q, self.heads, -1).permute(0, 2, 1, 3)
        #k_ = k_.reshape(b, n_k, self.heads, -1).permute(0, 2, 1, 3)
        #v_ = v_.reshape(b, n_k, self.heads, -1).permute(0, 2, 1, 3)


        q = rearrange(q, 'b n (h d) -> b h n d', h = h)
        k = rearrange(k, 'b n (h d) -> b h n d', h = h)
        v = rearrange(v, 'b n (h d) -> b h n d', h = h)

        dots = einsum('b h i d, b h j d -> b h i j', q, k) * self.scale

        #dots = torch.matmul(q_, k_.permute(0, 1, 3, 2)) #* self.scale


        #print(dots)
        #exit()

        attn = torch.softmax(dots, dim = -1)
        
        out = torch.matmul(attn, v)
        out = out.permute(0, 2, 1, 3).reshape(b, n_q, d)

        return self.to_out(out)



class Attention(nn.Module):
    def __init__(self, dim, heads = 8, dim_head = 64, dropout = 0.):
        super().__init__()
        inner_dim = dim_head *  heads
        project_out = not (heads == 1 and dim_head == dim)

        self.heads = heads
        self.scale = dim_head ** -0.5

        self.attend = nn.Softmax(dim = -1)
        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias = False)

        self.gru = nn.GRUCell(dim_head, dim_head)

        self.q = nn.Linear(dim, dim_head * heads)
        self.k = nn.Linear(dim, dim_head * heads)
        self.v = nn.Linear(dim, dim_head * heads)


        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            nn.Dropout(dropout)
        ) if project_out else nn.Identity()

    def forward(self, x, attn_workspace = None):
        if attn_workspace == None:
            b, n, _, h = *x.shape, self.heads
            qkv = self.to_qkv(x).chunk(3, dim = -1)
            q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h = h), qkv)

            dots = einsum('b h i d, b h j d -> b h i j', q, k) * self.scale

            attn = self.attend(dots)

            out = einsum('b h i j, b h j d -> b h i d', attn, v)
            out = rearrange(out, 'b h n d -> b n (h d)')
            return self.to_out(out), attn.mean(dim = 1)
        else:
            b, n, d, h = *x.shape, self.heads
            n_w = attn_workspace.shape[1]

            q, k, v = self.q(attn_workspace), self.k(x), self.v(x)

            q = q.reshape(b, n_w, h, -1).permute(0, 2, 1, 3)
            k = k.reshape(b, n, h, -1).permute(0, 2, 1, 3)
            v = v.reshape(b, n, h, -1).permute(0, 2, 1, 3)

            dots = einsum('b h i d, b h j d -> b h i j', q, k) * self.scale
            attn = self.attend(dots)

            out = einsum('b h i j, b h j d -> b h i d', attn, v)

            out = out.reshape(b * h * n_w, -1)
            attn_workspace = attn_workspace.reshape(b, n_w, h, -1).permute(0, 2, 1, 3).reshape(b * n_w * h, -1)

            attn_workspace = self.gru(out, attn_workspace)

            attn_workspace = attn_workspace.reshape(b, n_w, h, -1).permute(0, 2, 1, 3).reshape(b, n_w, -1)

            q, k, v = self.q(x), self.k(attn_workspace), self.v(attn_workspace)

            q = q.reshape(b, n, h, -1).permute(0, 2, 1, 3)
            k = k.reshape(b, n_w, h, -1).permute(0, 2, 1, 3)
            v = v.reshape(b, n_w, h, -1).permute(0, 2, 1, 3)

            dots = einsum('b h i d, b h j d -> b h i j', q, k) * self.scale
            attn = self.attend(dots)

            out = einsum('b h i j, b h j d -> b h i d', attn, v)

            out = rearrange(out, 'b h n d -> b n (h d)')

            out = self.to_out(out)

            att_map = torch.zeros(b, n, n).to(out.device)

            return (out, attn_workspace), att_map, 

class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim, dropout = 0.):
        super().__init__()
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                Attention(dim, heads = heads, dim_head = dim_head, dropout = dropout),
                #CrossAttention(dim, heads = heads, dim_head = dim_head, dropout = dropout),
                FeedForward(dim, mlp_dim, dropout = dropout)
            ]))
        self.norm1 = nn.LayerNorm(dim)
        self.norm2 = nn.LayerNorm(dim)

    def forward(self, x, attn_workspace = None):
        for attn, ff in self.layers:
            x_att, att = attn(self.norm1(x))
            #x_att = self.norm1(x_att)
            x = x_att + x

            x = ff(self.norm2(x)) + x
        return x, att

class CrossTransformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim, dropout = 0.):
        super().__init__()
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                CrossAttention(dim, heads = heads, dim_head = dim_head, dropout = dropout),
                PreNorm(dim, FeedForward(dim, mlp_dim, dropout = dropout))
            ]))
        self.gru = nn.GRUCell(dim, dim)
        self.layer_norm = nn.LayerNorm(dim)

    def forward(self, x, k):
        b, t, d = x.shape
        for attn, ff in self.layers:
            x += attn(x, k)            
            



            x = self.layer_norm(x)
            #x = ff(x) + x
        return x




class LReLU(nn.Module):
    def __init__(self, c=1./3):
        super(LReLU, self).__init__()
        self.c = c

    def forward(self, x):
        return torch.clamp(F.leaky_relu(x, self.c), -3., 3.)

def log_prob_gaussian(x, mu, log_vars, mean=False):
    lp = - 0.5 * math.log(2 * math.pi) \
        - log_vars / 2 - (x - mu) ** 2 / (2 * torch.exp(log_vars))

    if mean:
        #a = torch.mean(lp, -1)
        #print("hereee")
        return torch.mean(lp, -1)
    return torch.sum(lp, -1)

class TwoStreamTransformerV3(nn.Module):
    def __init__(self, *, image_size,
        inner_patch_size,
        outer_patch_size,
        num_classes,
        dim,
        depth,
        heads,
        mlp_dim,
        pool = 'cls',
        channels = 3,
        dim_head = 64,
        dropout = 0., 
        emb_dropout = 0.,
        num_workspace_slots = 5,
        p = 0.1,
        inner_depth = 6,
        num_templates=4,
        num_communication_steps = 1):
        super().__init__()
        image_height, image_width = pair(image_size)
        patch_height, patch_width = pair(outer_patch_size)

        print("INNER DEPTH:" + str(inner_depth))
        print("OUTER DEPTH:" + str(depth))
        
        self.p_height = patch_height
        self.p_width = patch_width
        self.depth = depth
        self.num_templates = num_templates
        self.dim = dim
        self.num_communication_steps = num_communication_steps

        self.p = p
        print("p:" + str(self.p))
        print("num_workspace_slots:" + str(num_workspace_slots))


        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        self.patch_dim = channels * patch_height * patch_width
        assert pool in {'cls', 'mean'}, 'pool type must be either cls (cls token) or mean (mean pooling)'

        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (p1 p2 c) h w', p1 = outer_patch_size, p2 = outer_patch_size),
            nn.Conv2d(self.patch_dim, dim, 1),
        )
        
        self.posterior_init = nn.Parameter(0.01 * torch.randn(1, 1, self.dim))
        self.prior_init = nn.Parameter(0.01 * torch.randn(1, 1, self.dim))

        self.posterior_mha = nn.MultiheadAttention(self.dim, heads, batch_first = True)
        self.post_mod = nn.Sequential(
            nn.Linear(self.dim, self.dim),
            LReLU(),
            nn.Linear(self.dim, self.dim * 2))
        self.prior_mha = nn.MultiheadAttention(self.dim, heads, batch_first = True)
        self.prior_mod = nn.Sequential(
            nn.Linear(self.dim, self.dim),
            LReLU(),
            nn.Linear(self.dim, self.dim * 2))

        self.aux_mod = nn.Sequential(
            nn.Linear(2 * self.dim, self.dim),
            LReLU(),
            nn.Linear(self.dim, 2 * self.dim))

        self.inner_patch_size = inner_patch_size
        self.outer_patch_size = outer_patch_size

        #self.pos_embedding = PositionalEncoding(dim, max_len = 1024 )#nn.Parameter(torch.randn(1, 1024, dim))
        self.cls_token = nn.Parameter(torch.randn(1, 1, dim))
        self.dropout = nn.Dropout(emb_dropout)

        self.depth = depth
        self.inner_depth = inner_depth

        self.transformer = nn.ModuleList([])
        for d in range(depth):
            self.transformer.append(nn.ModuleList([Transformer(dim, 1, heads, dim_head, mlp_dim, dropout)
                                                     for i in range(inner_depth)]))

        #self.transformer = Transformer(dim, self.depth * self.inner_depth, heads, dim_head, mlp_dim, dropout)

        self.num_workspace_slots = num_workspace_slots

        self.workspace_slots = nn.Parameter(torch.randn(1, 1, num_workspace_slots, dim))
        self.attention_workspace_slots = nn.Parameter(torch.randn(1, num_workspace_slots, dim))

        self.write_attention_ln1 = nn.LayerNorm(dim)
        self.write_attention = nn.MultiheadAttention(
                dim,
                heads,
                dropout=dropout,
                bias=True,
                add_bias_kv=True,

            ) #for _ in range(self.depth)])

        self.write_attention_ln2 = nn.LayerNorm(dim)

        self.write_attention_ffn = nn.Sequential(nn.Linear(dim, 2 * dim), nn.ELU(), nn.Linear(2 * dim, dim))

        self.preprocess_iterative = nn.Dropout(0.2)#nn.Sequential(nn.LayerNorm(dim), nn.Linear(dim , dim))

        self.classification_head = nn.Parameter(torch.randn(1, 1, dim))

        self.top_down_attention = nn.ModuleList([nn.MultiheadAttention(
                dim,
                heads,
                dropout=dropout,
                bias=True,
                add_bias_kv=True,
            ) for _ in range(depth)])

        self.gru_cell = nn.ModuleList([nn.GRUCell(dim, dim) for _ in range(self.num_templates)])#nn.GRU(dim, dim, batch_first = True)
        self.schema_select = nn.MultiheadAttention(dim, heads)

        self.comm_attention = nn.MultiheadAttention(dim, heads, dropout = dropout, bias = True, add_bias_kv = True)

        self.schemata_attention = nn.MultiheadAttention(dim, heads, dropout = dropout, bias = True, add_bias_kv = True)

        self.layer_norm = nn.LayerNorm(dim)

        self.rec_loss = nn.CrossEntropyLoss()
        self.kl_loss = nn.KLDivLoss()

        self.stop_mlp = nn.Sequential(nn.Linear(dim, dim), nn.ReLU(), nn.Linear(dim, 1))

        self.pool = pool
        self.to_latent = nn.Identity()

        self.mlp_head = nn.Sequential(
            nn.Linear(2 * dim, num_classes)
        )

        self.kl_loss = nn.KLDivLoss(reduction = 'none')

        self.Identity = nn.Identity()

        self.top_down_feedforward =  FeedForward(dim, mlp_dim, dropout = dropout)
        self.top_down_layernorm = nn.LayerNorm(dim)

    def read(self, rnn_slots, x, idx):
        x_ = x.permute(1, 0, 2)
        rnn_slots = rnn_slots.permute(1, 0, 2)

        x_ = self.top_down_layernorm(x_)

        x_write, _ = self.top_down_attention[idx](x_, rnn_slots, rnn_slots)
        #x_write = self.top_down_layernorm(x_write)
        x_write = x_write.permute(1, 0, 2)

        x = x + x_write

        #x = self.top_down_feedforward(self.top_down_layernorm(x)) + x # new change

        return x
    def write(self, x, slots):
        slots_ = slots.permute(1, 0, 2)
        x = x.permute(1, 0, 2)

        b, n, d = slots.size()


        write_info, _ = self.write_attention(self.write_attention_ln1(slots_), self.write_attention_ln1(x), self.write_attention_ln1(x))
        #write_info = self.write_attention_ln1(write_info)


        write_info = write_info.permute(1, 0, 2)

        slots = slots + write_info

        slots = self.write_attention_ffn(self.write_attention_ln2(slots)) + slots


        """
        
        
        slots = slots.reshape(b * n, d)
        write_info = write_info.reshape(b * n, d)

        slots = slots * 1.0 # Need for passing gradient, some weird pytorch error

        slots = self.gru[idx](write_info, slots)
        slots = slots.reshape(b, n, d)
        """

        return slots

    def gaussian_kld(self, mu_left, logvar_left, mu_right, logvar_right):
        """
        Compute KL divergence between a bunch of univariate Gaussian distributions
        with the given means and log-variances.
        We do KL(N(mu_left, logvar_left) || N(mu_right, logvar_right)).
        """
        gauss_klds = 0.5 * (logvar_right - logvar_left +
                            (torch.exp(logvar_left) / torch.exp(logvar_right)) +
                            ((mu_left - mu_right)**2.0 / torch.exp(logvar_right)) - 1.0)
        assert len(gauss_klds.size()) == 2
        return torch.sum(gauss_klds, 1)

    def reparametrize(self, mu, logvar, eps=None):
        std = logvar.mul(0.5).exp_()
        if eps is None:
            eps = torch.autograd.Variable(std.data.new(std.size()).normal_())
        return eps.mul(std).add_(mu)

    def scoff(self, x, state):
        b, n, d = x.size()

        x = x.reshape(b * n, d)
        state = state.reshape(b * n, d)

        state_out = []

        for gru in self.gru_cell:
            state_out.append(gru(x, state))

        state_out = torch.stack(state_out, dim = 1)

        state_out_ = state_out.permute(1, 0, 2)


        new_state, _ = self.schema_select(state.unsqueeze(0), state_out_, state_out_)

        #att_scores = torch.softmax(att_scores.squeeze(1), dim = 1)



        #new_state = (state_out * att_scores.unsqueeze(-1)).mean(dim = 1)

        #new_state = new_state.reshape(b, n, d)

        new_state = new_state.permute(1, 0, 2).squeeze(1).reshape(b, n, d)

        return new_state


    

    def forward(self, img):
        
        x = self.to_patch_embedding(img)

        x = rearrange(x, 'b c (p1 h) (p2 w) -> b c (p1 p2) h w', p1 = self.inner_patch_size, p2 = self.inner_patch_size)
        x = rearrange(x, 'b c p h w -> b p (h w) c')

        x_r = torch.flip(x, [1])
        x = torch.cat((x, x_r), dim = 0)
        num_chunks = x.size(1)

        b = x.size(0)

        workspace_slots = self.workspace_slots.repeat(b, x.size(1), 1, 1) #torch.zeros(b, self.depth, self.num_workspace_slots, x.size(-1), requires_grad = True).to(x.device)
        x_ori = x

        
        rnn_state = torch.zeros(b, self.num_workspace_slots, workspace_slots.size(-1)).to(x.device)

        
        posterior = []
        prior = []

        h = []
        b = []
        B, P, N, D = x.size()

        for p in range(P):
            x_cur = x[:, p]
            for outer in range(self.depth):
                for inner in range(self.inner_depth):
                    x_cur, _ = self.transformer[outer][inner](x_cur)

                x_cur = self.read(rnn_state, x_cur, outer)


            
            rnn_state = self.write(x_cur, rnn_state)
            h.append(rnn_state[: B // 2])
            b.append(rnn_state[B // 2:])


        b.reverse()

        h = torch.cat(h, dim = 0)
        b = torch.cat(b, dim = 0)

        h_b = torch.cat((h, b), dim = 1)

        post = self.posterior_init.repeat(h.shape[0], 1, 1)
        pr = self.prior_init.repeat(b.shape[0], 1, 1)
        post, _ = self.posterior_mha(post, h_b, h_b)
        post = post.squeeze(1)
        post_params = self.post_mod(post)
        post_params = torch.clamp(post_params, -8., 8.)
        post_mu, post_logvar = torch.chunk(post_params, 2, 1)
            
        pr, _ = self.prior_mha(pr, h, h)
        pr = pr.squeeze(1)
        pr_params = self.prior_mod(pr)
        pr_params = torch.clamp(pr_params, -8., 8.)
        pr_mu, pr_logvar = torch.chunk(pr_params, 2, 1)
        kld = self.gaussian_kld(post_mu, post_logvar, pr_mu, pr_logvar)

        z_step = self.reparametrize(post_mu, post_logvar)

        #if self.z_force:
        h_step_ = post * 0.
        #print(h_step_.shape)
        #print(z_step.shape)
        aux_params = self.aux_mod(torch.cat((h_step_, z_step), 1))
        aux_params = torch.clamp(aux_params, -8., 8.)
        aux_mu, aux_logvar = torch.chunk(aux_params, 2, 1)

        b = b.detach()
        #if self.use_l2:
        #    aux_step = torch.sum((b_step_ - F.tanh(aux_mu)) ** 2.0, 1)
        #else:
        #print(pr.shape)
        #print(aux_mu.shape)
        #print(aux_logvar.shape)
        aux_step = -log_prob_gaussian(
                    pr, torch.nn.functional.tanh(aux_mu), aux_logvar, mean=True)


        rnn_state = torch.split(rnn_state, x.shape[0] // 2, dim = 0)
        rnn_state = torch.cat(rnn_state, dim = -1)

        classication_scores = self.mlp_head(rnn_state.mean(dim = 1))

        #print(aux_step.shape)

        
        return classication_scores, kld.mean(dim = 0), aux_step.mean(dim = 0)
        


"""
0 1 2 3 4 
4 3 2 1 0
"""