import os
import sys
import openai
import requests
import csv
import json
import logging
from tqdm import tqdm
from tenacity import (
    retry,
    retry_if_not_exception_type,
    stop_after_attempt,
    wait_random_exponential,
    before_sleep_log,
)


JSONL_FILE = "WizardLM_testset.jsonl"
JSONL_URL = (
    "https://raw.githubusercontent.com/nlpxucan/WizardLM/"
    "ce0d433589335d419aa8101abd71685ae7d187f3/WizardLM/data/WizardLM_testset.jsonl"
)
DATA_FILE = "data.csv"

openai.api_type = "azure"
openai.api_base = os.getenv("OPENAI_API_BASE")
openai.api_version = "2023-03-15-preview"
openai.api_key = os.getenv("OPENAI_API_KEY")
engine = os.getenv("ENGINE")

logging.basicConfig(stream=sys.stderr, level=logging.DEBUG)
logger = logging.getLogger(__name__)


def download(url, file):
    response = requests.get(url)
    with open(file, "wb") as f:
        f.write(response.content)


def save(file, request, response):
    with open(file, "a") as f:
        writer = csv.writer(f)
        writer.writerow([request, response])


@retry(
    retry=retry_if_not_exception_type(
        (openai.error.InvalidRequestError, openai.error.AuthenticationError)
    ),
    wait=wait_random_exponential(min=8, max=500),
    stop=stop_after_attempt(30),
    before_sleep=before_sleep_log(logger, logging.DEBUG),
)
def get_response(request):
    response = openai.ChatCompletion.create(
        engine=engine,
        messages=[
            {
                "role": "system",
                "content": (
                    "You are an AI assistant that helps people find information."
                ),
            },
            {"role": "user", "content": request},
        ],
        temperature=0.7,
        max_tokens=7000,
        top_p=0.95,
        frequency_penalty=0,
        presence_penalty=0,
        stop=None,
        request_timeout=60,
    )
    response = response["choices"][0]["message"]["content"]
    return response


if __name__ == "__main__":
    if not os.path.exists(JSONL_FILE):
        download(JSONL_URL, JSONL_FILE)

    save(DATA_FILE, "request", "response")
    with open(JSONL_FILE) as f:
        for line in tqdm(f):
            request = line.strip()
            request = json.loads(request)["Instruction"]
            response = get_response(request)
            save(DATA_FILE, request, response)
