import sys

import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np
from functools import partial
import math
import logging
from collections import OrderedDict
from copy import deepcopy

# from timm.models.vision_transformer import VisionTransformer, _cfg
from timm.models.vision_transformer import default_cfgs, _cfg
from timm.models.registry import register_model
from timm.models.layers import trunc_normal_
from timm.layers.helpers import to_2tuple
from einops import rearrange, repeat

# from timm.models.layers import PatchEmbed, DropPath, lecun_normal_
from timm.models.layers import DropPath
from weight_init import lecun_normal_
from math import sqrt
from math import log

# from timm.models.helpers import build_model_with_cfg, named_apply, adapt_input_conv
from helpers import named_apply


'''
s6-deit version
'''


__all__ = [
    'deit_s6_tiny_patch16_224_v2', 'deit_s6_small_patch16_224_v2', 'deit_s6_base_patch16_224_v2',
]


# patchembed, MLP, Attention

class PatchEmbed(nn.Module):
    """ 2D Image to Patch Embedding
    """
    def __init__(self, img_size=224, patch_size=16, in_chans=3, embed_dim=768, norm_layer=None):
        super().__init__()
        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]

        self.proj = nn.Conv2d(in_chans, embed_dim, kernel_size=patch_size, stride=patch_size)
        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x).flatten(2).transpose(1, 2)
        x = self.norm(x)
        return x
        

class Mlp(nn.Module):
    """ MLP as used in Vision Transformer, MLP-Mixer and related networks
    """
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        drop_probs = to_2tuple(drop)

        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.drop1 = nn.Dropout(drop_probs[0])
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop2 = nn.Dropout(drop_probs[1])

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop1(x)
        x = self.fc2(x)
        x = self.drop2(x)
        return x


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv.unbind(0)   # make torchscript happy (cannot use tensor as tuple)

        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x
    
class s6_layer(nn.Module):
    """
    when heads = channels, channelwise (Q(K)' is then pointwise(channelwise) multiplication)
    
    Args:
        input_dim: input channel c (output channel is the same)
        k_size: kernel size of conv1d
        input : [b, c, h, w]
        output: [b, c, h, w]
        
        Wq, Wk: conv1d
        Wv: conv2d
        Q: [b, 1, c]
        K: [b, 1, c]
        V: [b, c, h, w]
    """
    def __init__(self, input_dim, ssm_channel, heads=None, dim_perhead=None, k_size=None):
        super(s6_layer, self).__init__()
        self.input_dim = input_dim
        self.k_size = k_size
        if (heads == None) and (dim_perhead == None):
            raise ValueError("arguments heads and dim_perhead cannot be None at the same time !")
        elif dim_perhead != None:
            heads = int(input_dim / dim_perhead)
        else:
            heads = heads
        self.heads = heads
        
        if k_size == None:
            t = int(abs((log(input_dim, 2) + 1) / 2.))
            k_size = t if t % 2 else t+1
        self.ssm_channel = ssm_channel
        self.dt_rank = 4
        norm_layer=partial(nn.LayerNorm, eps=1e-6)
        # s6 settings    
        # S4D real initialization
        self.A_log = nn.Parameter(torch.empty(self.input_dim, self.ssm_channel, device=torch.device('cuda' if torch.cuda.is_available() else 'cpu')))
        
        self.x_proj = nn.Linear(
            self.input_dim, self.dt_rank + self.ssm_channel * 2, bias=False
        )
        self.dt_proj = nn.Linear(self.dt_rank, self.input_dim, bias=True)
            
        self.act = nn.SiLU()
        self.norm = nn.LayerNorm(ssm_channel)
        self.tanh = nn.Tanh()
        self.normx = norm_layer(input_dim)
        self.normo = norm_layer(input_dim)
        self.Wx = nn.Linear(input_dim, ssm_channel)
        self.lambda_t = nn.Parameter(torch.zeros(1, 1, input_dim))
        self.lambda_t = torch.nn.init.kaiming_normal_(self.lambda_t)

    def forward(self, x, h):
        
        x = self.normx(x) # b len c
        h = self.normo(h) # b n c
        
        b, n, k = x.size()
        cls_token, tokens = torch.split(x, [1, n - 1], dim=1)

        # x = torch.cat((x, h), dim=1) # [b, len+1+n, c]
        
        # out_attn = self.attn(x)
        # x = x + self.drop_path(self.attn(self.norm1(x)))
        # out_attn = x + self.drop_path(self.mlp(self.norm2(x)))
        # out, _ = torch.split(out_attn, [x.size(1)-self.ssm_channel, self.ssm_channel], dim=1) # b len+1 c
        # # out_attn = x
        # cls_token, _, h = torch.split(self.norm3(out_attn), [1, x.size(1)-1-self.ssm_channel, self.ssm_channel], dim=1) # [b, len, c], [b, 1, c], [b, n, c]
        
        # x_out = torch.cat((cls_token, x_temp), dim=1) # [b, len+1, c]
        h = h.transpose(1,2) # [b, c, n]
        # h = self.norm(h.to(dtype=self.norm.weight.dtype)) # [b, c, n]   
        
        # s6 module updates
        # s6 parameters         
        cls_token = cls_token.transpose(1, 2) # [b, c, 1]
        # cls_token = self.act(cls_token) # [b, c, 1]
        x_dbl = self.x_proj(rearrange(cls_token, "b c l -> (b l) c"))  # [b*1, c]
        # x_dbl1 = torch.randn_like(x_dbl)
        dt, B, C = torch.split(x_dbl, [self.dt_rank, self.ssm_channel, self.ssm_channel], dim=-1)
        dt = self.dt_proj.weight @ dt.t()
        dt = rearrange(dt, "c (b l) -> b c l", l=1) # b c 1
        dt = F.softplus(dt + self.dt_proj.bias[..., None].float())
        B = rearrange(B, "(b l) n -> b n l", l=1) # b n 1
        C = rearrange(C, "(b l) n -> b n l", l=1) # b n 1
        A = -torch.exp(self.A_log.float()) # c n
        deltaA = torch.exp(torch.einsum('bcl,cn->bcln', dt, A)) # b c 1 n
        deltaB_y = torch.einsum('bcl,bnl,bcl->bcln', dt, B, cls_token) # b c 1 n  
        h = deltaA[:, :, 0] * h + deltaB_y[:, :, 0] # b c n
        # C = torch.zeros(cls_token.size(0), self.ssm_channel, 1, device=torch.device('cuda' if torch.cuda.is_available() else 'cpu')) # b c 1
        # out = torch.einsum('bdn,bn->bd', h, C[:, :, 0]) # b c
        h = self.normx(h.transpose(1,2)).transpose(1,2) # b c n
        h = self.tanh(h) # b c n
        # Wx(tokens) # b len n
        tokens = tokens + self.lambda_t.expand_as(tokens)*torch.einsum('bln,bcn->blc', self.Wx(tokens), h) # h influence x
        out = torch.cat((cls_token.transpose(1,2), tokens), dim=1)        

        # out = self.bn(out) # b c
        # out = self.tanh(out) # b c

        return out, h.transpose(1,2)
    
class s6_Block(nn.Module):

    def __init__(self, dim, num_heads, ssm_channel,
                 mlp_ratio=4., qkv_bias=False, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=partial(nn.LayerNorm, eps=1e-6)):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(dim, num_heads=num_heads, qkv_bias=qkv_bias, attn_drop=attn_drop, proj_drop=drop)
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        self.norm3 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)
        
        self.s6layer = s6_layer(input_dim=dim, dim_perhead=16, ssm_channel = ssm_channel)

    def forward(self, x, h):
        
        x = x + self.drop_path(self.attn(self.norm1(x)))
        x = x + self.drop_path(self.mlp(self.norm2(x))) # 1 197 192
        # print(x.shape) # 1 197 192
        # layer attention
        x1, h = self.s6layer(x, h)
        x = x + self.norm3(x1)
        return x, h
    
class ViT_s6(nn.Module):
    """ Vision Transformer
    A PyTorch impl of : `An Image is Worth 16x16 Words: Transformers for Image Recognition at Scale`
        - https://arxiv.org/abs/2010.11929
    Includes distillation token & head support for `DeiT: Data-efficient Image Transformers`
        - https://arxiv.org/abs/2012.12877
    """
######
    def __init__(self, img_size=224, patch_size=16, in_chans=3, num_classes=1000, 
                 embed_dim=768, depth=12,
                 num_heads=12, ssm_channel=64,
                 mlp_ratio=4., qkv_bias=True, 
                 representation_size=None, distilled=False,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0., 
                 embed_layer=PatchEmbed, norm_layer=None,
                 act_layer=nn.GELU, weight_init=''):
        """
        Args:
            img_size (int, tuple): input image size
            patch_size (int, tuple): patch size
            in_chans (int): number of input channels
            num_classes (int): number of classes for classification head
            embed_dim (int): embedding dimension
            ssm_channel (int): number of channels for SSM
            depth (int): depth of transformer
            num_heads (int): number of attention heads
            mlp_ratio (int): ratio of mlp hidden dim to embedding dim
            qkv_bias (bool): enable bias for qkv if True
            representation_size (Optional[int]): enable and set representation layer (pre-logits) to this value if set
            distilled (bool): model includes a distillation token and head as in DeiT models
            drop_rate (float): dropout rate
            attn_drop_rate (float): attention dropout rate
            drop_path_rate (float): stochastic depth rate
            embed_layer (nn.Module): patch embedding layer
            norm_layer: (nn.Module): normalization layer
            weight_init: (str): weight init scheme
        """
        super().__init__()
        self.num_classes = num_classes
        self.ssm_channel = ssm_channel
        self.num_features = self.embed_dim = embed_dim  # num_features for consistency with other models
        self.num_tokens = 2 if distilled else 1
        norm_layer = norm_layer or partial(nn.LayerNorm, eps=1e-6)
        act_layer = act_layer or nn.GELU

        self.patch_embed = embed_layer(
            img_size=img_size, patch_size=patch_size, in_chans=in_chans, embed_dim=embed_dim)
        num_patches = self.patch_embed.num_patches

        self.cls_token = nn.Parameter(torch.zeros(1, 1, embed_dim))
        self.dist_token = nn.Parameter(torch.zeros(1, 1, embed_dim)) if distilled else None
        self.pos_embed = nn.Parameter(torch.zeros(1, num_patches + self.num_tokens, embed_dim))
        self.pos_drop = nn.Dropout(p=drop_rate)

        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, depth)]  # stochastic depth decay rule
        self.blocks = nn.Sequential(*[
            s6_Block(dim=embed_dim, num_heads=num_heads, ssm_channel=ssm_channel, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, drop=drop_rate,
                attn_drop=attn_drop_rate, drop_path=dpr[i], norm_layer=norm_layer, act_layer=act_layer)
            for i in range(depth)])
        self.norm = norm_layer(embed_dim)

        # Representation layer
        if representation_size and not distilled:
            self.num_features = representation_size
            self.pre_logits = nn.Sequential(OrderedDict([
                ('fc', nn.Linear(embed_dim, representation_size)),
                ('act', nn.Tanh())
            ]))
        else:
            self.pre_logits = nn.Identity()

        # Classifier head(s)
        # self.avgpool = nn.AdaptiveAvgPool2d((1))
        self.head = nn.Linear(self.num_features, num_classes) if num_classes > 0 else nn.Identity()
        self.head_dist = None
        if distilled:
            self.head_dist = nn.Linear(self.embed_dim, self.num_classes) if num_classes > 0 else nn.Identity()

        self.init_weights(weight_init)
        
        self.h_0 = torch.nn.Parameter(torch.zeros(self.ssm_channel, embed_dim))        
        self.h_0 = torch.nn.init.kaiming_normal_(self.h_0)

    def init_weights(self, mode=''):
        assert mode in ('jax', 'jax_nlhb', 'nlhb', '')
        head_bias = -math.log(self.num_classes) if 'nlhb' in mode else 0.
        trunc_normal_(self.pos_embed, std=.02)
        if self.dist_token is not None:
            trunc_normal_(self.dist_token, std=.02)
        if mode.startswith('jax'):
            # leave cls token as zeros to match jax impl
            named_apply(partial(_init_vit_weights, head_bias=head_bias, jax_impl=True), self)
        else:
            trunc_normal_(self.cls_token, std=.02)
            self.apply(_init_vit_weights)

    def _init_weights(self, m):
        # this fn left here for compat with downstream users
        _init_vit_weights(m)

    # @torch.jit.ignore()
    # def load_pretrained(self, checkpoint_path, prefix=''):
    #     _load_weights(self, checkpoint_path, prefix)

    @torch.jit.ignore
    def no_weight_decay(self):
        return {'pos_embed', 'cls_token', 'dist_token'}

    def get_classifier(self):
        if self.dist_token is None:
            return self.head
        else:
            return self.head, self.head_dist

    def reset_classifier(self, num_classes, global_pool=''):
        self.num_classes = num_classes
        self.head = nn.Linear(self.embed_dim, num_classes) if num_classes > 0 else nn.Identity()
        if self.num_tokens == 2:
            self.head_dist = nn.Linear(self.embed_dim, self.num_classes) if num_classes > 0 else nn.Identity()

    def forward_features(self, x):
        x = self.patch_embed(x) # 1 196 192
        cls_token = self.cls_token.expand(x.shape[0], -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        if self.dist_token is None:
            x = torch.cat((cls_token, x), dim=1)
        else:
            x = torch.cat((cls_token, self.dist_token.expand(x.shape[0], -1, -1), x), dim=1)
        x = self.pos_drop(x + self.pos_embed) # 1 197 192
        b, _, k = x.size()
        h = self.h_0.repeat(b, 1, 1) 
        for block in self.blocks:
            x, h = block(x, h)
        x = self.norm(x)
        
        # feas = self.avgpool(x[:, 1:, ])
        # feas = torch.flatten(feas, 1)
        # out = x[:, 0] + feas
        if self.dist_token is None:
            return self.pre_logits(x[:, 0])
        else:
            return x[:, 0], x[:, 1]

    def forward(self, x):
        x = self.forward_features(x)
        if self.head_dist is not None:
            x, x_dist = self.head(x[0]), self.head_dist(x[1])  # x must be a tuple
            if self.training and not torch.jit.is_scripting():
                # during inference, return the average of both classifier predictions
                return x, x_dist
            else:
                return (x + x_dist) / 2
        else:
            x = self.head(x)
        return x
    
def _init_vit_weights(module: nn.Module, name: str = '', head_bias: float = 0., jax_impl: bool = False):
    """ ViT weight initialization
    * When called without n, head_bias, jax_impl args it will behave exactly the same
      as my original init for compatibility with prev hparam / downstream use cases (ie DeiT).
    * When called w/ valid n (module name) and jax_impl=True, will (hopefully) match JAX impl
    """
    if isinstance(module, nn.Linear):
        if name.startswith('head'):
            nn.init.zeros_(module.weight)
            nn.init.constant_(module.bias, head_bias)
        elif name.startswith('pre_logits'):
            lecun_normal_(module.weight)
            nn.init.zeros_(module.bias)
        else:
            if jax_impl:
                nn.init.xavier_uniform_(module.weight)
                if module.bias is not None:
                    if 'mlp' in name:
                        nn.init.normal_(module.bias, std=1e-6)
                    else:
                        nn.init.zeros_(module.bias)
            else:
                trunc_normal_(module.weight, std=.02)
                if module.bias is not None:
                    nn.init.zeros_(module.bias)
    elif jax_impl and isinstance(module, nn.Conv2d):
        # NOTE conv was left to pytorch default in my original init
        lecun_normal_(module.weight)
        if module.bias is not None:
            nn.init.zeros_(module.bias)
    elif isinstance(module, (nn.LayerNorm, nn.GroupNorm, nn.BatchNorm2d)):
        nn.init.zeros_(module.bias)
        nn.init.ones_(module.weight)

@register_model
def deit_s6_tiny_patch16_224_v2(pretrained=False, **kwargs):
    model = ViT_s6(
        patch_size=16, embed_dim=192, depth=12, num_heads=3, 
        mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = _cfg()
    # checkpoint = torch.load('/home/qsliu/Documents/MRLA/deit/work_dirs/deit_s6_tiny_patch16_224_v2_v2imagenet_ssmnewver2/best_checkpoint.pth')
    # model.load_state_dict(checkpoint["model"])
    # if pretrained:
    #     checkpoint = torch.hub.load_state_dict_from_url(
    #         url="https://dl.fbaipublicfiles.com/deit/deit_tiny_patch16_224-a1311bcf.pth",
    #         map_location="cpu", check_hash=True
    #     )
    #     model.load_state_dict(checkpoint["model"])
    return model


@register_model
def deit_s6_small_patch16_224_v2(pretrained=False,pretrained_cfg=False,pretrained_cfg_overlay=False,**kwargs):
    model = ViT_s6(patch_size=16, embed_dim=384, depth=12, num_heads=6, 
        mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = _cfg()
    checkpoint = torch.load('/home/r12user3/Documents/MRLA/deit/work_dirs/deit_s6_small_patch16_224_v2_v2imagenet_ssmnewver2/best_checkpoint.pth')
    model.load_state_dict(checkpoint["model"])
    # if pretrained:
    #     checkpoint = torch.hub.load_state_dict_from_url(
    #         url="https://dl.fbaipublicfiles.com/deit/deit_small_patch16_224-cd65a155.pth",
    #         map_location="cpu", check_hash=True
    #     )
    #     model.load_state_dict(checkpoint["model"])
    return model


@register_model
def deit_s6_base_patch16_224_v2(pretrained=False,pretrained_cfg=False,pretrained_cfg_overlay=False,**kwargs):
    model = ViT_s6(
        patch_size=16, embed_dim=768, depth=12, num_heads=12,
        mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), **kwargs)
    model.default_cfg = _cfg()
    checkpoint = torch.load('/home/r12user3/Documents/MRLA/deit/work_dirs/deit_s6_base_patch16_224_v2_v2imagenet_ssmnewver2/best_checkpoint.pth')
    model.load_state_dict(checkpoint["model"])
    # if pretrained:
    #     checkpoint = torch.hub.load_state_dict_from_url(
    #         url="https://dl.fbaipublicfiles.com/deit/deit_base_patch16_224-b5f2ef4d.pth",
    #         map_location="cpu", check_hash=True
    #     )
    #     model.load_state_dict(checkpoint["model"])
    return model


    
if __name__ == '__main__':
    from params_flops import compute_params
    compute_params(deit_s6_tiny_patch16_224_v1())