import sys
import os
import re
import csv
import argparse
import random
from pathlib import Path
from tqdm import tqdm
import numpy as np
import torch
import pandas as pd
import torch.nn as nn
from configs.supported import process_outputs_functions

from models.variable_width_resnet import __all__ as vwmodels
import configs.supported as supported

try:
    import wandb
except Exception as e:
    pass

def update_average(prev_avg, prev_counts, curr_avg, curr_counts):
    denom = prev_counts + curr_counts
    if isinstance(curr_counts, torch.Tensor):
        denom += (denom==0).float()
    elif isinstance(curr_counts, int) or isinstance(curr_counts, float):
        if denom==0:
            return 0.
    else:
        raise ValueError('Type of curr_counts not recognized')
    prev_weight = prev_counts/denom
    curr_weight = curr_counts/denom
    return prev_weight*prev_avg + curr_weight*curr_avg

# Adapted from https://sumit-ghosh.com/articles/parsing-dictionary-key-value-pairs-kwargs-argparse-python/
class ParseKwargs(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, dict())
        values = values[0].split(' ')
        for value in values:
            key, value_str = value.split('=')
            if value_str.replace('-','').isnumeric():
                processed_val = int(value_str)
            elif value_str.replace('-','').replace('.','').isnumeric():
                processed_val = float(value_str)
            elif value_str in ['True', 'true']:
                processed_val = True
            elif value_str in ['False', 'false']:
                processed_val = False
            else:
                processed_val = value_str
            getattr(namespace, self.dest)[key] = processed_val

def parse_bool(v):
    if v.lower()=='true':
        return True
    elif v.lower()=='false':
        return False
    else:
        raise argparse.ArgumentTypeError('Boolean value expected.')

def save_model(algorithm, epoch, best_val_metric, path):
    state = {}
    state['algorithm'] = algorithm.state_dict()
    state['epoch'] = epoch
    state['best_val_metric'] = best_val_metric
    torch.save(state, path)

def match_keys(d, ref):
    """
    Matches the format of keys between d (a dict) and ref (a list of keys).

    Helper function for situations where two algorithms share the same model, and we'd like to warm-start one
    algorithm with the model of another. Some algorithms (e.g. FixMatch) save the featurizer, classifier within a sequential,
    and thus the featurizer keys may look like 'model.module.0._' 'model.0._' or 'model.module.model.0._',
    and the classifier keys may look like 'model.module.1._' 'model.1._' or 'model.module.model.1._'
    while simple algorithms (e.g. ERM) use no sequential 'model._'
    """
    # hard-coded exceptions
    # d = {re.sub('model.1.', 'model.classifier.', k): v for k,v in d.items()}
    # d = {k: v for k,v in d.items() if 'pre_classifier' not in k} # this causes errors
    d = {re.sub('model.module.', 'model.', k): v for k,v in d.items()}

    # probe the proper transformation from d.keys() -> reference
    # do this by splitting d's first key on '.' until we get a string that is a strict substring of something in ref
    success = False
    probe = list(d.keys())[0].split('.')
    for i in range(len(probe)):
        probe_str = '.'.join(probe[i:])
        matches = list(filter(lambda ref_k: len(ref_k) >= len(probe_str) and probe_str == ref_k[-len(probe_str):], ref))
        matches = list(filter(lambda ref_k: not 'layer' in ref_k, matches)) # handle resnet probe being too simple, e.g. 'weight'
        if len(matches) == 0: continue
        else:
            success = True
            append = [m[:-len(probe_str)] for m in matches]
            remove = '.'.join(probe[:i]) + '.'
            break
    if not success: raise Exception("These dictionaries have irreconcilable keys")

    return_d = {}
    for a in append:
        for k,v in d.items(): return_d[re.sub(remove, a, k)] = v

    # hard-coded exceptions
    if 'model.classifier.weight' in return_d:
       return_d['model.1.weight'], return_d['model.1.bias'] = return_d['model.classifier.weight'], return_d['model.classifier.bias']
    return return_d


def load(algorithm, path, has_ep_val=True):
    state = torch.load(path)
    if has_ep_val:
        try:
            algorithm.load_state_dict(state['algorithm'])
        except:
            d = {k.replace('model.module.', 'model.'): v for k, v in state['algorithm'].items()}
            algorithm.load_state_dict(d)
        return state['epoch'], state['best_val_metric']
    else:
        algorithm.load_state_dict(state['state_dict'])
        return state['epoch'], state['best_acc1']

def log_group_data(datasets, grouper, logger):
    for k, dataset in datasets.items():
        name = dataset['name']
        dataset = dataset['dataset']
        logger.write(f'{name} data...\n')
        if grouper is None:
            logger.write(f'    n = {len(dataset)}\n')
        else:
            _, group_counts = grouper.metadata_to_group(
                dataset.metadata_array,
                return_counts=True)
            group_counts = group_counts.tolist()
            for group_idx in range(grouper.n_groups):
                logger.write(f'    {grouper.group_str(group_idx)}: n = {group_counts[group_idx]:.0f}\n')
    logger.flush()

# for wilds benchmark algorithms only 
def evaluate(algorithm, datasets, epoch, general_logger, config, is_best):
    algorithm.eval()
    torch.set_grad_enabled(False)
    for split, dataset in datasets.items():
        if (not config.evaluate_all_splits) and (split not in config.eval_splits):
            continue
        epoch_y_true = []
        epoch_y_pred = []
        epoch_metadata = []
        iterator = tqdm(dataset['loader']) if config.progress_bar else dataset['loader']
        for batch in iterator:
            batch_results = algorithm.evaluate(batch)
            epoch_y_true.append(detach_and_clone(batch_results['y_true']))
            y_pred = detach_and_clone(batch_results['y_pred'])
            if config.process_outputs_function is not None:
                y_pred = process_outputs_functions[config.process_outputs_function](y_pred)
            epoch_y_pred.append(y_pred)
            epoch_metadata.append(detach_and_clone(batch_results['metadata']))

        epoch_y_pred = collate_list(epoch_y_pred)
        epoch_y_true = collate_list(epoch_y_true)
        epoch_metadata = collate_list(epoch_metadata)
        results, results_str = dataset['dataset'].eval(
            epoch_y_pred,
            epoch_y_true,
            epoch_metadata)

        results['epoch'] = epoch
        dataset['eval_logger'].log(results)
        general_logger.write(f'Eval split {split} at epoch {epoch}:\n')
        general_logger.write(results_str)

        # Skip saving train preds, since the train loader generally shuffles the data
        if split != 'train':
            save_pred_if_needed(epoch_y_pred, dataset, epoch, config, is_best, force_save=True)

def save_pred_if_needed(y_pred, dataset, epoch, config, is_best, force_save=False):
    if config.save_pred:
        prefix = get_pred_prefix(dataset, config)
        if force_save or (config.save_step is not None and (epoch + 1) % config.save_step == 0):
            save_pred(y_pred, prefix + f'epoch:{epoch}_pred')
        if (not force_save) and config.save_last:
            save_pred(y_pred, prefix + f'epoch:last_pred')
        if config.save_best and is_best:
            save_pred(y_pred, prefix + f'epoch:best_pred')

class Logger(object):
    def __init__(self, fpath=None, mode='w'):
        self.console = sys.stdout
        self.file = None
        if fpath is not None:
            self.file = open(fpath, mode)

    def __del__(self):
        self.close()

    def __enter__(self):
        pass

    def __exit__(self, *args):
        self.close()

    def write(self, msg):
        self.console.write(msg)
        if self.file is not None:
            self.file.write(msg)

    def flush(self):
        self.console.flush()
        if self.file is not None:
            self.file.flush()
            os.fsync(self.file.fileno())

    def close(self):
        self.console.close()
        if self.file is not None:
            self.file.close()

class SysLog(object):
    def __init__(self, *files):
        self.files = files
    def write(self, obj):
        for f in self.files:
            f.write(obj)
            f.flush() # If you want the output to be visible immediately
    def flush(self) :
        for f in self.files:
            f.flush()


class BatchLogger:
    def __init__(self, csv_path, mode='w', use_wandb=False):
        self.path = csv_path
        self.mode = mode
        self.file = open(csv_path, mode)
        self.is_initialized = False

        # Use Weights and Biases for logging
        self.use_wandb = use_wandb
        if use_wandb:
            self.split = Path(csv_path).stem

    def setup(self, log_dict):
        columns = log_dict.keys()
        # Move epoch and batch to the front if in the log_dict
        for key in ['batch', 'epoch']:
            if key in columns:
                columns = [key] + [k for k in columns if k != key]

        self.writer = csv.DictWriter(self.file, fieldnames=columns)
        if self.mode=='w' or (not os.path.exists(self.path)) or os.path.getsize(self.path)==0:
            self.writer.writeheader()
        self.is_initialized = True

    def log(self, log_dict):
        if self.is_initialized is False:
            self.setup(log_dict)
        self.writer.writerow(log_dict)
        self.flush()

        if self.use_wandb:
            results = {}
            for key in log_dict:
                new_key = f'{self.split}/{key}'
                results[new_key] = log_dict[key]
            wandb.log(results)

    def flush(self):
        self.file.flush()

    def close(self):
        self.file.close()

def set_seed(seed):
    """Sets seed"""
    if torch.cuda.is_available():
        torch.cuda.manual_seed(seed)
    torch.manual_seed(seed)
    np.random.seed(seed)
    random.seed(seed)
    torch.backends.cudnn.benchmark = False
    torch.backends.cudnn.deterministic = True

def log_config(config, logger):
    for name, val in vars(config).items():
        logger.write(f'{name.replace("_"," ").capitalize()}: {val}\n')
    logger.write('\n')

def initialize_wandb(config, project=f"wilds"):
    name = config.dataset + '_' + config.algorithm if config.name is None else config.name
    wandb.init(name=name,
               project=config.wandb_name if config.wandb_name is not None else project)
    wandb.config.update(config)

def save_pred(y_pred, path_prefix):
    # Single tensor
    if torch.is_tensor(y_pred):
        df = pd.DataFrame(y_pred.numpy())
        df.to_csv(path_prefix + '.csv', index=False, header=False)
    # Dictionary
    elif isinstance(y_pred, dict) or isinstance(y_pred, list):
        torch.save(y_pred, path_prefix + '.pth')
    else:
        raise TypeError("Invalid type for save_pred")

def get_replicate_str(config):
    if config.dataset == 'poverty':
        replicate_str = f"fold:{config.dataset_kwargs['fold']}"
    else:
        replicate_str = f"seed:{config.seed}"
    return replicate_str

def get_pred_prefix(dataset, config):
    dataset_name = dataset['dataset'].dataset_name
    split = dataset['split']
    replicate_str = get_replicate_str(config)
    prefix = os.path.join(
        config.log_dir,
        f"{dataset_name}_split:{split}_{replicate_str}_")
    return prefix

def get_model_aug_str(config):
    s = ''
    if config.prune_method != '-':
        if config.prune_method == 'global':
            s += f'_{config.prune_method}_{config.prune_perc}{config.prune_criterion[0]}'
        elif config.prune_method == 'threshold':
            s += f'_{config.prune_method}{config.prune_layer_th}_{config.prune_perc}{config.prune_criterion[0]}'
    if config.reinit_method != '-':
        s += f'_reinit_{config.reinit_method}_{config.reinit_layer_th}'
    if hasattr(config, 'head_weight_decay') and config.head_weight_decay is not None:
        s += f'_hwd_{int(config.head_layers)}_{config.head_weight_decay}'
    return s

def get_expt_name(config, with_rep=False, is_train=True):
    model_name = config.model + ('_w:' + str(config.width) if config.model in vwmodels else '')
    replicate_str = '_' + get_replicate_str(config) if with_rep else ''
    tr_spec_str = ''
    if config.algorithm in supported.ssl_algorithms:
        if config.ssl_uniform_over_groups:
            tr_spec_str = '_unig'
        if config.ssl_train_set != 'train':
            tr_spec_str = '_' + config.ssl_train_set[:2]
        if config.weights_model != '':
            model_name = '' if model_name == 'resnet50' else model_name + '_'
            model_name += config.weights_model
        if config.adapt:
            model_name += '_A'
    ds_name = config.dataset 
    algm_name = config.algorithm if is_train else config.ssl_algorithm  # to support run_eval
    return f"{ds_name}_{algm_name}_{model_name}{tr_spec_str}{replicate_str}"


def get_model_prefix(dataset, config):
    dataset_name = dataset['dataset'].dataset_name
    split = dataset['split']
    replicate_str = get_replicate_str(config)
    prefix = os.path.join(
        config.log_dir,
        f"split:{split}_{replicate_str}_")
    if config.algorithm not in supported.ssl_algorithms:
        prefix = os.path.join(
        config.log_dir,
        f"{config.dataset}_{replicate_str}_")
    return prefix


def move_to(obj, device):
    if isinstance(obj, dict):
        return {k: move_to(v, device) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [move_to(v, device) for v in obj]
    elif isinstance(obj, float) or isinstance(obj, int):
        return obj
    else:
        # Assume obj is a Tensor or other type
        # (like Batch, for MolPCBA) that supports .to(device)
        return obj.to(device)

def detach_and_clone(obj):
    if torch.is_tensor(obj):
        return obj.detach().clone()
    elif isinstance(obj, dict):
        return {k: detach_and_clone(v) for k, v in obj.items()}
    elif isinstance(obj, list):
        return [detach_and_clone(v) for v in obj]
    elif isinstance(obj, float) or isinstance(obj, int):
        return obj
    else:
        raise TypeError("Invalid type for detach_and_clone")

def collate_list(vec):
    """
    If vec is a list of Tensors, it concatenates them all along the first dimension.

    If vec is a list of lists, it joins these lists together, but does not attempt to
    recursively collate. This allows each element of the list to be, e.g., its own dict.

    If vec is a list of dicts (with the same keys in each dict), it returns a single dict
    with the same keys. For each key, it recursively collates all entries in the list.
    """
    if not isinstance(vec, list):
        raise TypeError("collate_list must take in a list")
    elem = vec[0]
    if torch.is_tensor(elem):
        return torch.cat(vec, 0)       # modified from return torch.cat(vec)
    elif isinstance(elem, list):
        return [obj for sublist in vec for obj in sublist]
    elif isinstance(elem, dict):
        return {k: collate_list([d[k] for d in vec]) for k in elem}
    else:
        raise TypeError("Elements of the list to collate must be tensors or dicts.")

def remove_key(key):
    """
    Returns a function that strips out a key from a dict.
    """
    def remove(d):
        if not isinstance(d, dict):
            raise TypeError("remove_key must take in a dict")
        return {k: v for (k,v) in d.items() if k != key}
    return remove

class Namespace(object):
    def __init__(self, somedict):
        for key, value in somedict.items():
            assert isinstance(key, str) and re.match("[A-Za-z_-]", key)
            if isinstance(value, dict):
                self.__dict__[key] = Namespace(value)
            else:
                self.__dict__[key] = value
    
    def __getattr__(self, attribute):

        raise AttributeError(f"Can not find {attribute} in namespace. Please write {attribute} in your config file(xxx.yaml)!")

class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self, name, fmt=':f'):
        self.name = name
        self.fmt = fmt
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def __str__(self):
        fmtstr = '{name} {val' + self.fmt + '} ({avg' + self.fmt + '})'
        return fmtstr.format(**self.__dict__)


class ProgressMeter(object):
    def __init__(self, num_batches, meters, prefix=""):
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch):
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        print('\t'.join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = '{:' + str(num_digits) + 'd}'
        return '[' + fmt + '/' + fmt.format(num_batches) + ']'


class WrappedModel(nn.Module):
	def __init__(self, module):
		super(WrappedModel, self).__init__()
		self.module = module # that I actually define.
	def forward(self, x):
		return self.module(x)

def get_model_module(model):
    if torch.cuda.device_count() > 1:
        return model.module
    return model

def get_features(loader, model, path=None, recompute=False, return_md=True, return_numpy=False): # model is a featurizer
    if path is not None and os.path.exists(path) and not recompute:
        return torch.load(path)
    
    print('getting features...')
    start_test = True
    with torch.no_grad():
        iter_test = iter(loader)
        # for i in tqdm.trange(len(loader)):
        for i in range(len(loader)):
            data = next(iter_test)
            inputs = data[0]
            labels = data[1]
            metadata = data[2]
            inputs = inputs.cuda()
            feats = model(inputs)
            if start_test:
                all_metadata = metadata.float().cpu()
                all_feature = feats.float().cpu()
                all_label = labels.float()
                start_test = False
            else:
                all_metadata = torch.cat((all_metadata, metadata.float().cpu()), 0)
                all_feature = torch.cat((all_feature,feats.float().cpu()),0)
                all_label = torch.cat((all_label, labels.float()), 0)
    if return_numpy:
        all_feature = all_feature.cpu().numpy()
        all_label = all_label.cpu().numpy()
        all_metadata = all_metadata.cpu().numpy()
    
    res = all_feature, all_label, all_metadata
    if not return_md:
        res = all_feature
    if path is not None:
        torch.save(res, path)
        print(f'saved features to {path}')
    return res

def eval_sklearn_clf(clf_key, x_train, y_train, m_train, x_test, y_test, m_test, test_set, return_pred=False):
    from sklearn.linear_model import LogisticRegression
    from sklearn.neighbors import KNeighborsClassifier
    from sklearn.model_selection import GridSearchCV

    lr_grid = {'C': [1., .7, .3, .1, 0.07, 0.03, 0.01], 
                'penalty': ['l1'], 'solver': ['liblinear']}
    knn_grid = {'n_neighbors': [3, 5, 10, 20]}
    seed = 42
    clfs = {
            'lin': GridSearchCV(LogisticRegression(random_state=seed), lr_grid, cv=5),
             'knn': GridSearchCV(KNeighborsClassifier(), knn_grid, cv=5),
            }

    clf = clfs[clf_key]
    clf.fit(x_train, y_train)
    
    y_pred = torch.tensor(clf.best_estimator_.predict(x_test))
    
    results, results_str = test_set.eval(y_pred, y_test, m_test)
    print(f'{clf_key} best parameters: ', clf.best_params_)
    print(f'sklearn {clf_key} result ----------------')
    print(results_str)
    
    if return_pred:
        results, y_pred
    return results, results['acc_avg']

def quick_eval(config, datasets, model):
    model.eval()
    from utils import get_features, eval_sklearn_clf
    from sklearn.preprocessing import StandardScaler
    train_dataloader = datasets['train']['loader']
    val_dataloader = datasets[config.eval_set]['loader']
    train_set = datasets['train']['dataset']
    val_set = datasets[config.eval_set]['dataset']
    x_train, y_train, train_md = get_features(train_dataloader, model)
    x_val, y_val, val_md = get_features(val_dataloader, model)
    print(x_train.shape, x_val.shape)

    sc = StandardScaler()
    x_train = sc.fit_transform(x_train)
    x_val = sc.transform(x_val)
    
    res = {}
    keys = ['lin', 'knn']

    for key in keys:
        res[key] = eval_sklearn_clf(key, x_train, y_train, train_md, x_val, y_val, val_md, val_set)
    
    return res


def standard_group_eval(metric, grouper, y_pred, y_true, metadata, aggregate=True):
    """
    Args:
        - metric (Metric): Metric to use for eval
        - grouper (CombinatorialGrouper): Grouper object that converts metadata into groups
        - y_pred (Tensor): Predicted targets
        - y_true (Tensor): True targets
        - metadata (Tensor): Metadata
    Output:
        - results (dict): Dictionary of results
        - results_str (str): Pretty print version of the results
    """
    results, results_str = {}, ''
    if aggregate:
        results.update(metric.compute(y_pred, y_true))
        results_str += f"Average {metric.name}: {results[metric.agg_metric_field]:.3f}\n"
    g = grouper.metadata_to_group(metadata)
    group_results = metric.compute_group_wise(y_pred, y_true, g, grouper.n_groups)
    for group_idx in range(grouper.n_groups):
        group_str = grouper.group_field_str(group_idx)
        group_metric = group_results[metric.group_metric_field(group_idx)]
        group_counts = group_results[metric.group_count_field(group_idx)]
        results[f'{metric.name}_{group_str}'] = group_metric
        results[f'count_{group_str}'] = group_counts
        if group_results[metric.group_count_field(group_idx)] == 0:
            continue
        results_str += (
            f'  {grouper.group_str(group_idx)}  '
            f"[n = {group_results[metric.group_count_field(group_idx)]:6.0f}]:\t"
            f"{metric.name} = {group_results[metric.group_metric_field(group_idx)]:5.3f}\n")
    results[f'{metric.worst_group_metric_field}'] = group_results[f'{metric.worst_group_metric_field}']
    results_str += f"Worst-group {metric.name}: {group_results[metric.worst_group_metric_field]:.3f}\n"
    return results, results_str
