from torch.optim.lr_scheduler import ReduceLROnPlateau, StepLR, CosineAnnealingLR

def initialize_scheduler(config, optimizer, n_train_steps):
    if config.lars:
        optimizer = optimizer.optim

    # construct schedulers
    if config.scheduler is None:
        return None
    elif config.scheduler=='ReduceLROnPlateau':
        assert config.scheduler_metric_name, f'scheduler metric must be specified for {config.scheduler}'
        scheduler = ReduceLROnPlateau(
            optimizer,
            **config.scheduler_kwargs)
        step_every_batch = False
        use_metric = True
    elif config.scheduler == 'StepLR':
        scheduler = StepLR(optimizer, **config.scheduler_kwargs)
        step_every_batch = False
        use_metric = False
    elif config.scheduler == 'cosine_schedule_with_warmup':
        from transformers import get_cosine_schedule_with_warmup
        scheduler = get_cosine_schedule_with_warmup(
            optimizer,
            **config.scheduler_kwargs)
        step_every_batch = True
        use_metric = False
    elif config.scheduler == 'CosineAnnealingLR':
        scheduler = CosineAnnealingLR(
            optimizer,
            **config.scheduler_kwargs)
        step_every_batch = False
        use_metric = True
    else:
        raise ValueError('Scheduler not recognized.')
    # add an step_every_batch field
    scheduler.step_every_batch = step_every_batch
    scheduler.use_metric = use_metric
    return scheduler

def step_scheduler(scheduler, metric=None):
    if isinstance(scheduler, ReduceLROnPlateau):
        assert metric is not None
        scheduler.step(metric)
    else:
        scheduler.step()
