### Implement two models: A-Radon SW regression, B-SW regression
library(pracma)
library(fdapace)
library(fdadensity)
library(parallel)
library(Matrix)
#library(caret)
library(reticulate)
library(frechet)

reticulate::use_python("~/virtualenvs/bin/python3", required = T)
py_config()
#py_version changed!!!
source("utils.R")
source("GetFVE2D.R")

##### Radon LQD ##########





create_response = function(n, N, eps, dSup, qSup, tSup, df_dens=NULL, df=NULL, respType = 'densTrans', fileSurfix=''){
  ## create response for each input respType
  # input: n: dimension of distribution samples
  #        N: dimension fo theta_grid
  #        eps: regularization for densTrans type
  #        dSup: density support for densTrans type 
  #        qSup: quantile support for densTrans type in the output
  #        tSup: theta support for sampleTrans type
  #        df_dens: a list of samples of multivariate density functions, each of dimension NXN
  #        df: a list of random observations, each of dimension ni X 2
  #        respType: densTrans
  # output: LQDout: output a list (for each angle) of lqd function with dimension nXN 
  #         Ymat: output a matrix of dimension n * (N*N)  - n X (qSup grid X theta grid) 
  #         eps, Const - used solely on respType-densTrans
  if (! (respType %in% c('densTrans', 'sampleTrans'))){
    stop('create response type not accepted')
  }
  Const = ones(n, N)  # n x length(theta_grid)
  LQDout = NULL
  py_file = paste("../data/station", fileSurfix, ".npy", sep='')
  py_file_transformed = py_file
  py_code = 'RadonTransform.py'
  py_version = '/usr/local/bin/python3.9'
  np = reticulate::import("numpy")
  
  # type1 -- Radon transform over density function, and create quantile function
  # output: list of dim n X length(rho_grid), each matrix corresponds to a theta value slice
  if (respType == 'densTrans'){
    ##  Radon transform
    #output data to Python process Radon transform 
    np$save(py_file, reticulate::r_to_py(df_dens))
    
    Sys.sleep(20)  # system need some time to store the data
    
    #Process Python file and output to station_radon.npy
    #rstudioapi::terminalExecute("/usr/local/bin/python3 /Users/hango/Desktop/UCDavis/research/SlicedWas/code/RadonTransform.py")
    py_cmd = paste(py_version, py_code, py_file, sep=' ')
    system(py_cmd)
    Sys.sleep(20)  # system need some time to store the data
    
    #load Radon transform data
    df_radon = np$load(py_file_transformed)
    df_radon = lapply(1:n, function(i){
      t(df_radon[i,,])
    })
    
    ## Create response Quantile matrix
    Const = zeros(n, N) # n x length(theta_grid)
    Ymat  = matrix(nrow = 0, ncol = N^2)
    for(i in 1:n){
      out = Transform2Ddens2lqd(eps, df_radon[[i]], dSup, qSup)
      out_lqd = out$lqd
      out_const = out$const
      Ymat = rbind(Ymat, as.vector(t(out_lqd)))
      Const[i,] = out_const
    }
    # out: list of dim n X N, each corresponds to a theta value
    # LQDout = lapply(1:N, function(k){
    #   Ymat[, seq((k-1)*N+1,k*N, by=1)]
    # })
    LQDout = lapply(1:n, function(i){
      mat = matrix(Ymat[i,], nrow=N)
      return(t(mat))
    })
  }else{
    #type 2: Radon transform over multivariate random sample, and create a slice sample 
    #output: list of a list of dimension n, out[[i]] is a list of size n, each corresponds to a sliced random sample 
    Yout = lapply(tSup, function(theta0){
      theta =  theta0 / 180 * pi #pi/2 +  theta0 / 180 * pi # pi + theta / 2*pi  theta
      vT = matrix(c(cos(theta), sin(theta)), nrow=2)
      Xsliced = lapply(df, function(X){
        return(as.matrix(X) %*% vT)  #slice the optimizer NX1
      })
      return(Xsliced)
    })
  }
  return (list(LQDout=LQDout, Ymat=Ymat, df_radon = df_radon, eps=eps, Const=Const))
}


density_rec = function(lqd_list, nout, eps, consInte, rho_grid, fileSurfix=''){
  #reconstruct density function from the sliced Frechet regression
  #input: df_frechet: list of frechet regression fitted density for each slice, each of dimension n X N
  #       n: dimension of random samples
  #output: a list of a sample of reconstructed density functions
  
  py_file_sliced = paste("../data/station_radon_rec", fileSurfix, ".npy", sep='')
  py_file_rec = py_file_sliced
  py_code = 'InverseRadonTransform.py'
  py_version = '/usr/local/bin/python3.9'
  np = reticulate::import("numpy")
  
  ## Part1: sliced density reconstruction
  df_radon_rec = lapply(1:nout, function(i){
    # densmat = t(sapply(df_frechet, function(mat){
    #   mat$dens[i,]
    # })) # Each row correpsonds to a theta value 
    densmat = lqd_list[[i]] # N(theta) X N(grid)
    densmat = InverseTransformLqd(eps, densmat, consInte, rho_grid)
    return(t(densmat))
  })

  ## Part2: output data to Python process Radon transform 
  np$save(py_file_sliced, reticulate::r_to_py(df_radon_rec))
  Sys.sleep(20)  # system need some time to store the data
  
  ## Part3: Inverse radon transform
  #call Python file to enable inverse radon transform and output to station_rec.npy
  #rstudioapi::terminalExecute("/usr/local/bin/python3 /Users/hango/Desktop/UCDavis/research/SlicedWas/code/InverseRadonTransform.py")
  py_cmd = paste(py_version, py_code, py_file_sliced, sep=' ')
  system(py_cmd)
  Sys.sleep(20)  # system need some time to store the data
  
  df_rec = np$load(py_file_rec)
  
  return(df_rec)
  
}






radon_lqd = function(df, userBoundX, userBoundY, eps, respType='densTrans', cores=4, h=10, K=5, outDens=TRUE, outModes=FALSE, fileSurfix='', verbose=TRUE){
    ## Main function for radon sliced frechet regression 
    ## Input: df: a  list of random distribution samples, each includes a sample of observations with niX2 matrix 
    #         userBoundX: bound for the first dimension 
    #         userBoundY: bound for the second dimension
    #         eps: regularization if respType is densTrans
    #         respType: densTrans: first fit a multivariate density followed with Radon transform 
    #                   sampleTrans: directly apply Radon transform to multivariate observations
    #         cores: number of cores available in the computing
    #         h: bandwidth adjustment on the top of default CV bandwidth in the multivariate density function fitting
    #         K: FPCA components used to reconstruct the multivariate density
    #         outDens: whether density output is needed
    #  Output: n: dimension of random distributions 
    #          N: dimension of working grid
    #          X_grid: output grid for X
    #          Y_grid: output grid for Y
    #          dens:   multivariate density fitting
    #          dens_rec: multivariate reconstructed density function
    #          dens_bary: multivariate radon barycenter, not available now
    #          FVE: a list of FVE result - FVE_L2: FVE w.r.t. L2 distance between dens and dens_rec
    #                                    - FVE_SW: FVE w.r.t. SW distance between df and fitted regressor 
    
    ## Part0: standardization
    boundCenter = c(mean(userBoundX), mean(userBoundY))
    boundScale = c((userBoundX[2]-userBoundX[1])/2, (userBoundY[2]-userBoundY[1])/2 )
    df = lapply(df, function(obs_mat){
      return(t(apply(obs_mat, 1, function(x) (x-boundCenter) / boundScale  )))
    })
    boundX = c(-1, 1)
    boundY = c(-1, 1)
    
    ## Part1: data grid 
    N = 51
    grid_df = get_data_grid(df, N=N, boundX = boundX, boundY=boundY)
    n = grid_df$n
    X_grid = grid_df$X_grid #working grid X
    Y_grid = grid_df$Y_grid #working grid Y
    X_usergrid = boundScale[1] * X_grid + boundCenter[1]
    Y_usergrid = boundScale[2] * Y_grid + boundCenter[2]
    theta_grid = grid_df$theta_grid #[0, 180]
    rho_grid = grid_df$rho_grid #[-1, 1]
    qSup = grid_df$qSup #[0,1]
    #print("Grid process finished!")

    ## Part2: density fitting
    df_dens = NULL
    if(outDens==TRUE | respType=='densTrans'){
      df_dens = mclapply(1:n, function(i){
        return(fmat(X.mat = kron(X_grid, ones(1,N)),
                    Y.mat = kron(ones(N,1), t(Y_grid)),
                    data = df[[i]],
                    lower = c(boundX[1], boundY[1]),
                    upper = c(boundX[2], boundY[2]),
                    h = h) )
      }, mc.cores = cores)
      if (verbose){
        print("Density modeling finished!")}
    }
    
    ## Part3: response creation
    response_obj = create_response(n, N, eps, dSup=rho_grid, qSup=qSup, tSup = theta_grid, df_dens=df_dens, df=df, respType = respType, fileSurfix=fileSurfix)
    LQDlist = response_obj$LQDout
    LQDmat = response_obj$Ymat
    df_radon_dens = response_obj$df_radon
    Const = response_obj$Const
    consInte = mean(Const)
    eps = response_obj$eps
    if (verbose){
      print("Response process finished!")}


    ## Part4: FPCA
    df_FPCA = TwoDFPCA(LQDmat, pc = K)
    lqd_rec_list = lapply(1:n, function(i){
      v = df_FPCA$Xest[i,]
      mat = matrix(v, nrow=length(theta_grid))
      return(t(mat))
    })
    scores = df_FPCA$scores
    # df_radon_rec = lapply(1:n, function(i){
    #   v = df_FPCA$Xest[i,]
    #   mat = matrix(v, nrow=length(theta_grid))
    #   return(InverseTransformLqd(eps, mat, Const[i,], rho_grid))
    # })
    # # transpose
    # df_radon_rec = lapply(df_radon_rec, function(v){
    #   return(t(v))
    # })
    if (verbose){
    print("2DFPCA finished")
    print(df_FPCA$psi$cumFVE)}
    
    
    ## Part5: density reconstruction
    df_rec = NULL
    if (outDens==TRUE){
      df_rec = density_rec(lqd_rec_list, nout=n, eps=eps, consInte=consInte, rho_grid=rho_grid, fileSurfix=fileSurfix)
      #return from the working grid to the usergrid
      df_rec = lapply(1:n, function(i){
        temp = df_rec[i,,]
        temp[temp<0] = 0
        temp = temp / (boundScale[1] * boundScale[2])
        temp = temp / trapz2DRcpp(X_usergrid, Y_usergrid, temp)
        return(temp)
      })
      #return from the working grid to the usergrid
      df_dens = lapply(df_dens, function(mat){
        mat[mat<0] = 0
        mat =  mat / (boundScale[1] * boundScale[2])
        mat = mat / trapz2DRcpp(X_usergrid, Y_usergrid, mat)
        return(mat)
      })
      if (verbose){
        print("Inverse radon finished!")}
    }
    
    
    df_mov = NULL
    if (outModes==TRUE){
      lqd_mov_list = df_FPCA$mov
      lqd_mov_list = unlist(lqd_mov_list, recursive = FALSE)
      lqd_mov_list = lapply(lqd_mov_list, function(v){
        mat = matrix(v, nrow=length(theta_grid))
        return(t(mat))
      })
      df_mov = density_rec(lqd_mov_list, nout=length(lqd_mov_list), eps=eps, consInte=consInte, rho_grid=rho_grid, fileSurfix=fileSurfix)
      #return from the working grid to the usergrid
      df_mov = lapply(1:length(lqd_mov_list), function(i){
        temp = df_mov[i,,]
        temp[temp<0] = 0
        temp = temp / (boundScale[1] * boundScale[2])
        temp = temp / trapz2DRcpp(X_usergrid, Y_usergrid, temp)
        return(temp)
      })
      df_mov = lapply(1:K, function(i){
        df_mov[((i-1)*5+1): (i*5)]
      })
    }
  
    
    ## Part6: FVE
    FVE = list()
    if (!is.null(df_rec)){
      resultL2 = GetFVE2D(df_rec, df_dens, X_grid, Y_grid, metric = 'L2')$FVE
      FVE$FVEL2 = resultL2
    }
    FVE$FVESW = GetFVESW(df_radon_dens, lqd_rec_list, eps, consInte, rho_grid)
    
    

    return(list(n = n, 
                N=N, 
                X_grid = X_usergrid, 
                Y_grid = Y_usergrid, 
                dens = df_dens, 
                dens_rec = df_rec,
                dens_mov = df_mov,
                dens_FPCA = df_FPCA,
                scores = scores,
                theta_grid = theta_grid,
                rho_grid = rho_grid,
                qSup = qSup,
                FVE = FVE))
}







