### utils functions
library(pracma)
library(ks)

#get data grid
get_data_grid = function(df, N, boundX, boundY){
    #data grid 
    n = length(df)
    X_grid = seq(boundX[1], boundX[2], length.out = N)
    Y_grid = seq(boundY[1], boundY[2], length.out = N)
    theta_grid = 90 + seq(0,180,length.out = N+1)[1:N]   #match with Python package POT 
    rho_grid = seq(-1, 1, length.out = N)
    qSup = seq(0, 1, length.out = N)
    return(list(n = n, X_grid = X_grid, Y_grid = Y_grid, theta_grid = theta_grid, rho_grid = rho_grid, qSup = qSup))
}

#obtain smoothed densities based on random samples
fmat <- function(X.mat, Y.mat, data, lower, upper, h = 3){
    #generate n random sample follows truncated normal distribution 
    smooth2D = ks::kde(data, xmin = lower, xmax = upper, H = ks::Hbcv(data)*h) #Hpi(data) * h
    smooth2Dsum = trapz2DRcpp(smooth2D$eval.points[[1]], smooth2D$eval.points[[2]], smooth2D$estimate)
    smooth2DEst = predict(smooth2D, x = cbind(as.vector(X.mat), as.vector(Y.mat))) / smooth2Dsum
    return(matrix(smooth2DEst, nrow = nrow(X.mat)))
}

#standardized density function + transformation to quantile
Transform2Ddens2quantile = function(eps, densmat, rho_grid, qSup){
    densmat = densmat + eps
    #scale dens mat
    const = apply(densmat, 1, function(v) fdapace::trapzRcpp(rho_grid, v))
    dens_nor =  densmat / kron(const, ones(1, length(rho_grid)))
    #dens to quantile
    quant_nor = t(apply(dens_nor, 1, function(dens){
      fdadensity::dens2quantile(dens, dSup = rho_grid, qSup = qSup)
    }) )
    return(list(quantile = quant_nor, const = const))
}

#standardized density function + transformation to lqd
Transform2Ddens2lqd = function(eps, densmat, rho_grid, lqdSup){
  densmat = densmat + eps
  #scale dens mat
  const = apply(densmat, 1, function(v) fdapace::trapzRcpp(rho_grid, v))
  dens_nor =  densmat / kron(const, ones(1, length(rho_grid)))
  #dens to lqd
  dens_lqd = t(apply(dens_nor, 1, function(v){
    fdadensity::dens2lqd(v, dSup = rho_grid, lqdSup = lqdSup)
  }) )
  return(list(lqd = dens_lqd, const = const))
}

# lqd_dens = t(apply(dens_lqd, 1, function(v){
#   fdadensity::lqd2dens(v, dSup = rho_grid, lqdSup = seq(0, 1, length.out=length(rho_grid)))
# }))

#inverse standardized density function
InverseTransform = function(eps, densmat, consInte, N){
    #return(densmat * kron(const, ones(1, N)) - eps)
    return(densmat * consInte - eps)
}

#inverse lqd function
InverseTransformLqd = function(eps, densmat, consInte, rho_grid){
    lqdmat = t(apply(densmat, 1, function(v){
      fdadensity::lqd2dens(v, dSup = rho_grid)
    }))
    lqdmat = lqdmat * consInte - eps
    return(lqdmat)
}

#Local Frechet 
K = function(x,h,ker){
  ker = kerFctn(ker)
  k = 1
  for(i in 1:ncol(x)){
    k=k*ker(x[,i]/h[i])
  }
  return(as.numeric(k))
}

getLFRweights=function(xin, x0, n, ker, bw){
  #x0 is a vector in R^p that corresponds to the covariate value for which we want to predict
  aux=K(xin-matrix(t(x0),nrow=n,ncol=length(x0),byrow=TRUE), bw, ker)
  mu0 = mean(aux)
  mu1 = colMeans(aux*(xin - matrix(t(x0),nrow=n,ncol=length(x0),byrow=TRUE)))
  mu2=0
  for(i in 1:n){
    mu2 = mu2 + aux[i]*(xin[i,]-x0) %*% t(xin[i,]-x0)/n
  }
  sL = array(0,n)
  for(i in 1:n){
    sL[i] =aux[i]*(1-t(mu1)%*%solve(mu2)%*%(xin[i,]-x0))
  }
  s = sum(sL)
  return(sL/s)
}

kerFctn <- function(kernel_type){
  ## copyright from Frechet Package
  if (kernel_type=='gauss'){
    ker <- function(x){
      dnorm(x) #exp(-x^2 / 2) / sqrt(2*pi)
    }
  } else if(kernel_type=='rect'){
    ker <- function(x){
      as.numeric((x<=1) & (x>=-1))
    }
  } else if(kernel_type=='epan'){
    ker <- function(x){
      n <- 1
      (2*n+1) / (4*n) * (1-x^(2*n)) * (abs(x)<=1)
    }
  } else if(kernel_type=='gausvar'){
    ker <- function(x) {
      dnorm(x)*(1.25-0.25*x^2)
    }
  } else if(kernel_type=='quar'){
    ker <- function(x) {
      (15/16)*(1-x^2)^2 * (abs(x)<=1)
    }
  } else {
    stop('Unavailable kernel')
  }
  return(ker)
}

#### calculate the integration over unit squares by parts
library(fdapace)
trapz2DRcpp <- function(X_list, Y_list, Z = Z){
  
  Z.intX = apply(Z, 2, function(z) trapzRcpp(X_list, z))
  Z.intXY = trapzRcpp(Y_list, Z.intX)
  return(Z.intXY)
}


