import torch
import torch.nn as nn
import numpy as np
import math




def conv3x3(in_planes, out_planes, stride=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.relu(out)

        out = self.conv2(out)
        out = out * self.alpha

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, planes)
        self.conv2 = conv3x3(planes, planes, stride)
        self.conv3 = conv1x1(planes, planes * self.expansion)
        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride


    def ortho_block_init(self):
        if self.downsample is not None:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            dim3 = self.conv3.weight.shape
            delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            d31 = math.ceil((dim3[2]-1)/2)
            d32 = math.ceil((dim3[3]-1)/2)

        
            dim4 = self.downsample.weight.shape
            df1 = math.ceil((dim4[2]-1)/2)
            df2 = math.ceil((dim4[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim4[0]/2),math.ceil(dim4[1]/2))) - delta3 @ delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.downsample.weight.data = torch.zeros(self.downsample.weight.shape)
            self.downsample.weight.data[:, :, df1, df2] = delta
           
            delta1 = torch.cat((delta1, -delta1), dim=0)
            delta1 = torch.cat((delta1, -delta1), dim=1)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12] = delta1
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
            delta3 = torch.cat((delta3, -delta3), dim=0)
            delta3 = torch.cat((delta3, -delta3), dim=1)
            self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
            self.conv3.weight.data[:, :, d31, d32] = delta3

        else:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            
            dim2 = self.conv2.weight.shape
            dim3 = self.conv3.weight.shape
            delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            d31 = math.ceil((dim3[2]-1)/2)
            d32 = math.ceil((dim3[3]-1)/2)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12].diagonal().copy_(torch.ones_like(torch.diag(self.conv1.weight.data[:, :, d11, d12])))
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22].diagonal().copy_(torch.ones_like(torch.diag(self.conv2.weight.data[:, :, d21, d22])))
            delta3 = torch.cat((delta3, -delta3), dim=0)
            delta3 = torch.cat((delta3, -delta3), dim=1)
            identity = torch.zeros(dim3)
            identity.diagonal().copy_(torch.ones_like(torch.diag(identity[:, :, d31, d32])))
            self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
            self.conv3.weight.data[:, :, d31, d32] = delta3
            self.conv3.weight.data -= identity

        for m in self.modules():
            m.to(curr_device)

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = out * self.alpha

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(self, block, layers, num_classes=1000):
        super(ResNet, self).__init__()
        self.num_layers = sum(layers)
        self.inplanes = 64
        self.conv1 = nn.Conv2d(3, 64, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bias1 = nn.Parameter(torch.zeros(1))
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.bias2 = nn.Parameter(torch.zeros(1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = conv1x1(self.inplanes, planes * block.expansion, stride)

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes))

        return nn.Sequential(*layers)


    def _ortho_block_init(self):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (Bottleneck, BasicBlock)):
                m.ortho_block_init()
                print('Initialized Ortho Block init')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]


    def forward(self, x):
        x = self.conv1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = x.view(x.size(0), -1)
        x = self.fc(x)

        return x


def ResNet50_b_nobn_imagenet(input_shape, num_classes):
    r"""ResNet-50 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_

    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return ResNet(Bottleneck, [3, 4, 6, 3], num_classes)