'''ResNet in PyTorch.

For Pre-activation ResNet, see 'preact_resnet.py'.

Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
import torch
import torch.nn as nn
import torch.nn.functional as F
import math
import torch.optim as optim

'''
The scaled weight standardization convolutional block is defined
'''

class ScaledStdConv2d(nn.Conv2d):
    """Conv2d layer with Scaled Weight Standardization.
    Paper: `Characterizing signal propagation to close the performance gap in unnormalized ResNets` -
        https://arxiv.org/abs/2101.08692
    Adapted from timm: https://github.com/rwightman/pytorch-image-models/blob/4ea593196414684d2074cbb81d762f3847738484/timm/models/layers/std_conv.py
    """

    def __init__(self, in_channels, out_channels, kernel_size, stride=1, padding=0, dilation=1, groups=1,
                 bias=True, gain=True, gamma=math.sqrt(2 / (1 - (1 / math.pi))), eps=1e-5):
        super().__init__(
            in_channels, out_channels, kernel_size, stride=stride,
            padding=padding, dilation=dilation, groups=groups, bias=bias)
        self.gain = nn.Parameter(torch.ones(
            self.out_channels, 1, 1, 1)) if gain else None
        # gamma * 1 / sqrt(fan-in)
        self.scale = gamma * self.weight[0].numel() ** -0.5
        self.eps = eps

        self.register_buffer('weight_mask', torch.ones(self.weight.shape))
        if self.bias is not None:
            self.register_buffer('bias_mask', torch.ones(self.bias.shape))

    def get_weight(self):
    
        mean = torch.mean(
            self.weight, dim=[1, 2, 3], keepdim=True)
        std = torch.std(
            self.weight, dim=[1, 2, 3], keepdim=True, unbiased=False)
        weight = self.scale * (self.weight - mean) / (std + self.eps)

        if self.gain is not None:
            weight = weight * self.gain
        return weight

    def _conv_forward(self, input, weight, bias):
        if self.padding_mode != 'zeros':
            return F.conv2d(F.pad(input, self._padding_repeated_twice, mode=self.padding_mode),
                            weight, bias, self.stride,
                            _pair(0), self.dilation, self.groups)
        return F.conv2d(input, weight, bias, self.stride,
                        self.padding, self.dilation, self.groups)

    def forward(self, input):
        W = self.weight_mask * self.get_weight()
        if self.bias is not None:
            b = self.bias_mask * self.bias
        else:
            b = self.bias
        return self._conv_forward(input, W, b)




class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = ScaledStdConv2d(in_planes, planes, kernel_size=1, bias=False)
        self.conv2 = ScaledStdConv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.conv3 = ScaledStdConv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)

        self.alpha = nn.Parameter(torch.tensor(0.2), requires_grad=False)

        self.shortcut = nn.Sequential(
                ScaledStdConv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False)
        )

    def ortho_block_init(self):
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        dim3 = self.conv3.weight.shape
        delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        d31 = math.ceil((dim3[2]-1)/2)
        d32 = math.ceil((dim3[3]-1)/2)

        if len(self.shortcut) > 0:
            dim4 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim4[2]-1)/2)
            df2 = math.ceil((dim4[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim4[0]/2),math.ceil(dim4[1]/2))) - delta3 @ delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        delta3 = torch.cat((delta3, -delta3), dim=0)
        delta3 = torch.cat((delta3, -delta3), dim=1)
        self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
        self.conv3.weight.data[:, :, d31, d32] = delta3

        for m in self.modules():
            m.to(curr_device)

    
    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = F.relu(self.conv2(out))
        out = self.conv3(out)
        out = self.alpha * out + self.shortcut(x)
        out = F.relu(out)
        return out

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = ScaledStdConv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.conv2 = ScaledStdConv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)

        self.alpha = nn.Parameter(torch.tensor(0.2), requires_grad=False)
       
        self.shortcut = nn.Sequential(
            ScaledStdConv2d(in_planes, self.expansion*planes,
                        kernel_size=1, stride=stride, bias=False),
            )

    def ortho_block_init(self):
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        if len(self.shortcut) > 0:
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        
        for m in self.modules():
            m.to(curr_device)


    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = self.conv2(out)
        out = self.alpha*out + self.shortcut(x)
        out = F.relu(out)
        return out




class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(ResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = ScaledStdConv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layer_list = []
        for stride in strides:
            layer_list.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layer_list)

    
    def _ortho_block_init(self):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (BasicBlock, Bottleneck)):
                m.ortho_block_init()
                print('Initialized Ortho Block init')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]

    def _ortho_alpha_init(self):
        i=0
        for name, m in self.named_modules():
            if isinstance(m, nn.Conv2d):
                i+=1
                if i == 1:
                    dim = m.weight.size()
                    din = math.ceil(m.weight.size()[1])
                    dout = math.ceil(m.weight.size()[0]/2)
                    w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                    w0 = torch.cat((w0,-w0),dim=0)
                    df1 = math.ceil((dim[2]-1)/2)
                    df2 = math.ceil((dim[3]-1)/2)
                    ww = torch.zeros(dim, device=m.weight.device)                
                    ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                    m.weight.data = ww

                else:
                    dim = m.weight.size()
                    din = math.ceil(m.weight.size()[1]/2)
                    dout = math.ceil(m.weight.size()[0]/2)
                    w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                    w0 = torch.cat((w0,-w0),dim=0)
                    w0 = torch.cat((w0,-w0),dim=1)
                    df1 = math.ceil((dim[2]-1)/2)
                    df2 = math.ceil((dim[3]-1)/2)
                    ww = torch.zeros(dim, device=m.weight.device)                
                    ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                    m.weight.data = ww

            if isinstance(m, nn.Linear):
                i+=1
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)                
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]
            

    def _add_noisy_init(self):
        for name, m, in self.named_modules():
            if isinstance(m, (nn.Conv2d, nn.Linear)):
                fan_in, fan_out = nn.init._calculate_fan_in_and_fan_out(m.weight)
                std = math.sqrt(2/fan_in)
                noise_scale = 0.05
                noisy = torch.zeros_like(m.weight, device=m.weight.device)
                nn.init.normal_(noisy, mean=0.0, std=std * noise_scale)
                m.weight.data = torch.where(m.weight.data==0, noisy, m.weight.data)
                
    def forward(self, x):
        out = F.relu(self.conv1(x))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out


def NFResNet18(input_shape, num_classes):
    return ResNet(BasicBlock, [2, 2, 2, 2], num_classes)

def NFResNet50(input_shape, num_classes):
    return ResNet(Bottleneck, [3, 4, 6, 3], num_classes)


def NFResNet101(input_shape, num_classes):
    return ResNet(Bottleneck, [3, 4, 23, 3], num_classes)


def NFResNet152(input_shape, num_classes):
    return ResNet(Bottleneck, [3, 8, 36, 3], num_classes)


