'''ResNet in PyTorch.

For Pre-activation ResNet, see 'preact_resnet.py'.

Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
# from Utils.load import device
from cmath import phase
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import math





class BottleneckLLSkip(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(BottleneckLLSkip, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)

        self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
        )

    def ortho_block_init(self):
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        dim3 = self.conv3.weight.shape
        delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        d31 = math.ceil((dim3[2]-1)/2)
        d32 = math.ceil((dim3[3]-1)/2)

        if len(self.shortcut) > 0:
            dim4 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim4[2]-1)/2)
            df2 = math.ceil((dim4[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim4[0]/2),math.ceil(dim4[1]/2))) - delta3 @ delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        delta3 = torch.cat((delta3, -delta3), dim=0)
        delta3 = torch.cat((delta3, -delta3), dim=1)
        self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
        self.conv3.weight.data[:, :, d31, d32] = delta3

        for m in self.modules():
            m.to(curr_device)

    
    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out = self.alpha * out + self.shortcut(x)
        out = F.relu(out)
        return out



class BasicBlockLLSkip(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlockLLSkip, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)

        
        self.shortcut = nn.Sequential(
            nn.Conv2d(in_planes, self.expansion*planes,
                        kernel_size=1, stride=stride, bias=False),
            nn.BatchNorm2d(self.expansion*planes)
        )

    def ortho_block_init(self):
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        if len(self.shortcut) > 0:
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        
        for m in self.modules():
            m.to(curr_device)

    def set_alpha(self, alpha):
        self.alpha = nn.Parameter(torch.tensor(alpha), requires_grad=True)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = out*self.alpha + self.shortcut(x)
        out = F.relu(out)
        return out


class BasicBlockLLSkipAlpha(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlockLLSkipAlpha, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)

        
        self.shortcut = nn.Sequential(
            nn.Conv2d(in_planes, self.expansion*planes,
                        kernel_size=1, stride=stride, bias=False),
            nn.BatchNorm2d(self.expansion*planes)
        )

    def ortho_block_init(self):
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        if len(self.shortcut) > 0:
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - self.alpha * delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        
        for m in self.modules():
            m.to(curr_device)

    def set_alpha(self, alpha):
        self.alpha = nn.Parameter(torch.tensor(alpha), requires_grad=True)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = out*self.alpha + self.shortcut(x)
        out = F.relu(out)
        return out



class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.alpha = nn.Parameter(torch.tensor(0.5), requires_grad=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def ortho_block_init(self, alpha):
        self.alpha.data = torch.tensor(alpha)
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        if len(self.shortcut) > 0:
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        
        for m in self.modules():
            m.to(curr_device)

    def looks_linear_uniform(self, alpha):
        self.alpha.data = torch.tensor(alpha)
        curr_device = self.conv1.weight.device
        dim1 = self.conv1.weight.shape
        delta1 = nn.init.kaiming_uniform_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
        dim2 = self.conv2.weight.shape
        delta2 = nn.init.kaiming_uniform_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
        d11 = math.ceil((dim1[2]-1)/2)
        d12 = math.ceil((dim1[3]-1)/2)
        d21 = math.ceil((dim2[2]-1)/2)
        d22 = math.ceil((dim2[3]-1)/2)
        if len(self.shortcut) > 0:
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.kaiming_uniform_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
        delta1 = torch.cat((delta1, -delta1), dim=0)
        delta1 = torch.cat((delta1, -delta1), dim=1)
        self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
        self.conv1.weight.data[:, :, d11, d12] = delta1
        delta2 = torch.cat((delta2, -delta2), dim=0)
        delta2 = torch.cat((delta2, -delta2), dim=1)
        self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
        self.conv2.weight.data[:, :, d21, d22] = delta2
        
        for m in self.modules():
            m.to(curr_device)


    def set_alpha(self, alpha):
        self.alpha = nn.Parameter(torch.tensor(alpha), requires_grad=True)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = out*self.alpha + self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(ResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layer_list = []
        for stride in strides:
            layer_list.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layer_list)

    def _ortho_block_init(self):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (BasicBlockLLSkip, BottleneckLLSkip)):
                m.ortho_block_init()
                print('Initialized Ortho Block init')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]


    def _looks_linear_uniform(self, alpha):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.kaiming_uniform_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (BasicBlockLLSkip)):
                m.looks_linear_uniform(alpha)
                print('Initialized Looks Linear Uniform')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.kaiming_uniform_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]



    def _set_alpha(self, alpha):
        for name, m in self.named_modules():
            if isinstance(m, (BasicBlockLLSkipAlpha)):
                m.set_alpha(alpha)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out



def ResNet18_llskip(input_shape, num_classes):
    return ResNet(BasicBlockLLSkip, [2, 2, 2, 2], num_classes)

def ResNet18_llskip_alpha(input_shape, num_classes):
    return ResNet(BasicBlockLLSkipAlpha, [2, 2, 2, 2], num_classes)

def ResNet50_llskip(input_shape, num_classes):
    return ResNet(BottleneckLLSkip, [3, 4, 6, 3], num_classes)


def ResNet101_llskip(input_shape, num_classes):
    return ResNet(BottleneckLLSkip, [3, 4, 23, 3], num_classes)
