import torch
import torch.nn as nn
# from .utils import load_state_dict_from_url
import math
import torch.nn.functional as F

class BottleneckLLIdentity(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(BottleneckLLIdentity, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion *
                               planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.alpha = nn.Parameter(torch.tensor(1.0), requires_grad=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                    nn.Conv2d(in_planes, self.expansion*planes,
                            kernel_size=1, stride=stride, bias=False),
                    nn.BatchNorm2d(self.expansion*planes)
            )

    def ortho_block_init(self):
        if len(self.shortcut) > 0:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            dim3 = self.conv3.weight.shape
            delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            d31 = math.ceil((dim3[2]-1)/2)
            d32 = math.ceil((dim3[3]-1)/2)

        
            dim4 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim4[2]-1)/2)
            df2 = math.ceil((dim4[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim4[0]/2),math.ceil(dim4[1]/2))) - delta3 @ delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
            delta1 = torch.cat((delta1, -delta1), dim=0)
            delta1 = torch.cat((delta1, -delta1), dim=1)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12] = delta1
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
            delta3 = torch.cat((delta3, -delta3), dim=0)
            delta3 = torch.cat((delta3, -delta3), dim=1)
            self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
            self.conv3.weight.data[:, :, d31, d32] = delta3

        else:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            
            dim2 = self.conv2.weight.shape
            dim3 = self.conv3.weight.shape
            delta3 = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            d31 = math.ceil((dim3[2]-1)/2)
            d32 = math.ceil((dim3[3]-1)/2)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12].diagonal().copy_(torch.ones_like(torch.diag(self.conv1.weight.data[:, :, d11, d12])))
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22].diagonal().copy_(torch.ones_like(torch.diag(self.conv2.weight.data[:, :, d21, d22])))
            delta3 = torch.cat((delta3, -delta3), dim=0)
            delta3 = torch.cat((delta3, -delta3), dim=1)
            identity = torch.zeros(dim3)
            identity.diagonal().copy_(torch.ones_like(torch.diag(identity[:, :, d31, d32])))
            self.conv3.weight.data = torch.zeros(self.conv3.weight.shape)
            self.conv3.weight.data[:, :, d31, d32] = delta3
            self.conv3.weight.data -= identity

        for m in self.modules():
            m.to(curr_device)

    
    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out = self.alpha * out + self.shortcut(x)
        out = F.relu(out)
        return out


class BasicBlockLLIdentity(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlockLLIdentity, self).__init__()
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.alpha = nn.Parameter(torch.tensor(0.5), requires_grad=True)

        
        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def ortho_block_init(self):
        if len(self.shortcut) > 0:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
        
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
            delta1 = torch.cat((delta1, -delta1), dim=0)
            delta1 = torch.cat((delta1, -delta1), dim=1)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12] = delta1
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
        
        else:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12].diagonal().copy_(torch.ones_like(torch.diag(self.conv1.weight.data[:, :, d11, d12])))
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            identity = torch.zeros(dim2)
            identity.diagonal().copy_(torch.ones_like(torch.diag(identity[:, :, d21, d22])))
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
            self.conv2.weight.data -= identity

        for m in self.modules():
            m.to(curr_device)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out = out*self.alpha + self.shortcut(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10):
        super(ResNet, self).__init__()
        self.in_planes = 64

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3,
                               stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layer_list = []
        for stride in strides:
            layer_list.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layer_list)

    def _ortho_block_init(self):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (BasicBlockLLIdentity, BottleneckLLIdentity)):
                m.ortho_block_init()
                print('Initialized Ortho Block init')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]


    
    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.avg_pool2d(out, 4)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out

def ResNet18_llidentity(input_shape, num_classes):
    return ResNet(BasicBlockLLIdentity, [2, 2, 2, 2], num_classes)

def ResNet50_llidentity(input_shape, num_classes):
    return ResNet(BottleneckLLIdentity, [3, 4, 6, 3], num_classes)
