import torch
import torch.nn as nn
# from .utils import load_state_dict_from_url
import math
import torch.nn.functional as F

'''
The scaled weight standardization convolutional block is defined
'''

class ScaledStdConv2d(nn.Conv2d):
    """Conv2d layer with Scaled Weight Standardization.
    Paper: `Characterizing signal propagation to close the performance gap in unnormalized ResNets` -
        https://arxiv.org/abs/2101.08692
    Adapted from timm: https://github.com/rwightman/pytorch-image-models/blob/4ea593196414684d2074cbb81d762f3847738484/timm/models/layers/std_conv.py
    """

    def __init__(self, in_channels, out_channels, kernel_size, stride=1, padding=0, dilation=1, groups=1,
                 bias=True, gain=True, gamma=math.sqrt(2 / (1 - (1 / math.pi))), eps=1e-5):
        super().__init__(
            in_channels, out_channels, kernel_size, stride=stride,
            padding=padding, dilation=dilation, groups=groups, bias=bias)
        self.gain = nn.Parameter(torch.ones(
            self.out_channels, 1, 1, 1)) if gain else None
        # gamma * 1 / sqrt(fan-in)
        self.scale = gamma * self.weight[0].numel() ** -0.5
        self.eps = eps

        self.register_buffer('weight_mask', torch.ones(self.weight.shape))
        if self.bias is not None:
            self.register_buffer('bias_mask', torch.ones(self.bias.shape))

    def get_weight(self):
    
        mean = torch.mean(
            self.weight, dim=[1, 2, 3], keepdim=True)
        std = torch.std(
            self.weight, dim=[1, 2, 3], keepdim=True, unbiased=False)
        weight = self.scale * (self.weight - mean) / (std + self.eps)

        if self.gain is not None:
            weight = weight * self.gain
        return weight

    def _conv_forward(self, input, weight, bias):
        if self.padding_mode != 'zeros':
            return F.conv2d(F.pad(input, self._padding_repeated_twice, mode=self.padding_mode),
                            weight, bias, self.stride,
                            _pair(0), self.dilation, self.groups)
        return F.conv2d(input, weight, bias, self.stride,
                        self.padding, self.dilation, self.groups)

    def forward(self, input):
        W = self.weight_mask * self.get_weight()
        if self.bias is not None:
            b = self.bias_mask * self.bias
        else:
            b = self.bias
        return self._conv_forward(input, W, b)


def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return ScaledStdConv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                         padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return ScaledStdConv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample nn downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.relu(out)

        out = self.conv2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample nn downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride
        self.alpha = nn.Parameter(torch.tensor(0.2), requires_grad=False)


    def ortho_block_init(self):
        if self.downsample is not None:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            delta1 = nn.init.orthogonal_(torch.empty(math.ceil(dim1[0]/2),math.ceil(dim1[1]/2)))
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
        
            dim3 = self.shortcut[0].weight.shape
            df1 = math.ceil((dim3[2]-1)/2)
            df2 = math.ceil((dim3[3]-1)/2)
            delta = nn.init.orthogonal_(torch.empty(math.ceil(dim3[0]/2),math.ceil(dim3[1]/2))) - delta2 @ delta1
            delta = torch.cat((delta, -delta), dim=0)
            delta = torch.cat((delta, -delta), dim=1)
            self.shortcut[0].weight.data = torch.zeros(self.shortcut[0].weight.shape)
            self.shortcut[0].weight.data[:, :, df1, df2] = delta
           
            delta1 = torch.cat((delta1, -delta1), dim=0)
            delta1 = torch.cat((delta1, -delta1), dim=1)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12] = delta1
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
        
        else:
            curr_device = self.conv1.weight.device
            dim1 = self.conv1.weight.shape
            
            dim2 = self.conv2.weight.shape
            delta2 = nn.init.orthogonal_(torch.empty(math.ceil(dim2[0]/2),math.ceil(dim2[1]/2)))
            d11 = math.ceil((dim1[2]-1)/2)
            d12 = math.ceil((dim1[3]-1)/2)
            d21 = math.ceil((dim2[2]-1)/2)
            d22 = math.ceil((dim2[3]-1)/2)
            self.conv1.weight.data = torch.zeros(self.conv1.weight.shape)
            self.conv1.weight.data[:, :, d11, d12].diagonal().copy_(torch.ones_like(torch.diag(self.conv1.weight.data[:, :, d11, d12])))
            delta2 = torch.cat((delta2, -delta2), dim=0)
            delta2 = torch.cat((delta2, -delta2), dim=1)
            identity = torch.zeros(dim2)
            identity.diagonal().copy_(torch.ones_like(torch.diag(identity[:, :, d21, d22])))
            self.conv2.weight.data = torch.zeros(self.conv2.weight.shape)
            self.conv2.weight.data[:, :, d21, d22] = delta2
            self.conv2.weight.data -= identity

        for m in self.modules():
            m.to(curr_device)

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.relu(out)

        out = self.conv3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out = self.alpha * out + identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(self, block, layer_list, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 norm_layer=None):
        super(ResNet, self).__init__()
        
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = ScaledStdConv2d(3, self.inplanes, kernel_size=7, stride=2, padding=3,
                                   bias=False)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layer_list[0])
        self.layer2 = self._make_layer(block, 128, layer_list[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layer_list[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layer_list[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        

        

    def _ortho_block_init(self):
        for name, m in self.named_modules():
            if name == 'conv1':
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0]/2)
                w0 = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                w0 = torch.cat((w0,-w0),dim=0)
                ww = torch.zeros(dim, device=m.weight.device)
                df1 = math.ceil((dim[2]-1)/2)
                df2 = math.ceil((dim[3]-1)/2)
                ww[:,:,df1,df2] = w0[:dim[0],:dim[1]]
                m.weight.data = ww

            if isinstance(m, (Bottleneck)):
                m.ortho_block_init()
                print('Initialized Ortho Block init')
            
            if isinstance(m, nn.Linear):
                dim = m.weight.size()
                din = math.ceil(m.weight.size()[1])
                dout = math.ceil(m.weight.size()[0])
                ww = nn.init.orthogonal_(torch.empty(dout, din, device=m.weight.device))
                ww = torch.cat((ww,-ww),dim=0)
                ww = torch.cat((ww,-ww),dim=1)
                m.weight.data = ww[:m.weight.size()[0],:m.weight.size()[1]]


    def _make_layer(self, block, planes, blocks, stride=1, dilate=False):
        norm_layer = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1

        downsample = nn.Sequential(
            conv1x1(self.inplanes, planes * block.expansion, stride)
            )
        
        
        layer_list = []
        layer_list.append(block(self.inplanes, planes, stride, downsample, self.groups,
                                self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        downsample = nn.Sequential(
            conv1x1(self.inplanes, self.inplanes, 1)
            )

        for _ in range(1, blocks):
            layer_list.append(block(self.inplanes, planes, 1, downsample, groups=self.groups,
                                    base_width=self.base_width, dilation=self.dilation,
                                    norm_layer=norm_layer))

        return nn.Sequential(*layer_list)

    def _forward_impl(self, x):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward(self, x):
        return self._forward_impl(x)


def _resnet(arch, block, layer_list):
    model = ResNet(block, layer_list)
    
    return model




def NFResNet101_imagenet(input_shape, num_classes):
    r"""ResNet-101 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_

    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet101', Bottleneck, [3, 4, 23, 3])


def NFResNet50_imagenet(input_shape, num_classes):
    r"""ResNet-50 model from
    `"Deep Residual Learning for Image Recognition" <https://arxiv.org/pdf/1512.03385.pdf>`_

    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
        progress (bool): If True, displays a progress bar of the download to stderr
    """
    return _resnet('resnet50', Bottleneck, [3, 4, 6, 3])


