function [acc,accVerif,accFals] = computeAccuracies(resFiles, ...
    numConfigs,testX,testT,options,epsilons,verifTimeout)
    % Number of runs
    N = length(resFiles);
    
    % Store accuracy scores
    acc = zeros(numConfigs,N);
    accVerif = zeros(length(epsilons),numConfigs,N);
    accFals = zeros(length(epsilons),numConfigs,N);

    v0 = size(testX,1);

    % Specify number of PGD steps.
    pgdIter = 40;

    % To speed up computations and reduce gpu memory, we only use single 
    % precision.
    inputDataClass = single(1);
    if options.nn.train.use_gpu
        % Data is moved to gpu.
        inputDataClass = gpuArray(inputDataClass);
    end

    % Allocate id-matrix for faster batch evaluation
    idMat = eye(v0,'like',inputDataClass);
    
    for r=1:length(resFiles)
        fprintf('Computing accuracies, run (%d/%d)...\n',r,length(resFiles));
        % Load results
        load(resFiles{r},'results');

        % Compute accuracy for each configuration
        for c=1:length(results)
            fprintf('--- config "%s" (%d/%d)...',results{c}.name,c,length(results));
            
            verifTimeout_ = verifTimeout(c,r);

            if verifTimeout_ < 0
                % Do not compute accuracies for this run.
                fprintf(' done [neg. timeout]\n');
                continue;
            end

            % Obtain trained neural network
            nn = results{c}.nn;
            if options.nn.train.use_gpu
                % move network to gpu
                nn.castWeights(gpuArray(single(1)));
            end

            % Get indices of layers for propagation. The final softmax layer is
            % considered as part of the loss function.
            idxLayer = 1:length(nn.layers);
            idxLayer = idxLayer( ...
                arrayfun(@(i) ~isa(nn.layers{i},'nnSoftmaxLayer'),idxLayer));
            
            % In each layer, store ids of active generators and identity 
            % matrices for fast adding of approximation errors.
            nn.prepareForZonoBatchEval(testX,options,idxLayer);

            % Compute clean accuracy.
            acc(c,r) = aux_computeCleanAccuracy(nn,testX,testT,options, ...
                inputDataClass);

            for j=1:length(epsilons)
                % Obtain perturbation radius
                epsilon = epsilons(j);

                propBatchSize = min( ...
                    options.nn.train.mini_batch_size, ...
                    options.nn.train.propagation_batch_size ...
                );
                while propBatchSize >= 0
                    try
                        % Reset random number generator.
                        rng('default')
                        % Constuct template generator matrix.
                        idBatch = repmat(idMat,1,1,propBatchSize);
                        % Compute accuracy.
                        [accVerif(j,c,r),accFals(j,c,r)] = ...
                            aux_computeVerifiedAndFalsifiedAccuracy(nn, ...
                                epsilon,testX,testT,idBatch,options, ...
                                    idxLayer,pgdIter,verifTimeout_);
                        % Clean up memory.
                        batchVars = {'xidBatch'};
                        clear(batchVars{:});
                        break;
                    catch e
                        % Try with new propagation batch size.
                        newPropBatchSize = floor(propBatchSize/2);
                        fprintf(' --- !!! Warning: reducing propagation batch size: %d -> %d\n',...
                            propBatchSize,newPropBatchSize);
                        if propBatchSize == 0
                            fprintf('!!! Failed to compute accuracy (%s). !!!', ...
                                results{c}.name);
                            break;
                        else
                            propBatchSize = newPropBatchSize;
                        end
                    end
                end
            end
            % gather network from gpu
            nn.castWeights(single(1));

            fprintf(' done\n');
        end
        % Clean up workspace
        clearvars results
    end
end

% _________________________________________________________________________
%% AUXILIARY FUNCTIONS
% -------------------------------------------------------------------------

function [accVerif,accFals,accVerif_,accFals_] = ...
        aux_computeVerifiedAndFalsifiedAccuracy(nn,epsilon,x,t, ...
    idBatch,options,idxLayer,pgdIter,verifTimeout)
    % Obtain target labels.
    [~,tLabels] = max(t);

    % Obtain mini batch size and propagation batch size.
    miniBatchSize = options.nn.train.mini_batch_size;
    propBatchSize = size(idBatch,3);

    v0 = size(x,1);
    if ~isempty(idBatch)
        idx1 = repmat(cast(1:v0,'like',idBatch),1,propBatchSize);
        idx2 = repelem(cast(1:propBatchSize,'like',idBatch),1,v0);
        % Scale non-zero entries by training perturbation noise
        nnzIdx = reshape(sub2ind(size(idBatch),idx1,idx1,idx2),[v0, propBatchSize]);
        % nnzIdx = cast(nnzIdx,'like',idBatch);
        % idBatch(nnzIdx) = epsilon*idBatch(nnzIdx);
    end

    % Total number of verified/falsified samples.
    nrVerified = 0;
    nrFalsified = 0;
    % Number of verified/falsified samples without nn.verify(..).
    nrVerified_ = 0;
    nrFalsified_ = 0;

    for l=1:miniBatchSize:size(x,2)
        % Compute indices for current batch
        miniBatchIdx = l:min(l + miniBatchSize - 1,size(x,2));
        thisBatchSize = length(miniBatchIdx);
        xBatch = cast(x(:,miniBatchIdx),'like',idBatch);
        tBatch = cast(t(:,miniBatchIdx),'like',idBatch);
        % Compute input bounds.
        xBatchL = max(xBatch - epsilon,0);
        xBatchU = min(xBatch + epsilon,1);
        noise = 1/2*(xBatchU - xBatchL);

        % 1. Try falsification with PGD.

        % Compute random initial start
        zInit = xBatch + epsilon*(2*rand(size(xBatch),'like',xBatch) - 1);
        % Compute perturbed inputs
        z = nn.computePGDAttack(zInit,tBatch,options,epsilon,pgdIter,...
            xBatchL,xBatchU,0.01,1,[],@(t,y) softmax(y) - t);
        % Compute predictions
        yBatch = nn.evaluate_(z,options);
        [~,kPred] = max(yBatch);
        % Count number of correct predictions
        falsified = (kPred ~= tLabels(miniBatchIdx));

        % 2. Try verification with IBP.

        % Obtain input interval
        xBatchIval = interval(xBatchL,xBatchU);
        yBatchIval = nn.evaluate_(xBatchIval,options,idxLayer);
        % Subtract lower bound of target from upper bounds of non-targets
        zBatch = (1-tBatch).*(yBatchIval.sup - ...
            reshape(pagemtimes(permute(tBatch,[1 3 2]),'transpose',...
                permute(yBatchIval.inf,[1 3 2]),'none'),[1 thisBatchSize]));
        % Count number of verified instances
        verified = all(zBatch <= 0);

        % 3. Try verification with Zonotopes.
        
        % Obtain number of output dimensions
        [vK,~] = size(tBatch);
        % Construct matrix for verification
        Ct = repmat(eye(vK,'like',tBatch),1,1,thisBatchSize);
        tMask = logical(tBatch);
        Ct(:,tMask) = [];
        Ct = permute(reshape(Ct,[vK vK-1 thisBatchSize]),[2 1 3]);
        Ct(:,tMask) = -1;
        Ct = reshape(Ct,[vK-1 vK thisBatchSize]);

        if ~isempty(idBatch)
            unknown = ~verified & ~falsified;
            if any(unknown) && ~options.nn.interval_center
                unknownIdx = find(unknown);
                for j=1:propBatchSize:length(unknownIdx)
                    % Compute indices for elements in current batch
                    propIdx_ = j:min(j+propBatchSize-1,length(unknownIdx));
                    propIdx = unknownIdx(propIdx_);
                    
                    % Construct input zonotope
                    xBatch_ = 1/2*(xBatchU(:,propIdx) + xBatchL(:,propIdx));
                    % if options.nn.interval_center
                    %     xBatch_ = permute(cat(3,xBatch_,xBatch_),[1 3 2]);
                    % end
                    GxBatch_ = idBatch(:,:,1:length(propIdx_));
                    GxBatch_(nnzIdx(:,1:length(propIdx_))) = ...
                        noise(:,1:length(propIdx_)).*GxBatch_(nnzIdx(:,1:length(propIdx_)));
        
                    % Compute output set for entire batch
                    [cyBatch,GyBatch] = nn.evaluateZonotopeBatch_(xBatch_,GxBatch_, ...
                        options,idxLayer);
                    % if options.nn.interval_center    
                    %     ycl = reshape(cyBatch(:,1,:),size(cyBatch,[1 3]));
                    %     ycu = reshape(cyBatch(:,2,:),size(cyBatch,[1 3]));
                    %     cyBatch = 1/2*(ycu + ycl);
                    % end
                    % Multiply matrix C with output sets
                    Ct_ = Ct(:,:,propIdx);
                    cyBatchDiff = squeeze(pagemtimes(Ct_,permute(cyBatch,[1 3 2])));
                    GyBatchDiff = pagemtimes(Ct_,GyBatch);
                    rDiff = reshape(sum(abs(GyBatchDiff),2),size(cyBatchDiff));
                    % Check verification.
                    verified(propIdx) = all(cyBatchDiff + rDiff <= 0);
                end
            end
        end
        % Aggregate results (without nn.verify(...)).
        nrVerified = nrVerified + sum(verified);
        nrFalsified = nrFalsified + sum(falsified);

        % 4. Try splitting for unknown instances.
        unknown = ~verified & ~falsified;
        unknownIdx = find(unknown);
        for idx=unknownIdx
            % Copy options.
            options_ = options;
            while true && verifTimeout > 0
                try
                    % Do verification.
                    [res,~,~] = nn.verify(xBatch(:,idx),noise(:,idx), ...
                        Ct(:,:,idx),0,true,options_,verifTimeout,true);
                    % Write results.
                    % fprintf("Result: '%s' (%d)\n",res,idx);
                    if strcmp(res,'VERIFIED')
                        verified(idx) = 1;
                    elseif strcmp(res,'COUNTEREXAMPLE')
                        falsified(idx) = 1;
                    end
                    break;
                catch e
                    if ismember(e.identifier, ...
                            {'parallel:gpu:array:pmaxsize', ...
                                'parallel:gpu:array:OOM', ...
                                'MATLAB:array:SizeLimitExceeded',...
                                'MATLAB:nomem'}) ...
                            && options_.nn.train.mini_batch_size > 1
                        options_.nn.train.mini_batch_size = ...
                            floor(1/2*options_.nn.train.mini_batch_size);
                        fprintf('--- OOM error: half batchSize %d...\n', ...
                            options_.nn.train.mini_batch_size);
                    else
                        % Print the error message. 
                        % fprintf(e.message);
                        % No result.
                        break;
                    end
                end
            end
        end

        % Aggregate results.
        nrVerified_ = nrVerified_ + sum(verified);
        nrFalsified_ = nrFalsified_ + sum(falsified);

        % To save memory, we clear all variables that are only used
        % during the batch computation
        batchVars = {'xBatch','tBatch','cyBatch','GyBatch','cyBatchDiff',...
            'GyBatchDiff','rDiff','xBatchIval','yBatchIval','zBatch'};
        clear(batchVars{:});
    end
    accVerif = nrVerified/size(t,2);
    accFals = 1 - nrFalsified/size(t,2);
    accVerif_ = nrVerified_/size(t,2);
    accFals_ = 1 - nrFalsified_/size(t,2);

    % Print nn.verify(...) contribution.
    fprintf(" --- [%.2f,%.2f] -> nn.verify(timeout [s]: %d) -> [%.2f,%.2f]... ", ...
        accVerif*100,accFals*100,verifTimeout,accVerif_*100,accFals_*100);
end

function acc = aux_computeCleanAccuracy(nn,x,t,options,inputDataClass)
    % Obtain miniBatchSize
    miniBatchSize = options.nn.train.mini_batch_size;
    % Obtain target labels
    [~,tLabels] = max(t);

    correct = 0;
    for l=1:miniBatchSize:size(x,2)
        % Compute indices for current batch
        miniBatchIdx = l:min(l + miniBatchSize,size(x,2)) - 1;
        xBatch = cast(x(:,miniBatchIdx),'like',inputDataClass);
        % Compute predictions
        yBatch = nn.evaluate_(xBatch,options);
        [~,kPred] = max(yBatch);
        % Count number of correct predictions
        correct = correct + sum(kPred == tLabels(miniBatchIdx));

        % To save memory, we clear all variables that are only used
        % during the batch computation
        batchVars = {'xBatch','tBatch','yBatch'};
        clear(batchVars{:});
    end
    % Compute accurarcy
    acc = correct/size(x,2);
end
