function [configs,epsilons,options,mu,sigma] = ...
        getTrainingConfigurations(dataset,useGpu,shortRun)
    fprintf('Loading training configuration "%s"...',dataset);
    switch dataset
        case 'mnist'
            % Obtain training configurations.
            configs = getMNISTConfigs(); 
            % Specify perturbation radii.
            epsilons = [0.1];
            % Specify normalization.
            mu = 0; ... 0.1306; ... 
            sigma = 1; ... 0.3081; ... 
            % Create training parameters.
            options.nn = struct( ...
                'use_approx_error',true,...
                'poly_method','bounds',... 
                'train',struct( ...
                    'use_gpu',useGpu,...
                    'optim',nnAdamOptimizer(5e-4, ...
                        0.9,0.999,1e-8,1e-6,false,[],1,10),...
                    'lr_decay',0.2,...
                    'lr_decay_epoch',[50 60],...
                    'shuffle_data','every_epoch',...
                    'loss','softmax+log',...
                    'mini_batch_size',256,...
                    'max_epoch',70,... SABR/Shi (IBP)
                    'warm_up',1,...
                    'ramp_up',20,...
                    'propagation_batch_size',inf ...
                )...
            );
            if shortRun
              % Adjust epochs.
              options.nn.train.max_epoch = 10;
              options.nn.train.ramp_up = 5;
              options.nn.train.lr_decay_epoch = [6 8];
            end
        case 'svhn'
            % Obtain training configurations.
            configs = getSVHNConfigs();
            % Specify perturbation radii.
            epsilons = [0.01];
            % Specify normalization.
            mu = permute([0.4377 0.4438 0.4728],[1 3 2]); % 0.5;
            sigma = permute([0.1980 0.2010 0.1970],[1 3 2]); % 0.2;
            % Create training parameters.
            options.nn = struct( ...
                'use_approx_error',true,...
                'poly_method','bounds',... 
                'train',struct( ...
                    'use_gpu',useGpu,...
                    'optim',nnAdamOptimizer(5e-4, ...
                        0.9,0.999,1e-8,1e-6,false,[],1,10),...
                    'lr_decay',0.2,...
                    'lr_decay_epoch',[50 60],...
                    'shuffle_data','every_epoch',...
                    'loss','softmax+log',...
                    'mini_batch_size',128,...
                    'max_epoch',70,...
                    'warm_up',20,...
                    'ramp_up',50,...
                    'propagation_batch_size',inf ...
                )...
            );
        case 'cifar10'
            % Obtain training configurations.
            configs = getCIFAR10Configs();
            % Specify perturbation radii.
            epsilons = [2/255];
            % Specify normalization.
            mu = permute([0.4914 0.4822 0.4465],[1 3 2]); % 0.4734;
            sigma = permute([0.2470 0.2435 0.2616],[1 3 2]); % 0.2516;
            % Create training parameters
            options.nn = struct( ...
                'use_approx_error',true,...
                'poly_method','bounds',... 
                'train',struct( ...
                    'use_gpu',useGpu,...
                    'optim',nnAdamOptimizer(5e-4, ...
                        0.9,0.999,1e-8,1e-6,false,[],1,10),...
                    'lr_decay',0.2,...
                    'lr_decay_epoch',[120 140],...
                    'shuffle_data','every_epoch',...
                    'loss','softmax+log',...
                    'mini_batch_size',128,...
                    'warm_up',20,...
                    'ramp_up',120,...
                    'propagation_batch_size',inf ...
                )...
            );
        case 'tinyImageNet'
            % Obtain training configurations.
            configs = getTinyImageNetConfigs();
            % Specify perturbation radii.
            epsilons = [1/255];
            % Specify normalization.
            mu = permute([0.4802 0.4481 0.3975],[1 3 2]);
            sigma = permute([0.2764 0.2689 0.2816],[1 3 2]);
            % Create training parameters
            options.nn = struct( ...
                'use_approx_error',true,...
                'poly_method','bounds',... 
                'train',struct( ...
                    'use_gpu',useGpu,...
                    'optim',nnAdamOptimizer(5e-4, ...
                        0.9,0.999,1e-8,1e-6,false,[],1,10),...
                    'lr_decay',0.2,...
                    'lr_decay_epoch',[50 60],...
                    'shuffle_data','every_epoch',...
                    'loss','softmax+log',...
                    'mini_batch_size',128,...
                    'max_epoch',70,...
                    'warm_up',20,...
                    'ramp_up',50,...
                    'propagation_batch_size',inf ...
                )...
            );
        otherwise
            error('Invalid name for dataset: %s',dataset)
    end

    % Set default training parameters.
    options = nnHelper.validateNNoptions(options);

    fprintf(' done\n');
end

%% Auxiliary Functions. ---------------------------------------------------

function configs = getMNISTConfigs()
    configs = {};

    configs{end+1} = struct(...
        'method','point',...
        'name','point'...
    );

    % configs{end+1} = struct(...        
    %     'method','madry',...
    %     'name','Madry',...
    %     'noise',0.1,...
    %     'pgd_iterations',8, ... % originally 40
    %     'pgd_stepsize',0.5,... % originally 0.001
    %     'pgd_stepsize_decay',0.1,... % originally 1
    %     'pgd_stepsize_decay_iter',[4 7]... % originally []
    % );

    configs{end+1} = struct(...
        'method','gowal',...
        'name','Gowal (IBP; eps=0.1, kappa=0.5)',...
        'noise',0.1,...
        'kappa',0.5 ...
    );

    configs{end+1} = struct(...
        'method','sabr',...
        'name','SABR',...
        'noise',0.1,...
        'lambda',0.4,...
        'pgd_iterations',8, ...
        'pgd_stepsize',0.5,...
        'pgd_stepsize_decay',0.1,...
        'pgd_stepsize_decay_iter',[4 7] ...
    );

    configs{end+1} = struct(...
        'method','trades',...
        'name','TRADES',...
        'noise',0.1,...
        'lambda',1/6,...
        'pgd_iterations',8, ... % originally 20
        'pgd_stepsize',0.5,... % originally 0.001
        'pgd_stepsize_decay',0.1,... % originally 1
        'pgd_stepsize_decay_iter',[4 7]... % originally []
    );

    configs{end+1} = struct(...
        'method','set',...
        'name','Set (fgsm2-10-10; eps=0.1; tau=0.1)',...
        'poly_method','bounds',...
        'approx_error_order','length',...
        'noise',0.1,...
        'tau',0.1,...
        'volume_heuristic','f-radius',...
        'zonotope_weight_update','sum',...
        'init_gens','fgsm2',...
        'num_init_gens',10,...
        'num_approx_err',10,...
        'exact_backprop',true,...
        'interval_center',true...
    );

    % configs{end+1} = struct(...
    %     'method','set',...
    %     'name','Set (fgsm2-10-10; eps=0.1; tau=0.1; no grad-clip)',...
    %     'poly_method','bounds',...
    %     'approx_error_order','length',...
    %     'noise',0.1,...
    %     'tau',0.1,...
    %     'volume_heuristic','set-eval',...
    %     'zonotope_weight_update','sum',...
    %     'init_gens','fgsm2',...
    %     'num_init_gens',10,...
    %     'num_approx_err',10,...
    %     'exact_backprop',true,...
    %     'interval_center',true,...
    %     'optim',nnAdamOptimizer(5e-4, ...
    %         0.9,0.999,1e-8,1e-6,false,[],1) ...
    % );

    % configs{end+1} = struct(...
    %     'method','set',...
    %     'name','Set (fgsm2-10-10; eps=0.1; tau=0.1; f-radius)',...
    %     'poly_method','bounds',...
    %     'approx_error_order','length',...
    %     'noise',0.1,...
    %     'tau',0.1,...
    %     'volume_heuristic','f-radius',...
    %     'zonotope_weight_update','sum',...
    %     'init_gens','fgsm2',...
    %     'num_init_gens',10,...
    %     'num_approx_err',10,...
    %     'exact_backprop',true,...
    %     'interval_center',true...
    % );

    % configs{end+1} = struct(...
    %     'method','set',...
    %     'name','Set (sens-10-10; eps=0.1; tau=0.1; set-eval)',...
    %     'poly_method','bounds',...
    %     'approx_error_order','length',...
    %     'noise',0.1,...
    %     'tau',0.01,...
    %     'volume_heuristic','set-eval',...
    %     'zonotope_weight_update','sum',...
    %     'init_gens','sensitivity',...
    %     'num_init_gens',10,...
    %     'num_approx_err',10,...
    %     'exact_backprop',true,...
    %     'interval_center',true...
    % );

    % % Ablation Study. ----------------------------------------------------- 
    % --- 1. Weighting in set-based loss.
    taus = [0 0.1 0.2 0.3];
    for tau=taus
        % configs{end+1} = struct(...
        %     'method','set',...
        %     'name',sprintf('Set (fgsm-10+10; eps=0.1; tau=%.2f)',tau),...
        %     'poly_method','bounds',...
        %     'approx_error_order','length',...
        %     'noise',0.1,...
        %     'tau',tau,...
        %     'volume_heuristic','f-radius',...
        %     'zonotope_weight_update','sum',...
        %     'init_gens','fgsm2',...
        %     'num_init_gens',10,...
        %     'num_approx_err',10,...
        %     'exact_backprop',true,...
        %     'interval_center',true,...
        %     'lr_decay_epoch',[6 8],...
        %     'mini_batch_size',32,...
        %     'max_epoch',9,...
        %     'warm_up',1,...
        %     'ramp_up',3 ...
        % );

        % configs{end+1} = struct(...
        %     'method','set',...
        %     'name',sprintf('Set (fgsm2-10+10; eps=0.1; tau=%.2f; set-eval)',tau),...
        %     'poly_method','bounds',...
        %     'approx_error_order','length',...
        %     'noise',0.1,...
        %     'tau',tau,...
        %     'volume_heuristic','set-eval',...
        %     'zonotope_weight_update','sum',...
        %     'init_gens','fgsm2',...
        %     'num_init_gens',10,...
        %     'num_approx_err',10,...
        %     'exact_backprop',true,...
        %     'interval_center',true ...
        % );
    end

    % --- 2. Input sets. {'fgsm','l_inf'} ['fgsm' from (1)]
    % configs{end+1} = struct(...
    %     'method','set',...
    %     'name','Set (l_inf+10; eps=0.1; tau=0.1)',...
    %     'poly_method','bounds',...
    %     'approx_error_order','length',...
    %     'noise',0.1,...
    %     'tau',0.1,...
    %     'volume_heuristic','f-radius',...
    %     'zonotope_weight_update','sum',...
    %     'init_gens','l_inf',...
    %     'num_init_gens',inf,...
    %     'num_approx_err',10,...
    %     'exact_backprop',true,...
    %     'interval_center',true ...
    % );

    % --- 3. Set-propagation method. 
    % {'bounds','singh','IBP'} ['bounds' from (1)]
    % configs{end+1} = struct(...
    %         'method','set',...
    %         'name','Set (fgsm2-10+10; eps=0.1; tau=0.1; singh)',...
    %         'poly_method','singh',...
    %         'approx_error_order','length',...
    %         'noise',0.1,...
    %         'tau',0.1,...
    %         'volume_heuristic','set-eval',...
    %         'zonotope_weight_update','sum',...
    %         'init_gens','fgsm2',...
    %         'num_init_gens',10,...
    %         'num_approx_err',10,...
    %         'exact_backprop',true,...
    %         'interval_center',true ...
    % );
    % configs{end+1} = struct(...
    %     'method','set',...
    %     'name','Set (fgsm3-10+0; eps=0.1; tau=0.1; ival)',...
    %     'poly_method','bounds',...
    %     'approx_error_order','length',...
    %     'noise',0.1,...
    %     'tau',0.1,...
    %     'volume_heuristic','set-eval',...
    %     'zonotope_weight_update','sum',...
    %     'init_gens','fgsm3',...
    %     'num_init_gens',10,...
    %     'num_approx_err',0,...
    %     'exact_backprop',true,...
    %     'interval_center',true ...
    % );
end

function configs = getSVHNConfigs()
    
    configs = {};

    configs{end+1} = struct(...
        'method','point',...
        'name','point'...
    );

    % configs{end+1} = struct(...        
    %     'method','madry',...
    %     'name','Madry',...
    %     'noise',0.01,...
    %     'pgd_iterations',8, ... % originally 40
    %     'pgd_stepsize',0.5,... % originally 0.001
    %     'pgd_stepsize_decay',0.1,... % originally 1
    %     'pgd_stepsize_decay_iter',[4 7]... % originally []
    % );

    configs{end+1} = struct(...
        'method','gowal',...
        'name','Gowal (IBP)',...
        'noise',0.01,...
        'kappa',0.5 ...
    );

    configs{end+1} = struct(...
        'method','sabr',...
        'name','SABR',...
        'noise',0.01,...
        'lambda',0.2,...
        'pgd_iterations',8, ...
        'pgd_stepsize',0.5,...
        'pgd_stepsize_decay',0.1,...
        'pgd_stepsize_decay_iter',[4 7] ...
    );

    configs{end+1} = struct(...
        'method','trades',...
        'name','TRADES',...
        'noise',0.01,...
        'lambda',1/6,...
        'pgd_iterations',8, ... % originally 20
        'pgd_stepsize',0.5,... % originally 0.001
        'pgd_stepsize_decay',0.1,... % originally 1
        'pgd_stepsize_decay_iter',[4 7]... % originally []
    );

    configs{end+1} = struct(...
        'method','set',...
        'name','Set (fgsm2-10-10; eps=0.01; tau=0.01)',...
        'poly_method','bounds',...
        'approx_error_order','length',...
        'noise',0.01,...
        'tau',0.01,...
        'volume_heuristic','f-radius',...
        'zonotope_weight_update','sum',...
        'init_gens','fgsm2',...
        'num_init_gens',10,...
        'num_approx_err',10,...
        'exact_backprop',true,...
        'interval_center',true ...
    );
end

function configs = getCIFAR10Configs()

    dataAug = @(x) aux_augTrainXRandCropAndFlip(x,[36 36 3],[32 32 3]);

    configs = {};

    configs{end+1} = struct(...
        'method','point',...
        'name','point',...
        'data_augmentation',dataAug...
    );

    % configs{end+1} = struct(...        
    %     'method','madry',...
    %     'name','Madry',...
    %     'noise',2/255,...
    %     'pgd_iterations',8, ... % originally 40
    %     'pgd_stepsize',0.5,... % originally 0.001
    %     'pgd_stepsize_decay',0.1,... % originally 1
    %     'pgd_stepsize_decay_iter',[4 7],... % originally []
    %     'data_augmentation',dataAug...
    % );

    configs{end+1} = struct(...
        'method','gowal',...
        'name','Gowal (IBP)',...
        'noise',2/255,...
        'kappa',0.5,...
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','sabr',...
        'name','SABR',...
        'noise',2/255,...
        'lambda',0.1,...
        'pgd_iterations',8, ...
        'pgd_stepsize',0.5,...
        'pgd_stepsize_decay',0.1,...
        'pgd_stepsize_decay_iter',[4 7],...
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','trades',...
        'name','TRADES',...
        'noise',2/255,...
        'lambda',1/6,...
        'pgd_iterations',8, ... % originally 20
        'pgd_stepsize',0.5,... % originally 0.001
        'pgd_stepsize_decay',0.1,... % originally 1
        'pgd_stepsize_decay_iter',[4 7],... % originally []
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','set',...
        'name','Set (fgsm2-10-10; eps=2/255; tau=0.005)',...
        'poly_method','bounds',...
        'approx_error_order','length',...
        'noise',2/255,...
        'tau',0.005,...
        'volume_heuristic','f-radius',...
        'zonotope_weight_update','sum',...
        'init_gens','fgsm2',...
        'num_init_gens',10,...
        'num_approx_err',10,...
        'exact_backprop',true,...
        'interval_center',true,...
        'data_augmentation',dataAug...
    );
end

function configs = getTinyImageNetConfigs()

    dataAug = @(x) aux_augTrainXRandCropAndFlip(x,[64 64 3],[56 56 3]);

    configs = {};

    configs{end+1} = struct(...
        'method','point',...
        'name','point',...
        'data_augmentation',dataAug...
    );
    % 
    % configs{end+1} = struct(...        
    %     'method','madry',...
    %     'name','Madry',...
    %     'noise',1/255,...
    %     'pgd_iterations',8, ... % originally 40
    %     'pgd_stepsize',0.5,... % originally 0.001
    %     'pgd_stepsize_decay',0.1,... % originally 1
    %     'pgd_stepsize_decay_iter',[4 7],... % originally []
    %     'data_augmentation',dataAug...
    % );

    configs{end+1} = struct(...
        'method','gowal',...
        'name','Gowal (IBP)',...
        'noise',1/255,...
        'kappa',0.5,...
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','sabr',...
        'name','SABR',...
        'noise',1/255,...
        'lambda',0.4,...
        'pgd_iterations',8, ...
        'pgd_stepsize',0.5,...
        'pgd_stepsize_decay',0.1,...
        'pgd_stepsize_decay_iter',[4 7],...
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','trades',...
        'name','TRADES',...
        'noise',1/255,...
        'lambda',1/6,...
        'pgd_iterations',8, ... % originally 20
        'pgd_stepsize',0.5,... % originally 0.001
        'pgd_stepsize_decay',0.1,... % originally 1
        'pgd_stepsize_decay_iter',[4 7],... % originally []
        'data_augmentation',dataAug...
    );

    configs{end+1} = struct(...
        'method','set',...
        'name','Set (fgsm2-10-10; eps=1/255; tau=0.1)',...
        'poly_method','bounds',...
        'approx_error_order','length',...
        'noise',1/255,...
        'tau',0.1,...
        'volume_heuristic','f-radius',...
        'zonotope_weight_update','sum',...
        'init_gens','fgsm2',...
        'num_init_gens',10,...
        'num_approx_err',10,...
        'exact_backprop',true,...
        'interval_center',true,...
        'propagation_batch_size',inf,...
        'data_augmentation',dataAug...
    );
end

% Auxiliary Function for data augmentation. -------------------------------

function x_ = aux_augTrainXRandCropAndFlip(x,imgSize,cropSize)
    % Obtain batch size.
    [~,bs] = size(x);
    % Apply random crops and flips.
    imgs_ = applyCropsAndFlips(reshape(x,[imgSize bs]),cropSize,'random',true);
    % Reshape images to vectors.
    x_ = reshape(imgs_,[],bs);
end