function trainModels(dataPath,resultsPath,dataset,nnSize,N,verbose)

% 0. CONFIGURATION. -------------------------------------------------------

nnActFun = 'relu'; % options: {'relu','tanh'}

% Use GPU for training, when available.
useGpu = gpuDeviceCount('available') > 0;
fprintf('GPU available: %d\n',useGpu);

fprintf('Running evaluation for "%s (%s)" on "%s" [%s]\n', ...
    nnSize,nnActFun,dataset,datetime('now','Format','yyyy-MMM-dd-HH-mm-ss'));
fprintf('Saving results in %s\n',resultsPath);

% Restrict number of CPU threads to 1.
maxNumCompThreads(1);

% 1. TRAINING NEURAL NETWORKS. --------------------------------------------

% Check for only short run.
shortRun = endsWith(resultsPath,'-short/evaluation');

% Load training configurations.
[configs,epsilons,options,mu,sigma] = ...
    getTrainingConfigurations(dataset,useGpu,shortRun);

% Number of configurations.
numConfigs = length(configs);

% Load the dataset.
[trainX,trainT,~,valX,valT,~,testX,testT,~,inSize] ...
    = loadDataset(dataset,dataPath);

% Check directory contains trained networks; find all result files.
[resFiles,~] = findAllResultFiles(resultsPath);

% Specify seeds.
seeds = 1:N;

if isempty(resFiles)
    % Main Training Loop --------------------------------------------------
    for r=1:N
        fprintf('Started training run (%d/%d)...\n',r,N);
        % Initialize struct to store training results.
        results = {};
    
        % Train Configurations --------------------------------------------
        for c=1:length(configs)
            % Obtain current configuration
            config = configs{c};
            % Obtain training parameters for current configuration 
            [configOptions,configNNSize] = ...
                optionsFromConfig(options,config,nnSize);

            % Initialize neural network -----------------------------------
            % Use the index of the run to seed the initialization of the 
            % weights.
            seed = seeds(r);
            nnInit = initializeNeuralNetwork(configNNSize,nnActFun, ...
                inSize,size(trainT,1),'shi',seed,mu,sigma);

            % Copy initialized neural network
            nn = nnInit.copyNeuralNetwork();
            nn.setInputSize(inSize);
    
            res = struct();
            res.options = configOptions;
            fprintf('--- Started training config "%s" (%d/%d)...', ...
                config.name,c,numConfigs);
            if verbose
                fprintf('\n');
            end
            try
                % Reset random number generator.
                rng(seed);
                % Train neural network
                [res.loss,trainingTime] = nn.train(trainX,trainT, ...
                    valX,valT,configOptions,verbose);
                fprintf(' done (training time: %.2f [s])\n',trainingTime);
            catch e
                res.loss = 0;
                trainingTime = -1;
                fprintf(' training failed! \n --- %s [%d]: %s\n', ...
                    e.stack(1).name,e.stack(1).line,e.message);
                % Store exception.
                res.exception = e;
            end
            % Store results
            res.trainingTime = trainingTime;
            res.nn = nn;
            res.seed = seed;
            res.name = config.name;
            % Store trained neural network for configuration
            results{c} = res;
        end
        fprintf('Training run (%d/%d) done. Saving results...',r,N);
        % Save results for all configurations (version 7.3 enables storing
        % large files: '-v7.3')
        save(sprintf('%s/run-%s',resultsPath,int2str(r)),'results','-v7.3');
        fprintf(' done\n');
    end
else
    fprintf('Found trained networks...\n');
end

% 2. COMPUTE ACCURACIES. --------------------------------------------------

if isfile(sprintf('%s/accs.mat',resultsPath))
    % Load computed accuracies.
    load(sprintf('%s/accs',resultsPath),'acc','accVerif','accFals', ...
        'epsilons','names')
else
    % Disable backpropagation
    options.nn.train.backprop = false;
    % Change approximation method to minimize size of the output set.
    options.nn.poly_method = 'bounds';
    % Propagate full sets.
    options.nn.train.propagation_batch_size = options.nn.train.mini_batch_size;
    options.nn.use_approx_error = true;
    options.nn.approx_error_order = 'sequential'; ... 'length'; ... 
    options.nn.train.num_init_gens = size(testX,1);
    options.nn.interval_center = false; ... 
    options.nn.train.num_approx_err = inf; ... 500; ... 
    
    % Find all result files.
    [resFiles,names] = findAllResultFiles(resultsPath);
    % Update number of runs.
    N = length(resFiles);
    % Update number of configurations.
    numConfigs = length(names);
    
    % Verify timeout.
    verifyTimeout = zeros(numConfigs,N);
    
    % Compute accuracies.
    [acc,accVerif,accFals] = computeAccuracies(resFiles,numConfigs, ...
        testX,testT,options,epsilons,verifyTimeout);
    
    % Save computed accuracies.
    fprintf('Saving accuracies...');
    save(sprintf('%s/accs',resultsPath),'acc','accVerif','accFals', ...
        'epsilons','names'); ... 
    fprintf(' done\n');
end

% Extract fastest training times.
[trainTime,names] = extractTrainTimes(resFiles,numConfigs);

% 3. GENERATE TABLES & FIGURES --------------------------------------------

fprintf('Generating results table...\n');
% Close log-text file.
% diary off

% filename = sprintf('%s/accs-table.txt',savePath);
% diary(filename)

header = arrayfun(@(e) { ...
    sprintf('falsified Acc. (eps=%.1f)',e), ...
    sprintf('fast-verif. Acc. (eps=%.1f)',e),'[max]', ...
},epsilons,'UniformOutput',false);
colFormat = arrayfun(@(e) {'sum{%.2f & %.2f}','sum{%.2f & %.2f}'}, ...
    epsilons,'UniformOutput',false);
% Init table,
table = CORAtable('modern', ...
    ['Method',{'Accuracy'},header{:}], ...
    ['s',{'sum{%.2f & %.2f}'},colFormat{:},'.2f'], ...
    'ColumnWidths',[max(cellfun(@length,names)),12,repelem(12,2*length(epsilons)) 4]);

% print table
table.printHeader()
for i=1:length(names)
    if ~isempty(epsilons)
        % Only select top-3 runs (based on verified accuracy).
        [~,runsIdx] = maxk(accFals(:,i,:) + accVerif(:,i,:),3,3);
        a = [accFals(:,i,runsIdx) accVerif(:,i,runsIdx)];
        a_ = permute(a,[3 2 1])*100;
        maxVerif = max(a_(:,2));
        table.printContentRow([names{i} acc(i,runsIdx)*100 ...
            num2cell(a_(:,:)',2)' maxVerif]);
    else
        table.printContentRow([names{i} acc(i,:)*100]);
    end
end
table.printFooter();

% Close table-text file.
% diary off

fprintf(' done\n');

fprintf('Generating training time table...\n');
% Init table,
table = CORAtable('modern',{'Method','(min) Train Time [s / Epoch]'}, ...
    {'s','.1f'},'ColumnWidths',[max(cellfun(@length,names)),0]);
% print table
table.printHeader()
for i=1:length(names)
    table.printContentRow({names{i} trainTime(i)/options.nn.train.max_epoch});
end
table.printFooter();

% Close table-text file.
% diary off

fprintf(' done\n');

end

% Auxiliary Functions. ----------------------------------------------------

function [configOptions,nnSize] = optionsFromConfig(options,config,nnSize)
    % make copy of training parameters
    configOptions = options;
    % transfer all config fields
    fieldNames = fieldnames(config);
    for i=1:length(fieldNames)
        if strcmp(fieldNames{i},'interval_center') || ...
                strcmp(fieldNames{i},'poly_method')|| ...
                strcmp(fieldNames{i},'approx_error_order')
            configOptions.nn.(fieldNames{i}) = config.(fieldNames{i});
        else
            configOptions.nn.train.(fieldNames{i}) = config.(fieldNames{i});
        end
    end
    if isfield(config,'nn_size')
        nnSize = config.nn_size;
    end
end

function [resFiles,names] = findAllResultFiles(savePath)
    % Store names of the configurations.
    names = {};
    % Find all results.
    runIdx = 1;
    resFiles = {};
    files = dir(savePath);
    for i=1:length(files)
        filename = files(i).name;
        % Ignore files with name 'rob' and 'acc' (these files store the
        % computed verified and PGD accuracy).
        if ~strcmp(filename,'.') && ~strcmp(filename,'..') ... 
                && ~startsWith(filename,'accs') ...
            resFiles{runIdx} = sprintf('%s/%s',savePath,filename);
            runIdx = runIdx + 1;

            if isempty(names)
                % Load results.
                load(resFiles{end},'results');
                % Extract configuration names.
                for c=1:length(results)
                    % Store names of the configurations
                    names = [names; results{c}.name];
                end
            end
        end
    end
end

function foundAcc = findComputedAccuracies(savePath)
    files = dir(savePath);
    for i=1:length(files)
        filename = files(i).name;
        if strcmp(filename,'accs') 
            % Found computed accuracies.

        end
    end
end
