#!/usr/bin/env python
# coding=utf-8

import argparse
import contextlib
import gc
import logging
import math
import os
import random
import shutil
from pathlib import Path
import json
import cv2
# import imgaug.augmenters as iaa

import accelerate
import numpy as np
import torch
import torch.nn.functional as F
import torch.utils.checkpoint
from torch.utils.data import Dataset
import transformers
from accelerate import Accelerator
from accelerate.logging import get_logger
from accelerate.utils import ProjectConfiguration, set_seed
from datasets import load_dataset
from huggingface_hub import create_repo, upload_folder
from packaging import version
from PIL import Image, ImageDraw
from torchvision import transforms
from tqdm.auto import tqdm
from transformers import AutoTokenizer, PretrainedConfig
from einops import rearrange, repeat

import diffusers
from diffusers import (
    AutoencoderKL,
    BrushNetModel,
    DDPMScheduler,
    StableDiffusionBrushNetPipeline,
    UNet2DConditionModel,
    UniPCMultistepScheduler,
)
from diffusers.optimization import get_scheduler
from diffusers.utils import check_min_version, is_wandb_available
from diffusers.utils.hub_utils import load_or_create_model_card, populate_model_card
from diffusers.utils.import_utils import is_xformers_available
from diffusers.utils.torch_utils import is_compiled_module

from omegaconf import OmegaConf

import datetime
if is_wandb_available():
    import wandb

# Will error if the minimal version of diffusers is not installed. Remove at your own risks.
# check_min_version("0.27.0.dev0")

def check_args(args):
    if args.dataset_name is None and args.train_data_dir is None:
        raise ValueError("Specify either `--dataset_name` or `--train_data_dir`")

    if args.dataset_name is not None and args.train_data_dir is not None:
        raise ValueError("Specify only one of `--dataset_name` or `--train_data_dir`")

    if args.proportion_empty_prompts < 0 or args.proportion_empty_prompts > 1:
        raise ValueError("`--proportion_empty_prompts` must be in the range [0, 1].")

    if args.validation_prompt is not None and args.validation_image is None:
        raise ValueError("`--validation_image` must be set if `--validation_prompt` is set")

    if args.validation_prompt is None and args.validation_image is not None:
        raise ValueError("`--validation_prompt` must be set if `--validation_image` is set")

    if args.train_json_dir is None:
        raise ValueError("`--train_json_dir` must be set to the path of the training json file.")

    if (
        args.validation_image is not None
        and args.validation_prompt is not None
        and len(args.validation_image) != 1
        and len(args.validation_prompt) != 1
        and len(args.validation_image) != len(args.validation_prompt)
    ):
        raise ValueError(
            "Must provide either 1 `--validation_image`, 1 `--validation_prompt`,"
            " or the same number of `--validation_prompt`s and `--validation_image`s"
        )

    if args.resolution % 8 != 0:
        raise ValueError(
            "`--resolution` must be divisible by 8 for consistently sized encoded images between the VAE and the brushnet encoder."
        )

    if args.metrics.enable and args.metrics.metric not in ["clip_score", "hps", "pick_score", "aesthetic_score", "image_reward", "vqa_score", "unified_reward", "pe_score", "ensemble", "random", "hpsv3"]:
            raise ValueError(f"Unsupported metric: {args.metrics.metric}, should be one of [clip_score, hps, pick_score, aesthetic_score, image_reward, vqa_score , unified_reward, pe_score, ensemble, random].")



full_args = OmegaConf.load("/root/test_env/configs/configs_brushnet.yaml")
args = full_args.train
check_args(args)
args.output_dir = os.path.join(args.output_dir, args.metrics.metric) if args.metrics.enable else args.output_dir
if not os.path.exists(os.path.join(args.output_dir, args.logging_dir)):
    os.makedirs(os.path.join(args.output_dir, args.logging_dir), exist_ok=True)
with open(os.path.join(args.output_dir, args.logging_dir, "config.yaml"), "w") as f:
    OmegaConf.save(args, f)

if not os.path.exists(args.output_dir):
    os.makedirs(args.output_dir, exist_ok=True)

logging_dir = os.path.join(args.output_dir, args.logging_dir)
if not os.path.exists(logging_dir):
    os.makedirs(logging_dir, exist_ok=True)
accelerator_project_config = ProjectConfiguration(project_dir=args.output_dir, logging_dir=logging_dir)
accelerator = Accelerator(
    gradient_accumulation_steps=args.gradient_accumulation_steps,
    mixed_precision=args.mixed_precision,
    log_with=args.report_to,
    project_config=accelerator_project_config,
)
logger = get_logger(__name__)

selected_files = None
# flag = False

def image_grid(imgs, rows, cols):
    assert len(imgs) == rows * cols

    w, h = imgs[0].size
    grid = Image.new("RGB", size=(cols * w, rows * h))

    for i, img in enumerate(imgs):
        grid.paste(img, box=(i % cols * w, i // cols * h))
    return grid

def log_training_samples() -> None:
    pass

def log_validation(
    vae, text_encoder, tokenizer, unet, brushnet, args, accelerator, weight_dtype, step, is_final_validation=False
):
    logger.info(f"[{accelerator.process_index}] Running validation... ")

    if not is_final_validation:
        brushnet = accelerator.unwrap_model(brushnet)
    else:
        brushnet = BrushNetModel.from_pretrained(args.output_dir, torch_dtype=weight_dtype)

    pipeline = StableDiffusionBrushNetPipeline.from_pretrained(
        args.pretrained_model_name_or_path,
        vae=vae,
        text_encoder=text_encoder,
        tokenizer=tokenizer,
        unet=unet,
        brushnet=brushnet,
        safety_checker=None,
        revision=args.revision,
        variant=args.variant,
        torch_dtype=weight_dtype,
    )
    pipeline.scheduler = UniPCMultistepScheduler.from_config(pipeline.scheduler.config)
    pipeline = pipeline.to(accelerator.device)
    pipeline.set_progress_bar_config(disable=True)

    if args.enable_xformers_memory_efficient_attention:
        pipeline.enable_xformers_memory_efficient_attention()

    if args.seed is None:
        generator = None
    else:
        generator = torch.Generator(device=accelerator.device).manual_seed(args.seed)

    if len(args.validation_image) == len(args.validation_prompt) and len(args.validation_image) == len(args.validation_mask):
        validation_images = args.validation_image
        validation_prompts = args.validation_prompt
        validation_masks = args.validation_mask
    else:
        raise ValueError(
            "number of `args.validation_image`, `args.validation_mask`, and `args.validation_prompt` should be checked in `parse_args`"
        )

    image_logs = []
    inference_ctx = contextlib.nullcontext() if is_final_validation else torch.autocast("cuda")

    for validation_prompt, validation_image, validation_mask in zip(validation_prompts, validation_images, validation_masks):
        # validation_image = Image.open(validation_image).convert("RGB")
        # validation_mask = Image.open(validation_mask).convert("RGB")
        # validation_image = Image.composite(validation_image, Image.new('RGB', (validation_image.size[0], validation_image.size[1]), (0, 0, 0)), validation_mask.convert("L"))

        init_image = cv2.imread(validation_image)[:,:,::-1]
        mask_image = 1.*(cv2.imread(validation_mask).sum(-1)>255)[:,:,np.newaxis]
        log_mask_image = 1 - mask_image
        init_image = init_image * (1-mask_image)
        init_image = Image.fromarray(init_image.astype(np.uint8)).convert("RGB")
        mask_image = Image.fromarray(mask_image.astype(np.uint8).repeat(3,-1)*255).convert("RGB")

        images = []

        for _ in range(args.num_validation_images):
            with inference_ctx:
                image = pipeline(
                    validation_prompt, init_image, mask_image, num_inference_steps=20, generator=generator
                ).images[0]

            images.append(image)

        image_logs.append(
            {"validation_image": init_image, "mask": log_mask_image, "images": images, "validation_prompt": validation_prompt}
        )

    tracker_key = "test" if is_final_validation else "validation"
    for tracker in accelerator.trackers:
        if tracker.name == "tensorboard":
            for log in image_logs:
                images = log["images"]
                validation_prompt = log["validation_prompt"]
                validation_image = log["validation_image"]

                formatted_images = []

                formatted_images.append(np.asarray(validation_image))

                for image in images:
                    formatted_images.append(np.asarray(image))

                formatted_images = np.stack(formatted_images)

                tracker.writer.add_images(validation_prompt, formatted_images, step, dataformats="NHWC")
        elif tracker.name == "wandb":
            formatted_images = []

            for log in image_logs:
                images = log["images"]
                validation_prompt = log["validation_prompt"]
                validation_image = log["validation_image"]
                masks = log["mask"]

                formatted_images.append(wandb.Image(validation_image, caption="BrushNet conditioning"))
                formatted_images.append(wandb.Image(masks, caption="Mask"))

                for image in images:
                    image = wandb.Image(image, caption=validation_prompt)
                    formatted_images.append(image)

            tracker.log({tracker_key: formatted_images})
        else:
            logger.warn(f"image logging not implemented for {tracker.name}")

        del pipeline
        gc.collect()
        torch.cuda.empty_cache()

        return image_logs


def import_model_class_from_model_name_or_path(pretrained_model_name_or_path: str, revision: str):
    text_encoder_config = PretrainedConfig.from_pretrained(
        pretrained_model_name_or_path,
        subfolder="text_encoder",
        revision=revision,
    )
    model_class = text_encoder_config.architectures[0]

    if model_class == "CLIPTextModel":
        from transformers import CLIPTextModel

        return CLIPTextModel
    elif model_class == "RobertaSeriesModelWithTransformation":
        from diffusers.pipelines.alt_diffusion.modeling_roberta_series import RobertaSeriesModelWithTransformation

        return RobertaSeriesModelWithTransformation
    else:
        raise ValueError(f"{model_class} is not supported.")


def save_model_card(repo_id: str, image_logs=None, base_model=str, repo_folder=None):
    img_str = ""
    if image_logs is not None:
        img_str = "You can find some example images below.\n\n"
        for i, log in enumerate(image_logs):
            images = log["images"]
            validation_prompt = log["validation_prompt"]
            validation_image = log["validation_image"]
            validation_image.save(os.path.join(repo_folder, "image_control.png"))
            img_str += f"prompt: {validation_prompt}\n"
            images = [validation_image] + images
            image_grid(images, 1, len(images)).save(os.path.join(repo_folder, f"images_{i}.png"))
            img_str += f"![images_{i})](./images_{i}.png)\n"

    model_description = f"""
# brushnet-{repo_id}

These are brushnet weights trained on {base_model} with new type of conditioning.
{img_str}
"""
    model_card = load_or_create_model_card(
        repo_id_or_path=repo_id,
        from_training=True,
        license="creativeml-openrail-m",
        base_model=base_model,
        model_description=model_description,
        inference=True,
    )

    tags = [
        "stable-diffusion",
        "stable-diffusion-diffusers",
        "text-to-image",
        "diffusers",
        "brushnet",
        "diffusers-training",
    ]
    model_card = populate_model_card(model_card, tags=tags)

    model_card.save(os.path.join(repo_folder, "README.md"))


class MyDataset():
    def __init__(self,resolution,tokenizer,random_mask, score_file=None):
        self.resolution = resolution
        self.tokenizer = tokenizer
        self.random_mask = random_mask
        if score_file is not None:
            with open(score_file, "r") as f:
                self.scores = json.load(f)
    
    def rle2mask(self, mask_rle, shape):# height width
        # Decode rle encoded mask.
        mask_rle=np.array(mask_rle)
        starts, lengths = [np.asarray(x, dtype=int) for x in (mask_rle[0:][::2], mask_rle[1:][::2])]
        starts -= 1
        ends = starts + lengths
        img = np.zeros(shape[0] * shape[1], dtype=np.uint8)
        for lo, hi in zip(starts, ends):
            img[lo:hi] = 1
        return img.reshape(shape, order='F')
    
    def tokenize_captions(self, caption, is_train=True):
        if random.random() < args.proportion_empty_prompts:
            caption=""
        elif isinstance(caption, str):
            caption=caption
        elif isinstance(caption, (list, np.ndarray)):
            # take a random caption if there are multiple
            caption=random.choice(caption) if is_train else caption[0]
        else:
            raise ValueError(
                f"Caption column `{caption_column}` should contain either strings or lists of strings."
            )
        inputs = self.tokenizer(
            caption, max_length=self.tokenizer.model_max_length, padding="max_length", truncation=True, return_tensors="pt"
        )
        return inputs.input_ids

    def __call__(self,examples):
        pixel_values=[]
        conditioning_pixel_values=[]
        masks=[]
        input_ids=[]
        conditioning_inpainting_pixel_values=[]
        inpainting_pixel_values=[]
        # image_ids=[]
        
        for example in examples:
            caption=example["caption"]
            image_id = example["image_id"]
            if args.metrics.enable:

                if args.metrics.metric == "random":
                    random_seed = random.sample(range(0, args.metrics.scaling), k=2)
                    win_seed = f"local_seed_{random_seed[0]}"
                    lose_seed = f"local_seed_{random_seed[1]}"

                elif not self.scores[image_id][args.metrics.metric]["draw"]:
                    if args.metrics.scaling == 16:
                        win_seed = f"local_{self.scores[image_id][args.metrics.metric]["max_seed"]}"
                        lose_seed = f"local_{self.scores[image_id][args.metrics.metric]["min_seed"]}"
                    else:
                        win_seed = f"local_{self.scores[image_id][args.metrics.metric][f"max_seed_{args.metrics.scaling}"]}"
                        lose_seed = f"local_{self.scores[image_id][args.metrics.metric][f"min_seed_{args.metrics.scaling}"]}"
                else: # random sample if draw
                    random_seed = random.sample(range(0, args.metrics.scaling), k=2)
                    win_seed = f"local_seed_{random_seed[0]}"
                    lose_seed = f"local_seed_{random_seed[1]}"

                win_image = cv2.imread(
                    example["image_path"].replace("local_seed_0", win_seed),
                    cv2.IMREAD_COLOR
                )
                lose_image = cv2.imread(
                    example["image_path"].replace("local_seed_0", lose_seed),
                    cv2.IMREAD_COLOR
                )

            else:
                win_image = cv2.imread(example["gt_image_path"], cv2.IMREAD_COLOR)
                lose_image = cv2.imread(example["image_path"], cv2.IMREAD_COLOR)

            inpainting_crop = example["crop"]
            height = example["height"]
            width = example["width"]
            
            mask = list(map(int, example["mask"].split(",")))
            mask = self.rle2mask(mask, (height, width))[:,:,np.newaxis]
            mask = 1 - mask

            w, h, _ = mask.shape
            if w>h:
                scale=self.resolution/h          
            else:
                scale=self.resolution/w
            w_new=int(np.ceil(w*scale))
            h_new=int(np.ceil(h*scale))

            if random.random()<0.3:
                kernel = np.ones((8,8),np.uint8)  
                mask_erosion = cv2.erode(mask,kernel,iterations = 1)
                mask_dilation = cv2.dilate(mask_erosion,kernel,iterations = 1)
                mask=1*(mask_dilation>0)[:,:,np.newaxis]
                mask=mask.astype(np.uint8)

            mask=cv2.resize(mask,(h_new,w_new),interpolation=cv2.INTER_CUBIC)[:,:,np.newaxis]
            random_crop=inpainting_crop
            mask=mask[random_crop[0]:random_crop[0]+self.resolution,random_crop[1]:random_crop[1]+self.resolution,:]

            if not args.metrics.enable:
                win_image = cv2.resize(win_image, (w_new, h_new), interpolation=cv2.INTER_CUBIC)
                win_image = win_image[random_crop[0]:random_crop[0]+self.resolution, random_crop[1]:random_crop[1]+self.resolution, :]

            image = win_image
            inpainting_image = lose_image
            masked_image = win_image * mask
            masked_inpainting_image = inpainting_image * mask
            
            image = cv2.cvtColor(image, cv2.COLOR_BGR2RGB)
            masked_image = cv2.cvtColor(masked_image, cv2.COLOR_BGR2RGB)
            inpainting_image = cv2.cvtColor(inpainting_image, cv2.COLOR_BGR2RGB)
            masked_inpainting_image = cv2.cvtColor(masked_inpainting_image, cv2.COLOR_BGR2RGB)
            image = (image.astype(np.float32) / 127.5) - 1.0
            masked_image = (masked_image.astype(np.float32) / 127.5) - 1.0
            inpainting_image = (inpainting_image.astype(np.float32) / 127.5) - 1.0
            masked_inpainting_image = (masked_inpainting_image.astype(np.float32) / 127.5) - 1.0

            mask=mask.astype(np.float32)

            pixel_values.append(torch.tensor(image).permute(2,0,1))
            inpainting_pixel_values.append(torch.tensor(inpainting_image).permute(2,0,1))
            conditioning_pixel_values.append(torch.tensor(masked_image).permute(2,0,1))
            conditioning_inpainting_pixel_values.append(torch.tensor(masked_inpainting_image).permute(2,0,1))
            masks.append(torch.tensor(mask).permute(2,0,1))
            input_ids.append(self.tokenize_captions(caption)[0])
            # image_ids.append(image_id)

        pixel_values = torch.stack(pixel_values)
        pixel_values = pixel_values.to(memory_format=torch.contiguous_format).float()
        inpainting_pixel_values = torch.stack(inpainting_pixel_values)
        inpainting_pixel_values = inpainting_pixel_values.to(memory_format=torch.contiguous_format).float()
        conditioning_pixel_values = torch.stack(conditioning_pixel_values)
        conditioning_pixel_values = conditioning_pixel_values.to(memory_format=torch.contiguous_format).float()
        conditioning_inpainting_pixel_values = torch.stack(conditioning_inpainting_pixel_values)
        conditioning_inpainting_pixel_values = conditioning_inpainting_pixel_values.to(memory_format=torch.contiguous_format).float()
        masks = torch.stack(masks)
        masks = masks.to(memory_format=torch.contiguous_format).float()
        input_ids = torch.stack(input_ids)
        # image_ids = torch.stack(image_ids)

        # if not self.printed:
        #     self.printed = True

        return {
            "pixel_values": pixel_values,
            "inpainting_pixel_values": inpainting_pixel_values,
            "conditioning_pixel_values": conditioning_pixel_values,
            "conditioning_inpainting_pixel_values": conditioning_inpainting_pixel_values,
            "masks":masks,
            "input_ids": input_ids,
            # "image_ids": image_ids,
        }
    

def main(args):
    global flag
    if args.report_to == "wandb" and args.hub_token is not None:
        raise ValueError(
            "You cannot use both --report_to=wandb and --hub_token due to a security risk of exposing your token."
            " Please use `huggingface-cli login` to authenticate with the Hub."
        )

    # Make one log on every process with the configuration for debugging.
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(lineno)d - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        level=logging.INFO,
        filename=os.path.join(logging_dir, "log.txt")
        
    )
    logger.info(accelerator.state, main_process_only=False)

    if accelerator.is_local_main_process:
        transformers.utils.logging.set_verbosity_warning()
        diffusers.utils.logging.set_verbosity_info()
    else:
        transformers.utils.logging.set_verbosity_error()
        diffusers.utils.logging.set_verbosity_error()

    # If passed along, set the training seed now.
    if args.seed is not None:
        logger.info(f"[{accelerator.process_index}] Using seed: {args.seed}")
        set_seed(args.seed)

    # Handle the repository creation
    if accelerator.is_main_process:
        if args.output_dir is not None:
            os.makedirs(args.output_dir, exist_ok=True)

        if args.push_to_hub:
            repo_id = create_repo(
                repo_id=args.hub_model_id or Path(args.output_dir).name, exist_ok=True, token=args.hub_token
            ).repo_id

    # Load the tokenizer
    if args.tokenizer_name:
        tokenizer = AutoTokenizer.from_pretrained(args.tokenizer_name, revision=args.revision, use_fast=False)
    elif args.pretrained_model_name_or_path:
        tokenizer = AutoTokenizer.from_pretrained(
            args.pretrained_model_name_or_path,
            subfolder="tokenizer",
            revision=args.revision,
            use_fast=False,
        )

    # import correct text encoder class
    text_encoder_cls = import_model_class_from_model_name_or_path(args.pretrained_model_name_or_path, args.revision)

    # Load scheduler and models
    noise_scheduler = DDPMScheduler.from_pretrained(args.pretrained_model_name_or_path, subfolder="scheduler")
    text_encoder = text_encoder_cls.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="text_encoder", revision=args.revision, variant=args.variant
    )
    vae = AutoencoderKL.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="vae", revision=args.revision, variant=args.variant
    )
    unet = UNet2DConditionModel.from_pretrained(
        args.pretrained_model_name_or_path, subfolder="unet", revision=args.revision, variant=args.variant
    )

    if args.brushnet_model_name_or_path and args.resume_from_checkpoint is None:
        logger.info(f"[{accelerator.process_index}] Loading existing brushnet weights from {args.brushnet_model_name_or_path}")
        brushnet = BrushNetModel.from_pretrained(args.brushnet_model_name_or_path)
        ref_brushnet = BrushNetModel.from_pretrained(args.brushnet_model_name_or_path)
    else:
        logger.info(f"[{accelerator.process_index}] Initializing brushnet weights from unet")
        brushnet = BrushNetModel.from_unet(unet)
        ref_brushnet = BrushNetModel.from_unet(unet)

    # Taken from [Sayak Paul's Diffusers PR #6511](https://github.com/huggingface/diffusers/pull/6511/files)
    def unwrap_model(model):
        model = accelerator.unwrap_model(model)
        model = model._orig_mod if is_compiled_module(model) else model
        return model

    # `accelerate` 0.16.0 will have better support for customized saving
    if version.parse(accelerate.__version__) >= version.parse("0.16.0"):
        # create custom saving & loading hooks so that `accelerator.save_state(...)` serializes in a nice format
        def save_model_hook(models, weights, output_dir):
            if accelerator.is_main_process:
                i = len(weights) - 1

                while len(weights) > 0:
                    weights.pop()
                    model = models[i]

                    sub_dir = "brushnet"
                    model.save_pretrained(os.path.join(output_dir, sub_dir))

                    i -= 1

        def load_model_hook(models, input_dir):
            while len(models) > 0:
                # pop models so that they are not loaded again
                model = models.pop()

                # load diffusers style into model
                load_model = BrushNetModel.from_pretrained(input_dir, subfolder="brushnet")
                model.register_to_config(**load_model.config)

                model.load_state_dict(load_model.state_dict())
                del load_model

        accelerator.register_save_state_pre_hook(save_model_hook)
        accelerator.register_load_state_pre_hook(load_model_hook)

    vae.requires_grad_(False)
    unet.requires_grad_(False)
    text_encoder.requires_grad_(False)
    ref_brushnet.requires_grad_(False)
    brushnet.train()

    if args.enable_xformers_memory_efficient_attention:
        if is_xformers_available():
            import xformers

            xformers_version = version.parse(xformers.__version__)
            if xformers_version == version.parse("0.0.16"):
                logger.warn(
                    "xFormers 0.0.16 cannot be used for training in some GPUs. If you observe problems during training, please update xFormers to at least 0.0.17. See https://huggingface.co/docs/diffusers/main/en/optimization/xformers for more details."
                )
            unet.enable_xformers_memory_efficient_attention()
            brushnet.enable_xformers_memory_efficient_attention()
            ref_brushnet.enable_xformers_memory_efficient_attention()
            logger.info(f"[{accelerator.process_index}] xformers memory efficient attention enabled")
        else:
            raise ValueError("xformers is not available. Make sure it is installed correctly")

    if args.gradient_checkpointing:
        brushnet.enable_gradient_checkpointing()

    # Check that all trainable models are in full precision
    low_precision_error_string = (
        " Please make sure to always have all model weights in full float32 precision when starting training - even if"
        " doing mixed precision training, copy of the weights should still be float32."
    )

    if unwrap_model(brushnet).dtype != torch.float32:
        raise ValueError(
            f"BrushNet loaded as datatype {unwrap_model(brushnet).dtype}. {low_precision_error_string}"
        )

    # Enable TF32 for faster training on Ampere GPUs,
    # cf https://pytorch.org/docs/stable/notes/cuda.html#tensorfloat-32-tf32-on-ampere-devices
    if args.allow_tf32:
        torch.backends.cuda.matmul.allow_tf32 = True

    if args.scale_lr:
        args.learning_rate = (
            args.learning_rate * args.gradient_accumulation_steps * args.train_batch_size * accelerator.num_processes
        )

    # Use 8-bit Adam for lower memory usage or to fine-tune the model in 16GB GPUs
    if args.use_8bit_adam:
        try:
            import bitsandbytes as bnb
        except ImportError:
            raise ImportError(
                "To use 8-bit Adam, please install the bitsandbytes library: `pip install bitsandbytes`."
            )

        optimizer_class = bnb.optim.AdamW8bit
    else:
        optimizer_class = torch.optim.AdamW

    # Optimizer creation
    params_to_optimize = brushnet.parameters()
    optimizer = optimizer_class(
        params_to_optimize,
        lr=args.learning_rate,
        betas=(args.adam_beta1, args.adam_beta2),
        weight_decay=args.adam_weight_decay,
        eps=args.adam_epsilon,
    )

    # train_dataset = make_train_dataset(args, tokenizer, accelerator)

    # train_dataloader = torch.utils.data.DataLoader(
    #     train_dataset,
    #     shuffle=True,
    #     collate_fn=collate_fn,
    #     batch_size=args.train_batch_size,
    #     num_workers=args.dataloader_num_workers,
    # )

    train_dataset = load_dataset("json", 
                        data_files={"train": args.train_json_dir}, 
                        split="train"
                    )
    
    # train_dataset = train_dataset.shard(num_shards=accelerator.num_processes, index=accelerator.process_index, contiguous=True) # IMPORTANT

    train_dataset_len= len(train_dataset)

    train_dataloader = torch.utils.data.DataLoader(
        train_dataset,
        shuffle=True,
        collate_fn=MyDataset(resolution = args.resolution, 
                             tokenizer=tokenizer,
                             random_mask=args.random_mask,
                             score_file=args.metrics.score_file if args.metrics.enable else None),
        batch_size=args.train_batch_size,
        num_workers=args.dataloader_num_workers,
    )
    train_dataloader_len=train_dataset_len//args.train_batch_size

    # Scheduler and math around the number of training steps.
    overrode_max_train_steps = False
    num_update_steps_per_epoch = math.ceil(train_dataloader_len / args.gradient_accumulation_steps)
    if args.max_train_steps is None:
        args.max_train_steps = args.num_train_epochs * num_update_steps_per_epoch
        overrode_max_train_steps = True

    lr_scheduler = get_scheduler(
        args.lr_scheduler,
        optimizer=optimizer,
        num_warmup_steps=args.lr_warmup_steps * accelerator.num_processes,
        num_training_steps=args.max_train_steps * accelerator.num_processes,
        num_cycles=args.lr_num_cycles,
        power=args.lr_power,
    )

    # Prepare everything with our `accelerator`.
    brushnet, optimizer, train_dataloader, lr_scheduler = accelerator.prepare(
        brushnet, optimizer, train_dataloader, lr_scheduler
    )

    # For mixed precision training we cast the text_encoder and vae weights to half-precision
    # as these models are only used for inference, keeping weights in full precision is not required.
    weight_dtype = torch.float32
    if accelerator.mixed_precision == "fp16":
        weight_dtype = torch.float16
    elif accelerator.mixed_precision == "bf16":
        weight_dtype = torch.bfloat16

    # Move vae, unet and text_encoder to device and cast to weight_dtype
    vae.to(accelerator.device, dtype=weight_dtype)
    unet.to(accelerator.device, dtype=weight_dtype)
    text_encoder.to(accelerator.device, dtype=weight_dtype)
    ref_brushnet.to(accelerator.device, dtype=weight_dtype)

    # We need to recalculate our total training steps as the size of the training dataloader may have changed.
    num_update_steps_per_epoch = math.ceil(train_dataloader_len / args.gradient_accumulation_steps)
    if overrode_max_train_steps:
        args.max_train_steps = args.num_train_epochs * num_update_steps_per_epoch
    # Afterwards we recalculate our number of training epochs
    args.num_train_epochs = math.ceil(args.max_train_steps / num_update_steps_per_epoch)

    # We need to initialize the trackers we use, and also store our configuration.
    # The trackers initializes automatically on the main process.
    if accelerator.is_main_process:
        tracker_config = dict(vars(args))

        # tensorboard cannot handle list types for config
        # tracker_config.pop("validation_prompt")
        # tracker_config.pop("validation_image")
        # tracker_config.pop("validation_mask")
        # tracker_config.pop("random_mask")

        # if args.wandb_id is not None:
        #     init_kwargs = {
        #         "wandb": {
        #             "id": args.wandb_id,
        #             "resume": "must",  
        #             "tags": ["resumed_run"],
        #         }
        #     }
        #     accelerator.init_trackers(args.tracker_project_name, config=tracker_config, init_kwargs=init_kwargs)
        # else:
            # accelerator.init_trackers(args.tracker_project_name, config=tracker_config)
        accelerator.init_trackers(args.tracker_project_name, config=tracker_config)

    # Train!
    total_batch_size = args.train_batch_size * accelerator.num_processes * args.gradient_accumulation_steps

    logger.info("***** Running training *****")
    logger.info(f"[{accelerator.process_index}]   Num examples = {train_dataset_len}")
    logger.info(f"[{accelerator.process_index}]   Num batches each epoch = {train_dataloader_len}")
    logger.info(f"[{accelerator.process_index}]   Num Epochs = {args.num_train_epochs}")
    logger.info(f"[{accelerator.process_index}]   Instantaneous batch size per device = {args.train_batch_size}")
    logger.info(f"[{accelerator.process_index}]   Total train batch size (w. parallel, distributed & accumulation) = {total_batch_size}")
    logger.info(f"[{accelerator.process_index}]   Gradient Accumulation steps = {args.gradient_accumulation_steps}")
    logger.info(f"[{accelerator.process_index}]   Total optimization steps = {args.max_train_steps}")
    logger.info(f"[{accelerator.process_index}]   dtype = {weight_dtype}")
    logger.info(f"[{accelerator.process_index}]   dpo loss weight = {args.dpo_loss_weight}")
    logger.info(f"[{accelerator.process_index}]   beta dpo = {args.beta_dpo}")
    logger.info(f"[{accelerator.process_index}]   dpo mask = {args.dpo_mask}")
    logger.info(f"[{accelerator.process_index}]   dpo new = {args.dpo_new}")
    logger.info(f"[{accelerator.process_index}]   mse loss weight = {args.mse_loss_weight}")
    logger.info(f"[{accelerator.process_index}]   learning rate = {args.learning_rate}")
    logger.info(f"[{accelerator.process_index}]   loss = {args.dpo_loss_weight} * dpo_loss + {args.mse_loss_weight} * mse_loss")
    logger.info(f"[{accelerator.process_index}]   Metric = {args.metrics.metric}" if args.metrics.enable else f"[{accelerator.process_index}] metrics are disabled")
    logger.info(f"[{accelerator.process_index}]   Scaling = {args.metrics.scaling}" if args.metrics.enable else f"[{accelerator.process_index}] metrics scaling is not used")
    logger.info(f"[{accelerator.process_index}]   Image annotation file = {args.train_json_dir}")
    logger.info(f"[{accelerator.process_index}]   Score file = {args.metrics.score_file}" if args.metrics.enable else f"[{accelerator.process_index}] metrics score file is not used")
    logger.info(f"[{accelerator.process_index}]   Note: {args.note}")
    global_step = 0
    first_epoch = 0

    # Potentially load in the weights and states from a previous save
    if args.resume_from_checkpoint:
        if args.resume_from_checkpoint != "latest":
            path = os.path.basename(args.resume_from_checkpoint)
        else:
            # Get the most recent checkpoint
            dirs = os.listdir(args.output_dir)
            dirs = [d for d in dirs if d.startswith("checkpoint")]
            dirs = sorted(dirs, key=lambda x: int(x.split("-")[1]))
            path = dirs[-1] if len(dirs) > 0 else None

        if path is None:
            logger.info(f"[{accelerator.process_index}] Checkpoint '{args.resume_from_checkpoint}' does not exist. Starting a new training run.")
            raise ValueError(f"Checkpoint '{args.resume_from_checkpoint}' does not exist.")
            args.resume_from_checkpoint = None
            initial_global_step = 0

        else:
            logger.info(f"[{accelerator.process_index}] Resuming from checkpoint {path}")
            accelerator.load_state(os.path.join(args.output_dir, path),map_location="cpu")
            # args.output_dir = os.path.join(args.output_dir, f"resumed_from_{args.resume_from_checkpoint}_{datetime.datetime.now().strftime('%Y%m%d_%H%M')}")
            # flag = True
            global_step = int(path.split("-")[1])

            initial_global_step = global_step
            # initial_global_step = 5
            resume_step = 0
            first_epoch = global_step // num_update_steps_per_epoch
            first_epoch = 0
    else:
        initial_global_step = 0


    progress_bar = tqdm(
        range(0, args.max_train_steps),
        initial=initial_global_step,
        desc="Steps",
        # Only show the progress bar once on each machine.
        disable=not accelerator.is_local_main_process,
    )

    check_only_once = True

    image_logs = None
    for epoch in range(first_epoch, args.num_train_epochs):
        for step, batch in enumerate(train_dataloader):
            accelerator.wait_for_everyone()
            
            with accelerator.accumulate(brushnet):
                with accelerator.autocast():
                    # Convert images to latent space
                    pixel_values, inpainting_pixel_values = batch["pixel_values"], batch["inpainting_pixel_values"]
                    if not args.metrics.enable:
                        inpainting_pixel_values = (1 - batch["masks"]) * inpainting_pixel_values + batch["masks"] * pixel_values

                    if accelerator.is_main_process:
                        if step == 0:
                            from torchvision.transforms.functional import to_pil_image
                            if pixel_values.dtype == torch.bfloat16:
                                image_1_win = pixel_values.to(dtype=torch.float32).to("cpu")
                                image_1_loss = inpainting_pixel_values.to(dtype=torch.float32).to("cpu")
                                image_2 = masks.to(dtype=torch.float32).to("cpu")
                            else:
                                image_1_win = pixel_values.to("cpu")
                                image_1_loss = inpainting_pixel_values.to("cpu")
                                image_2 = batch["masks"].to("cpu")
                            to_pil_image((image_1_win[0] + 1) / 2).save(f"{os.path.join(args.output_dir, 'logs', 'training_image_checks_win.png')}")
                            to_pil_image((image_1_loss[0] + 1) / 2).save(f"{os.path.join(args.output_dir, 'logs', 'training_image_checks_loss.png')}")
                            if image_2.ndim == 4:
                                to_pil_image(image_2[0].squeeze()).save(f"{os.path.join(args.output_dir, 'logs', 'training_image_checks_mask.png')}")
                            else:
                                to_pil_image(image_2[0]).save(f"{os.path.join(args.output_dir, 'training_image_checks_mask.png')}")

                    # feed_pixel_values = torch.cat([pixel_values, inpainting_pixel_values])
                    if args.dpo_new:
                        feed_pixel_values = torch.cat([inpainting_pixel_values, pixel_values])
                    else:
                        feed_pixel_values = torch.cat([pixel_values, inpainting_pixel_values])

                    if check_only_once:
                        logger.info(f"[{accelerator.process_index}] feed_pixel_values shape: {feed_pixel_values.shape}")
                        logger.info(f"[{accelerator.process_index}] mask shape: {batch['masks'].shape}")



                    latents = vae.encode(feed_pixel_values.to(dtype=weight_dtype)).latent_dist.sample()
                    latents = latents * vae.config.scaling_factor

                    if check_only_once:
                        logger.info(f"[{accelerator.process_index}] latents shape: {latents.shape}")

                    # feed_conditioning_pixel_values = torch.cat([batch["conditioning_pixel_values"],batch["conditioning_inpainting_pixel_values"]])
                    feed_win_conditioning_pixel_values, feed_loss_conditioning_pixel_values = batch["conditioning_pixel_values"], batch["conditioning_inpainting_pixel_values"]
                    conditioning_win_latents=vae.encode(feed_win_conditioning_pixel_values.to(dtype=weight_dtype)).latent_dist.sample()
                    conditioning_loss_latents=vae.encode(feed_loss_conditioning_pixel_values.to(dtype=weight_dtype)).latent_dist.sample()
                    conditioning_win_latents = conditioning_win_latents * vae.config.scaling_factor
                    conditioning_loss_latents = conditioning_loss_latents * vae.config.scaling_factor

                    masks = torch.nn.functional.interpolate(
                        batch["masks"], 
                        size=(
                            latents.shape[-2], 
                            latents.shape[-1]
                        )
                    )

                    conditioning_win_latents=torch.concat([conditioning_win_latents,masks],1)
                    conditioning_loss_latents=torch.concat([conditioning_loss_latents,masks],1)

                    if check_only_once:
                        logger.info(f"[{accelerator.process_index}] conditioning_win_latents shape: {conditioning_win_latents.shape}")
                        logger.info(f"[{accelerator.process_index}] conditioning_loss_latents shape: {conditioning_loss_latents.shape}")
                        logger.info(f"[{accelerator.process_index}] masks shape: {masks.shape}")

                    if args.dpo_new:
                        conditioning_latents = torch.cat([conditioning_loss_latents, conditioning_win_latents])
                    else:
                        conditioning_latents = torch.cat([conditioning_win_latents, conditioning_loss_latents])

                    if check_only_once:
                        logger.info(f"[{accelerator.process_index}] conditioning_latents shape: {conditioning_latents.shape}")
                    
                    check_only_once = False

                    # Sample noise that we'll add to the latents
                    noise = torch.randn_like(latents)
                    bsz = latents.shape[0]
                    # Sample a random timestep for each image
                    timesteps = torch.randint(0, noise_scheduler.config.num_train_timesteps, (bsz,), device=latents.device)
                    timesteps = timesteps.long()
                    # dpo things
                    timesteps = timesteps.chunk(2)[0].repeat(2)
                    noise = noise.chunk(2)[0].repeat(2, 1, 1, 1)

                    # Add noise to the latents according to the noise magnitude at each timestep
                    # (this is the forward diffusion process)
                    noisy_latents = noise_scheduler.add_noise(latents, noise, timesteps)

                    # Get the text embedding for conditioning
                    encoder_hidden_states = text_encoder(batch["input_ids"], return_dict=False)[0]
                    encoder_hidden_states = encoder_hidden_states.repeat(2, 1, 1)

                    down_block_res_samples, mid_block_res_sample, up_block_res_samples = brushnet(
                        noisy_latents,
                        timesteps,
                        encoder_hidden_states=encoder_hidden_states,
                        brushnet_cond=conditioning_latents,
                        return_dict=False,
                    )

                    # Predict the noise residual
                    model_pred = unet(
                        noisy_latents,
                        timesteps,
                        encoder_hidden_states=encoder_hidden_states,
                        down_block_add_samples=[
                            sample.to(dtype=weight_dtype) for sample in down_block_res_samples
                        ],
                        mid_block_add_sample=mid_block_res_sample.to(dtype=weight_dtype),
                        up_block_add_samples=[
                            sample.to(dtype=weight_dtype) for sample in up_block_res_samples
                        ],
                        return_dict=False,
                    )[0]

                    # Get the target for loss depending on the prediction type
                    if noise_scheduler.config.prediction_type == "epsilon":
                        target = noise
                    elif noise_scheduler.config.prediction_type == "v_prediction":
                        target = noise_scheduler.get_velocity(latents, noise, timesteps)
                    else:
                        raise ValueError(f"Unknown prediction type {noise_scheduler.config.prediction_type}")
                    
                    
                    loss_diff = model_pred - target
                    if args.dpo_mask:
                        loss_masks = 1 - masks
                        loss_masks = loss_masks.repeat(2,1,1,1)
                        loss_diff = loss_diff * loss_masks
                    model_losses = loss_diff.pow(2).mean(dim=[1,2,3])

                    # model_losses = (loss_masks.repeat(2,1,1,1 ) * (model_pred - target).pow(2)).mean(dim=[1,2,3])
                    model_losses_w, model_losses_l = model_losses.chunk(2)
                    raw_model_loss = 0.5 * (model_losses_w.mean() + model_losses_l.mean())
                    model_diff = model_losses_w - model_losses_l # These are both LBS (as is t)

                    with torch.no_grad(): # Get the reference policy (unet) prediction
                        ref_down_block_res_samples, ref_mid_block_res_sample, ref_up_block_res_samples = ref_brushnet(
                            noisy_latents,
                            timesteps,
                            encoder_hidden_states=encoder_hidden_states,
                            brushnet_cond=conditioning_latents,
                            return_dict=False,
                        )

                        ref_pred = unet(
                            noisy_latents,
                            timesteps,
                            encoder_hidden_states=encoder_hidden_states,
                            down_block_add_samples=[
                                sample.to(dtype=weight_dtype) for sample in ref_down_block_res_samples
                            ],
                            mid_block_add_sample=ref_mid_block_res_sample.to(dtype=weight_dtype),
                            up_block_add_samples=[
                                sample.to(dtype=weight_dtype) for sample in ref_up_block_res_samples
                            ],
                            return_dict=False,
                        )[0]

                        # ref_losses = (ref_pred - target).pow(2).mean(dim=[1,2,3])
                        ref_loss_diff = ref_pred - target
                        if args.dpo_mask:
                            ref_loss_diff = ref_loss_diff * loss_masks
                        ref_losses = ref_loss_diff.pow(2).mean(dim=[1,2,3])
                        ref_losses_w, ref_losses_l = ref_losses.chunk(2)
                        ref_diff = ref_losses_w - ref_losses_l
                        raw_ref_loss = ref_losses.mean()  

                    scale_term = -0.5 * args.beta_dpo
                    inside_term = scale_term * (model_diff - ref_diff)
                    # implicit_acc = (inside_term > 0).sum().float() / inside_term.size(0)

                    dpo_loss = -1 * F.logsigmoid(inside_term).mean()
                    # masked_model_pred = model_pred.chunk(2)[0] # calculate mse only on the winning part
                    # masked_target = target.chunk(2)[0] 
                    # mse_loss = F.mse_loss(masked_model_pred.float(), masked_target.float(), reduction="mean")
                    
                    loss = dpo_loss 
                    

                avg_loss = accelerator.gather(loss).mean().item()

                avg_dpo_loss = accelerator.gather(dpo_loss).mean().item()
                # avg_mse_loss = accelerator.gather(mse_loss).mean().item()
                avg_raw_model_mse = accelerator.gather(raw_model_loss).mean().item()
                avg_ref_mse = accelerator.gather(raw_ref_loss).mean().item()
                # avg_acc = accelerator.gather(implicit_acc).mean().item()
                # implicit_acc_accumulated += avg_acc / args.gradient_accumulation_steps

                accelerator.backward(loss)
                if accelerator.sync_gradients:
                    params_to_clip = brushnet.parameters()
                    accelerator.clip_grad_norm_(params_to_clip, args.max_grad_norm)
                optimizer.step()
                lr_scheduler.step()
                optimizer.zero_grad(set_to_none=args.set_grads_to_none)

            # Checks if the accelerator has just performed an optimization step, if so do "end of batch" logging
            logger.info(f"[{accelerator.process_index}] global_step: {global_step} / {args.max_train_steps}, step_loss: {avg_loss}, lr: {lr_scheduler.get_last_lr()[0]}")
            if accelerator.sync_gradients:
                progress_bar.update(1)
                global_step += 1
                # logs = {"step_loss": loss.detach().item(), "lr": lr_scheduler.get_last_lr()[0], "implicit_acc": avg_acc}
                logs = {"step_loss": avg_loss, "lr": lr_scheduler.get_last_lr()[0]}
                accelerator.log({"train_loss": avg_loss}, step=global_step)
                accelerator.log(logs, step=global_step)
                accelerator.log({"model_mse_unaccumulated": avg_raw_model_mse}, step=global_step)
                accelerator.log({"ref_mse_unaccumulated": avg_ref_mse}, step=global_step)
                # accelerator.log({"implicit_acc_accumulated": implicit_acc_accumulated}, step=global_step)
                accelerator.log({"dpo_loss_unaccumulated": avg_dpo_loss}, step=global_step)
                # accelerator.log({"mse_loss_unaccumulated": avg_mse_loss}, step=global_step)
                

                if accelerator.is_main_process:
                    if global_step % args.checkpointing_steps == 0:
                        # _before_ saving state, check if this save would set us over the `checkpoints_total_limit`
                        if args.checkpoints_total_limit is not None:
                            checkpoints = os.listdir(args.output_dir)
                            checkpoints = [d for d in checkpoints if d.startswith("checkpoint")]
                            checkpoints = sorted(checkpoints, key=lambda x: int(x.split("-")[1]))

                            # before we save the new checkpoint, we need to have at _most_ `checkpoints_total_limit - 1` checkpoints
                            if len(checkpoints) >= args.checkpoints_total_limit:
                                num_to_remove = len(checkpoints) - args.checkpoints_total_limit + 1
                                removing_checkpoints = checkpoints[0:num_to_remove]

                                logger.info(
                                    f"[{accelerator.process_index}] {len(checkpoints)} checkpoints already exist, removing {len(removing_checkpoints)} checkpoints"
                                )
                                logger.info(f"[{accelerator.process_index}] removing checkpoints: {', '.join(removing_checkpoints)}")

                                for removing_checkpoint in removing_checkpoints:
                                    removing_checkpoint = os.path.join(args.output_dir, removing_checkpoint)
                                    shutil.rmtree(removing_checkpoint)

                        save_path = os.path.join(args.output_dir, f"checkpoint-{global_step}")
                        accelerator.save_state(save_path)
                        logger.info(f"[{accelerator.process_index}] Saved state to {save_path}")

                    if args.validation_prompt is not None and (global_step  % args.validation_steps == 0 or global_step == 1):
                        image_logs = log_validation(
                            vae,
                            text_encoder,
                            tokenizer,
                            unet,
                            brushnet,
                            args,
                            accelerator,
                            weight_dtype,
                            global_step,
                        )
                

            progress_bar.set_postfix(**logs)

            if global_step >= args.train_steps:
                accelerator.wait_for_everyone()
                accelerator.end_training()


    # Create the pipeline using using the trained modules and save it.
    if accelerator.is_main_process:
        brushnet = unwrap_model(brushnet)
        brushnet.save_pretrained(args.output_dir)

        # Run a final round of validation.
        image_logs = None
        if args.validation_prompt is not None:
            image_logs = log_validation(
                vae=vae,
                text_encoder=text_encoder,
                tokenizer=tokenizer,
                unet=unet,
                brushnet=None,
                args=args,
                accelerator=accelerator,
                weight_dtype=weight_dtype,
                step=global_step,
                is_final_validation=True,
            )

        if args.push_to_hub:
            save_model_card(
                repo_id,
                image_logs=image_logs,
                base_model=args.pretrained_model_name_or_path,
                repo_folder=args.output_dir,
            )
            upload_folder(
                repo_id=repo_id,
                folder_path=args.output_dir,
                commit_message="End of training",
                ignore_patterns=["step_*", "epoch_*"],
            )

    accelerator.wait_for_everyone()
    accelerator.end_training()


if __name__ == "__main__":
    selected_files = list(os.path.join(args.train_data_dir, f"000{i:02}.tar") for i in range(0,88))
    main(args)
