import logging
import os
import sys

from .message_define import MyMessage
from .utils import transform_tensor_to_list, post_complete_message_to_sweep_process

sys.path.insert(0, os.path.abspath(os.path.join(os.getcwd(), "../../../")))
sys.path.insert(0, os.path.abspath(os.path.join(os.getcwd(), "../../../../FedML")))
try:
    from fedml_core.distributed.communication.message import Message
    from fedml_core.distributed.server.server_manager import ServerManager
except ImportError:
    from FedML.fedml_core.distributed.communication.message import Message
    from FedML.fedml_core.distributed.server.server_manager import ServerManager

import wandb



class FedAVGServerManager(ServerManager):
    def __init__(
        self,
        args,
        aggregator,
        comm=None,
        rank=0,
        size=0,
        backend="MPI",
        is_preprocessed=False,
        preprocessed_client_lists=None,
        time_stamp_num=-1,
        task_start_indices = None,
    ):
        super().__init__(args, comm, rank, size, backend)
        self.args = args
        self.aggregator = aggregator
        self.round_num = args.comm_round
        self.round_idx = 0
        self.is_preprocessed = is_preprocessed
        self.preprocessed_client_lists = preprocessed_client_lists
        self.time_stamp = 0
        self.task_start_indices=task_start_indices

    def run(self):
        super().run()

    def send_init_msg(self):
        # sampling clients
        client_indexes = self.aggregator.client_sampling(
            self.round_idx,
            self.time_stamp,
            self.args.client_num_per_round,
        )

        client_schedule = self.aggregator.client_schedule(
            self.round_idx, client_indexes
        )
        average_weight_dict = self.aggregator.get_average_weight(client_indexes)

        global_model_params = self.aggregator.get_global_model_params()

        if self.args.is_mobile == 1:
            global_model_params = transform_tensor_to_list(global_model_params)
        for process_id in range(1, self.size):
            self.send_message_init_config(
                process_id, global_model_params, average_weight_dict, client_schedule
            )

    def register_message_receive_handlers(self):
        self.register_message_receive_handler(
            MyMessage.MSG_TYPE_C2S_SEND_MODEL_TO_SERVER,
            self.handle_message_receive_model_from_client,
        )

    def handle_message_receive_model_from_client(self, msg_params):
        sender_id = msg_params.get(MyMessage.MSG_ARG_KEY_SENDER)
        model_params = msg_params.get(MyMessage.MSG_ARG_KEY_MODEL_PARAMS)
        client_runtime_info = msg_params.get(MyMessage.MSG_ARG_KEY_CLIENT_RUNTIME_INFO)
        self.aggregator.record_client_runtime(sender_id - 1, client_runtime_info)

        self.aggregator.add_local_trained_result(sender_id - 1, model_params)

        b_all_received = self.aggregator.check_whether_all_receive()
        logging.info("b_all_received = " + str(b_all_received))
        if b_all_received:
            global_model_params = self.aggregator.aggregate(self.round_idx)
            self.aggregator.test_on_server_for_all_clients(self.round_idx, self.time_stamp)

            # start the next round 
            self.round_idx += 1
            if self.round_idx in self.task_start_indices:
                self.time_stamp += 1
                if self.round_idx == (self.round_num+1):
                    self.finish()
                    print("here")
                    return
            wandb.log({"Timestamps": self.time_stamp, "round": self.round_idx})

            if self.is_preprocessed:
                if self.preprocessed_client_lists is None:
                    # sampling has already been done in data preprocessor
                    client_indexes = [self.round_idx] * self.args.client_num_per_round
                else:
                    client_indexes = self.preprocessed_client_lists[self.round_idx]
            else:
                # sampling clients
                client_indexes = self.aggregator.client_sampling(
                    self.round_idx,
                    self.time_stamp,
                    self.args.client_num_per_round,
                )
            client_schedule = self.aggregator.client_schedule(
                self.round_idx, client_indexes
            )
            average_weight_dict = self.aggregator.get_average_weight(client_indexes)

            global_model_params = self.aggregator.get_global_model_params()
            if self.args.is_mobile == 1:
                global_model_params = transform_tensor_to_list(global_model_params)

            print("indexes of clients: " + str(client_indexes))
            print("size = %d" % self.size)
            if self.args.is_mobile == 1:
                global_model_params = transform_tensor_to_list(global_model_params)

            for receiver_id in range(1, self.size):
                self.send_message_sync_model_to_client(
                    receiver_id,
                    global_model_params,
                    average_weight_dict,
                    client_schedule,
                    self.aggregator.orth_set
                )

    def send_message_init_config(
        self, receive_id, global_model_params, average_weight_dict, client_schedule
    ):
        message = Message(
            MyMessage.MSG_TYPE_S2C_INIT_CONFIG, self.get_sender_id(), receive_id
        )
        message.add_params(MyMessage.MSG_ARG_KEY_MODEL_PARAMS, global_model_params)
        message.add_params(MyMessage.MSG_ARG_KEY_AVG_WEIGHTS, average_weight_dict)
        message.add_params(MyMessage.MSG_ARG_KEY_CLIENT_SCHEDULE, client_schedule)
        self.send_message(message)

    def send_message_sync_model_to_client(
        self, receive_id, global_model_params, average_weight_dict, client_schedule, orth_space
    ):
        logging.info("send_message_sync_model_to_client. receive_id = %d" % receive_id)
        message = Message(
            MyMessage.MSG_TYPE_S2C_SYNC_MODEL_TO_CLIENT,
            self.get_sender_id(),
            receive_id,
        )
        message.add_params(MyMessage.MSG_ARG_KEY_MODEL_PARAMS, global_model_params)
        message.add_params(MyMessage.MSG_ARG_KEY_AVG_WEIGHTS, average_weight_dict)
        message.add_params(MyMessage.MSG_ARG_KEY_CLIENT_SCHEDULE, client_schedule)
        message.add_params(MyMessage.MSG_ARG_KEY_ORTH_SPACE, orth_space)
        self.send_message(message)
