import os
import json
import re
import sys
import time
import traceback
import numpy as np
import pandas as pd
from dotenv import load_dotenv
import google.generativeai as genai

sys.path.append(os.path.dirname(os.path.abspath(__file__)))
from tools.emulators import (
    query_lat_and_lon, future_temperature, diy_greenhouse, location_summary,
    history_temperature, diy_aerosol, is_land_or_sea, diy_aerosol_mean,
    diff_diy_aerosol_mean, diy_greenhouse_summary
)

# Tool mapping for dynamic tool calls
TOOL_MAP = {
    "query_lat_and_lon": query_lat_and_lon, "future_temperature": future_temperature,
    "diy_greenhouse": diy_greenhouse, "location_summary": location_summary,
    "history_temperature": history_temperature, "diy_aerosol": diy_aerosol,
    "is_land_or_sea": is_land_or_sea, "diy_aerosol_mean": diy_aerosol_mean,
    "diff_diy_aerosol_mean": diff_diy_aerosol_mean, "diy_greenhouse_summary": diy_greenhouse_summary,
}

# --- City Data Loading & Query Logic (Mirrored from create_open.py) ---
CITY_DATA_DF = None

def load_city_data(file_path):
    """
    Loads city data from the specified CSV file into a global DataFrame.
    This ensures coordinate lookup is fast, reliable, and uses the same source as create_open.py.
    """
    global CITY_DATA_DF
    try:
        base_dir = os.path.dirname(os.path.abspath(__file__))
        full_path = os.path.join(base_dir, file_path)
        
        if not os.path.exists(full_path):
            print(f"FATAL: City data file not found at {full_path}")
            CITY_DATA_DF = pd.DataFrame()
            return

        CITY_DATA_DF = pd.read_csv(full_path)
        CITY_DATA_DF.columns = [x.lower() for x in CITY_DATA_DF.columns]
        
        if 'city' in CITY_DATA_DF.columns:
            CITY_DATA_DF.dropna(subset=['city'], inplace=True)
            print(f"Successfully loaded city data from {full_path}")
        else:
            print(f"FATAL: 'city' column not found in {full_path}")

    except Exception as e:
        print(f"FATAL: An error occurred while loading city data: {e}")
        CITY_DATA_DF = pd.DataFrame()

# Load city data at script startup, just like in create_open.py
load_city_data("tools/Climate_offline/data/worldcities.csv")

def get_coords_for_city(city_name):
    """
    A robust, local function to get coordinates directly from the loaded DataFrame.
    This bypasses the emulator tool for reliability, mirroring create_open.py.
    """
    if CITY_DATA_DF is None or CITY_DATA_DF.empty:
        return None
    
    city_info = CITY_DATA_DF[CITY_DATA_DF['city'].str.lower() == city_name.lower()]
    
    if not city_info.empty:
        return {
            "latitude": city_info.iloc[0]['lat'],
            "longitude": city_info.iloc[0]['lng']
        }
    return None

# Open question template derivation function
def derive_quantitative_template(model, open_question):
    """
    Uses the generative model to derive a quantitative question template from an open-ended climate question.
    """
    # From search_topics.py 
    tools_description = """
    [
        {
            "name": "query_lat_and_lon",
            "description": "Finds the latitude and longitude for a given city name.",
            "parameters": { "type": "object", "properties": { "city_name": { "type": "string" } }, "required": ["city_name"] }
        },
        {
            "name": "history_temperature",
            "description": "Retrieves historical average temperature for a specific location and year. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" } }, "required": ["latitude", "longitude", "year"] }
        },
        {
            "name": "future_temperature",
            "description": "Predicts future average temperature for a specific location and year under a given scenario. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" } }, "required": ["latitude", "longitude", "year", "setting"] }
        },
        {
            "name": "diy_greenhouse",
            "description": "Models the local temperature impact of custom CO2 and CH4 emission changes. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_CO2": { "type": "number" }, "delta_CH4": { "type": "number" } }, "required": ["latitude", "longitude", "year", "setting", "delta_CO2", "delta_CH4"] }
        },
        {
            "name": "diy_aerosol",
            "description": "Models the local temperature impact of custom SO2 and Black Carbon (BC) emission changes at specified points. Both delta_SO2 and delta_BC are required. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string", "description": "A string representing a list of (lon, lat) tuples, e.g., '[(-10, 20), (15, 25)]'." } }, "required": ["latitude", "longitude", "year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "diy_aerosol_mean",
            "description": "Models the global average temperature impact of custom SO2 and Black Carbon (BC) emission changes at specified points. Returns a tuple: (description, temperature).",
            "parameters": { "type": "object", "properties": { "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string", "description": "A string representing a list of (lon, lat) tuples, e.g., '[(-10, 20), (15, 25)]'." } }, "required": ["year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "is_land_or_sea",
            "description": "Determines if a coordinate is on land or sea. Returns a tuple: (description, value), where value is 1 for land and 0 for sea.",
            "parameters": { "type": "object", "properties": { "lon": { "type": "number" }, "lat": { "type": "number" } }, "required": ["lon", "lat"] }
        },
        {
            "name": "diff_diy_aerosol_mean",
            "description": "Calculates the difference in global average temperature caused by custom SO2 and Black Carbon (BC) emission changes. Returns a tuple: (0, temperature_difference).",
            "parameters": { "type": "object", "properties": { "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string" } }, "required": ["year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "diy_greenhouse_summary",
            "description": "Provides a summary of temperature predictions under different scenarios with custom CO2 and CH4 changes. Returns a string.",
            "parameters": { "type": "object", "properties": { "longitude": { "type": "number" }, "latitude": { "type": "number" }, "delta_CO2": { "type": "number" }, "delta_CH4": { "type": "number" } }, "required": ["longitude", "latitude", "delta_CO2", "delta_CH4"] }
        },
        {
            "name": "location_summary",
            "description": "Retrieves historical and future temperature data for a location. Returns a tuple: (data_dict, description_string).",
            "parameters": { "type": "object", "properties": { "longitude": { "type": "number" }, "latitude": { "type": "number" } }, "required": ["longitude", "latitude"] }
        }
    ]
    """
    
    prompt = f"""
    You are an expert in climate AI evaluation. Given an open-ended climate question, you must:
    1. Generate a quantitative question template (with placeholders like {{{{city_name}}}}, {{{{year}}}}, etc).
    2. List all **input** placeholders used in the template, in order, as a list of strings in a key called "placeholders".
    3. For each placeholder, extract the actual value from the open-ended question (or infer it if needed), and return a list of values (all as strings, order must match placeholders) in a key called "values".
    4. Also return the answer_template and tools_required as before.

    **Important:**
    - Only include **input placeholders** (i.e., parameters that are required as input to the tools, such as city_name, year, setting, delta_SO2, delta_BC, delta_CO2, delta_CH4, modify_points, etc.) in "placeholders" and "values".
    - MUST ONLY use placeholders from the "Available Placeholders" list below or exact same name as the below tools description's key name, you should not do any slight modification to the placeholder names. Like `city_name1` is forbidden, only `city_name` is allowed.
    - Use `year_hist` for historical years and `year_future` for future years when applicable.
    - **Do NOT include any calculated/derived values** (such as hist_future_temp_diff, local_temp_diff, temp_diff_aerosol_greenhouse, etc.) in "placeholders" or "values". These should only appear in the answer_template, not as input parameters.
    - If a value is not present, infer it or use a reasonable default, but always fill all placeholders.
    
    **CRITICAL TEMPLATE FORMATTING RULES:**
    - The `question_template` and `answer_template` must use ONLY simple placeholder substitution (e.g. {{{{city_name}}}}, {{{{temperature}}}})
    - DO NOT include any Python code, conditional statements, expressions, or logic in the templates
    - DO NOT use constructs like {{'text' if condition else 'other'}} or {{{{ expression }}}}
    - Keep templates simple and direct with only {{{{placeholder_name}}}} format
    - All conditional logic should be handled by the system, not embedded in templates
    - !!You must use exact the same key name as example in this text. Like we have delta_CO2 here, then all usage should be delta_CO2, do not generate things like delta_CO2_1 or delta_CO2_2.!!
    
    **Available Tools:**
    ```json
    {tools_description}
    ```

    **Available Placeholders for Answer Templates (For non-mentioned please follow the tools description):**
    *   **Direct Tool Outputs**:
        *   `hist_temp`: Historical temperature from `history_temperature`.
        *   `future_temp`: Baseline future temperature from `future_temperature`.
        *   `aerosol_temp`: Local future temperature with aerosol changes from `diy_aerosol`.
        *   `greenhouse_temp`: Local future temperature with greenhouse gas changes from `diy_greenhouse`.
        *   `global_temp_diff`: Global average temperature difference from the second element of `diff_diy_aerosol_mean`'s tuple output.
        *   `land_sea_result`: "land" or "sea" from `is_land_or_sea`.
        *   `temp_summary`: Text summary from `diy_greenhouse_summary`.
    *   **Calculated/Derived Values** (for answer_template only, Shouldn't in question_template)(Use only when two placeholders in the brackets all exist):
        *   `hist_future_temp_diff`: Difference between future and historical temperatures (`future_temp` - `hist_temp`).
        *   `local_temp_diff`: Difference in local temperature with and without aerosol changes (`aerosol_temp` - `future_temp`).
        *   `modified_hist_future_temp_diff`: Difference between modified future temperature and historical temperature (`aerosol_temp` - `hist_temp`).
        *   `temp_diff_aerosol_greenhouse`: Difference between temperature with greenhouse changes and with aerosol changes (`greenhouse_temp` - `aerosol_temp`).
        *   `net_temp_change`: `modified_hist_future_temp_diff` - `hist_future_temp_diff`.
        *   `diff_global_local`: `local_temp_diff` - `global_temp_diff`.
        *   `total_temp_diff`: `greenhouse_temp` - `global_temp_diff`.


    **Examples:**
    - Open-Ended Question: "In the year 2045 under the ssp585 scenario, for the city of Sendenhorst, what would be the predicted local temperature if we implement an aerosol modification with 9.82% change in SO2 and -8.82% change in BC at points [(7.8278, 51.8439)], versus if we instead adjust greenhouse gases with a 23.96% change in CO2 and 38.42% change in CH4? What are the implications of these differing outcomes for local adaptation strategies in Sendenhorst?"
    - Your Output (JSON):
    {{
        "question_template": "In the year {{{{year}}}} under the {{{{setting}}}} scenario, for the city of {{{{city_name}}}}, what would be the predicted local temperature if we implement an aerosol modification with {{{{delta_SO2}}}}% change in SO2 and {{{{delta_BC}}}}% change in BC at points {{{{modify_points}}}}, versus if we instead adjust greenhouse gases with a {{{{delta_CO2}}}}% change in CO2 and {{{{delta_CH4}}}}% change in CH4? What is the difference between these two predicted temperatures?",
        "answer_template": "Under the {{{{setting}}}} scenario in {{{{year}}}}, the predicted local temperature for {{{{city_name}}}} with the specified aerosol modification is {{{{aerosol_temp}}}}°C. If greenhouse gas adjustments are made instead, the predicted local temperature is {{{{greenhouse_temp}}}}°C. The difference between the greenhouse gas scenario and the aerosol scenario is {{{{temp_diff_aerosol_greenhouse}}}}°C.",
        "tools_required": ["query_lat_and_lon", "diy_aerosol", "diy_greenhouse"],
        "placeholders": ["year", "setting", "city_name", "delta_SO2", "delta_BC", "modify_points", "delta_CO2", "delta_CH4"],
        "values": ["2045", "ssp585", "Sendenhorst", "9.82", "-8.82", "[(-7.8278, 51.8439)]", "23.96", "38.42"]
    }}

    - Open-Ended Question: "For San Bernardino in 2049 under the ssp370 scenario, what are the key policy considerations for local authorities regarding the distinct temperature outcomes resulting from a 44.96% increase in CO2 and a 45.61% increase in CH4 compared to a scenario with an 18.35% decrease in SO2 and a 10.1% decrease in Black Carbon at points [(-117.2943, 34.1416)]?"
    - Your Output (JSON):
    {{
        "question_template": "For {{{{city_name}}}} in {{{{year}}}} under the {{{{setting}}}} scenario, what would be the local temperature if CO2 emissions changed by {{{{delta_CO2}}}}% and CH4 by {{{{delta_CH4}}}}%? Separately, what would be the local temperature if SO2 emissions changed by {{{{delta_SO2}}}}% and Black Carbon by {{{{delta_BC}}}}% at points {{{{modify_points}}}}? What is the temperature difference between these two hypothetical scenarios?",
        "answer_template": "Under the {{{{setting}}}} scenario for {{{{city_name}}}} in {{{{year}}}}, a {{{{delta_CO2}}}}% change in CO2 and {{{{delta_CH4}}}}% change in CH4 results in a local temperature of {{{{greenhouse_temp}}}}°C. Alternatively, a {{{{delta_SO2}}}}% change in SO2 and {{{{delta_BC}}}}% change in BC at {{{{modify_points}}}} results in a local temperature of {{{{aerosol_temp}}}}°C. The difference between the greenhouse gas scenario and the aerosol scenario is {{{{temp_diff_aerosol_greenhouse}}}}°C.",
        "tools_required": ["query_lat_and_lon", "history_temperature", "diy_greenhouse", "diy_aerosol"],
        "placeholders": ["city_name", "year", "setting", "delta_CO2", "delta_CH4", "delta_SO2", "delta_BC", "modify_points"],
        "values": ["San Bernardino", "2049", "ssp370", "44.96", "45.61", "-18.35", "-10.1", "[(-117.2943, 34.1416)]"]
    }}

    - Open-Ended Question: "For Bad Kissingen, what is the total projected temperature change from its 1983 historical average to the 2051 forecast under the ssp370 scenario with an aerosol modification of -17.82% SO2 and -16.11% BC at [(10.0667, 50.2)]? Is this overall change considered negligible (less than 0.2°C), modest (0.2°C to 1.0°C), or significant (over 1.0°C) and is that a warming or cooling trend?",
    - Your Output (JSON):
    {{
        "question_template": "What was the historical average temperature for {{city_name}} in {{year_hist}}? Considering the {{setting}} scenario, what would be the projected temperature for {{city_name}} in {{year_future}} if SO2 emissions were modified by {{delta_SO2}}% and Black Carbon by {{delta_BC}}% at points {{modify_points}}?",
        "answer_template": "The historical average temperature for {{city_name}} in {{year_hist}} was {{hist_temp}}°C. With the specified SO2 and Black Carbon modifications at {{modify_points}}, the projected temperature for {{city_name}} in {{year_future}} under the {{setting}} scenario would be {{aerosol_temp}}°C. This represents a change of {{modified_hist_future_temp_diff}}°C compared to the historical temperature.",
        "tools_required": ["query_lat_and_lon", "history_temperature", "diy_aerosol"],
        "placeholders": ["city_name", "year_hist", "setting", "year_future", "delta_SO2", "delta_BC", "modify_points"],
        "values": ["Bad Kissingen", "1983", "ssp370", "2051", "-17.82", "-16.11", "[(10.0667, 50.2)]"]
    }}
    
    **Open-Ended Question:**
    "{open_question}"
    """

    retries = 5
    for attempt in range(retries):
        try:
            response = model.generate_content(prompt)
            cleaned_response = response.text.strip().replace("```json", "").replace("```", "").strip()
            template = json.loads(cleaned_response)
            if all(k in template for k in ['question_template', 'answer_template', 'tools_required']):
                return template
        except (ValueError, json.JSONDecodeError) as e:
            print(f"Warning: Failed to decode JSON on attempt {attempt + 1}/{retries}. Error: {e}")
        except Exception as e:
            print(f"Warning: API call failed on attempt {attempt + 1}/{retries}. Error: {e}")
        
        if attempt < retries - 1:
            print("Retrying in 30 seconds...")
            time.sleep(30)
    
    print("Error: Could not generate a valid template after multiple retries.")
    return None

# --- Helper Functions (adapted from create_open.py) ---

class SafeDict(dict):
    def __missing__(self, key):
        return f'{{{key}}}'

def fill_derived_template(template, params):
    """
    Executes tools to fill the derived quantitative template, mirroring the robust logic
    from create_open.py.
    """
    question_template = template['question_template'].replace('{{', '{').replace('}}', '}')
    answer_template = template['answer_template'].replace('{{', '{').replace('}}', '}')
    tools_required = template.get('tools_required', [])

    try:
        # Normalize LLM-generated parameter names to standard format
        param_mappings = {
            'delta_CO2_1': 'delta_CO2', 'delta_CO2_2': 'delta_CO2_alt',
            'delta_CH4_1': 'delta_CH4', 'delta_CH4_2': 'delta_CH4_alt',
            'delta_CO2_hist': 'delta_CO2', 'delta_CH4_hist': 'delta_CH4',
            'delta_CO2_future': 'delta_CO2_alt', 'delta_CH4_future': 'delta_CH4_alt',
        }
        
        # Apply parameter mappings
        for old_key, new_key in param_mappings.items():
            if old_key in params and new_key not in params:
                params[new_key] = params[old_key]
                print(f"DEBUG: Mapped {old_key} -> {new_key}: {params[old_key]}")

        # Type conversion for numeric parameters
        input_keys_float = [
            'delta_SO2', 'delta_BC', 'delta_CO2', 'delta_CH4',
            'delta_CO2_greenhouse', 'delta_CH4_greenhouse', 'longitude', 'latitude',
            'hist_temp', 'future_temp', 'aerosol_temp', 'greenhouse_temp',  # Add tool result keys
            'delta_CO2_alt', 'delta_CH4_alt'  # Alternative parameter names
        ]
        input_keys_int = ['year', 'year_hist', 'year_future']

        for key in input_keys_float:
            if key in params and params[key] is not None:
                try:
                    params[key] = float(params[key])
                except (ValueError, TypeError):
                    print(f"Warning: Could not convert parameter '{key}' with value '{params[key]}' to float, setting to None.")
                    params[key] = None

        for key in input_keys_int:
            if key in params and params[key] is not None:
                try:
                    params[key] = int(params[key])
                except (ValueError, TypeError):
                    print(f"Warning: Could not convert parameter '{key}' with value '{params[key]}' to int, setting to None.")
                    params[key] = None

        # Coordinate resolution
        if 'city_name' in params:
            coords = get_coords_for_city(params['city_name'])
            if coords:
                params['latitude'] = float(coords['latitude'])
                params['longitude'] = float(coords['longitude'])
            else:
                print(f"Skipping: Could not find coordinates for city '{params['city_name']}' in the local data file.")
                return None
                
        # Validate climate scenario setting
        valid_settings = ['ssp126', 'ssp245', 'ssp370', 'ssp585']
        if 'setting' in params:
            if params['setting'] not in valid_settings:
                print(f"Warning: Invalid setting '{params['setting']}', using default 'ssp245'")
                params['setting'] = 'ssp245'
        else:
            print(f"Warning: No setting provided, using default 'ssp245'")
            params['setting'] = 'ssp245'

        tool_results = {}

        # Tool execution with error handling
        def to_float(value, tool_name):
            if value is None:
                print(f"Skipping: Tool '{tool_name}' returned a None value.")
                return None
            if isinstance(value, (float, int, np.number)):
                return float(value)
            if isinstance(value, str):
                # Check if this is an error message first
                if "must be between" in value.lower() or "error" in value.lower() or "provided" in value.lower():
                    print(f"Skipping: Tool '{tool_name}' returned an error message: {value}")
                    return None
                
                numbers = re.findall(r"[-+]?\d*\.\d+|\d+", value)
                if numbers:
                    try:
                        return float(numbers[-1])
                    except (ValueError, TypeError): pass
            print(f"Skipping: Tool '{tool_name}' returned a non-numeric value or could not parse one: {value}")
            return None

        # Ensure modify_points is properly formatted
        if 'modify_points' in params and params['modify_points']:
            modify_points_str = params['modify_points']
            # Validate format - should be like "[(-3.77, 35.18)]"
            if not (modify_points_str.startswith('[') and modify_points_str.endswith(']')):
                print(f"Warning: Invalid modify_points format: {modify_points_str}, using coordinates from city")
                modify_points_str = f"[({params.get('longitude')}, {params.get('latitude')})]"
        else:
            # Default to city coordinates
            modify_points_str = f"[({params.get('longitude')}, {params.get('latitude')})]"
        
        params['modify_points'] = modify_points_str
        print(f"DEBUG: Using modify_points: {modify_points_str}")

        if 'history_temperature' in tools_required:
            year = params.get('year_hist') # Use standardized key
            if year is None:
                # Smart fallback: infer historical year if not provided
                future_year = params.get('year_future', params.get('year'))
                if future_year:
                    # Use a reasonable historical baseline within valid range (1850-2014)
                    # Choose a year that's reasonable for climate comparison
                    inferred_hist_year = min(2014, max(1980, 2000))  # Default to year 2000
                    print(f"DEBUG: Inferring historical year {inferred_hist_year} for comparison with future year {future_year}")
                    year = inferred_hist_year
                else:
                    print(f"Skipping: No historical or future year available for history_temperature.")
                    return None
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for history_temperature")
                return None
                
            val, _ = TOOL_MAP['history_temperature'](longitude=params['longitude'], latitude=params['latitude'], year=year)
            temp = to_float(val, 'history_temperature')
            if temp is None: return None
            tool_results['hist_temp'] = round(temp, 2)

        if 'future_temperature' in tools_required:
            year = params.get('year_future', params.get('year')) # Use standardized keys
            if year is None:
                print(f"Skipping: 'year_future' or 'year' not found in params for future_temperature.")
                return None
            # Ensure year is within valid range for future_temperature (2015-2100)
            year = int(year)
            if year < 2015:
                year = 2050  # Default to mid-century
                print(f"DEBUG: Adjusting year to {year} (within valid range 2015-2100)")
            elif year > 2100:
                year = 2100
                print(f"DEBUG: Adjusting year to {year} (within valid range 2015-2100)")
            
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for future_temperature")
                return None
                
            val, _ = TOOL_MAP['future_temperature'](longitude=params['longitude'], latitude=params['latitude'], year=year, setting=params['setting'])
            temp = to_float(val, 'future_temperature')
            if temp is None: return None
            tool_results['future_temp'] = round(temp, 2)

        if 'diy_aerosol' in tools_required:
            year = params.get('year_future', params.get('year')) # Use standardized keys
            if year is None:
                print(f"Skipping: 'year_future' or 'year' not found in params for diy_aerosol.")
                return None
            # Ensure year is within valid range for diy_aerosol (2015-2100)
            year = int(year)
            if year < 2015:
                year = 2050
                print(f"DEBUG: Adjusting aerosol year to {year} (within valid range 2015-2100)")
            elif year > 2100:
                year = 2100
                print(f"DEBUG: Adjusting aerosol year to {year} (within valid range 2015-2100)")
            # Ensure we have valid aerosol parameters
            delta_so2 = params.get('delta_SO2', 0.0)
            delta_bc = params.get('delta_BC', 0.0)
            
            if delta_so2 is None:
                print(f"Warning: No SO2 delta found, using default 0%")
                delta_so2 = 0.0
            if delta_bc is None:
                print(f"Warning: No BC delta found, using default 0%")
                delta_bc = 0.0
                
            try:
                delta_so2_val = float(delta_so2) / 100
                delta_bc_val = float(delta_bc) / 100
            except (ValueError, TypeError) as e:
                print(f"Skipping: Invalid aerosol parameters - SO2: {delta_so2}, BC: {delta_bc}, Error: {e}")
                return None
            
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for diy_aerosol")
                return None
                
            temp_val, _ = TOOL_MAP['diy_aerosol'](
                longitude=params['longitude'], latitude=params['latitude'], year=year,
                setting=params['setting'], delta_SO2=delta_so2_val,
                delta_BC=delta_bc_val, modify_points=modify_points_str
            )
            temp = to_float(temp_val, 'diy_aerosol')
            if temp is None: return None
            tool_results['aerosol_temp'] = round(temp, 2)

        if 'diff_diy_aerosol_mean' in tools_required:
            year = params.get('year_future', params.get('year')) # Use standardized keys
            if year is None:
                print(f"Skipping: 'year_future' or 'year' not found in params for diff_diy_aerosol_mean.")
                return None
            # Ensure year is within valid range for diff_diy_aerosol_mean (2015-2100)
            year = int(year)
            if year < 2015:
                year = 2050
                print(f"DEBUG: Adjusting aerosol_mean year to {year} (within valid range 2015-2100)")
            elif year > 2100:
                year = 2100
                print(f"DEBUG: Adjusting aerosol_mean year to {year} (within valid range 2015-2100)")
            # Ensure we have valid aerosol parameters for global calculation
            delta_so2 = params.get('delta_SO2', 0.0)
            delta_bc = params.get('delta_BC', 0.0)
            
            if delta_so2 is None:
                print(f"Warning: No SO2 delta found for global calc, using default 0%")
                delta_so2 = 0.0
            if delta_bc is None:
                print(f"Warning: No BC delta found for global calc, using default 0%")
                delta_bc = 0.0
                
            try:
                delta_so2_val = float(delta_so2) / 100
                delta_bc_val = float(delta_bc) / 100
            except (ValueError, TypeError) as e:
                print(f"Skipping: Invalid global aerosol parameters - SO2: {delta_so2}, BC: {delta_bc}, Error: {e}")
                return None
                
            _, temp_val = TOOL_MAP['diff_diy_aerosol_mean'](
                year=year, setting=params['setting'], delta_SO2=delta_so2_val,
                delta_BC=delta_bc_val, modify_points=modify_points_str
            )
            temp = to_float(temp_val, 'diff_diy_aerosol_mean')
            if temp is None: return None
            tool_results['global_temp_diff_val'] = round(temp, 2)

        if 'diy_greenhouse' in tools_required:
            year = params.get('year_future', params.get('year')) # Use standardized keys
            if year is None:
                print(f"Skipping: 'year_future' or 'year' not found in params for diy_greenhouse.")
                return None
            # Ensure year is within valid range for diy_greenhouse (2015-2100)
            year = int(year)
            if year < 2015:
                year = 2050
                print(f"DEBUG: Adjusting greenhouse year to {year} (within valid range 2015-2100)")
            elif year > 2100:
                year = 2100
                print(f"DEBUG: Adjusting greenhouse year to {year} (within valid range 2015-2100)")
            delta_co2_g = params.get('delta_CO2_greenhouse', params.get('delta_CO2'))
            delta_ch4_g = params.get('delta_CH4_greenhouse', params.get('delta_CH4'))
            
            # Ensure we have valid values for both CO2 and CH4
            if delta_co2_g is None:
                print(f"Warning: No CO2 delta found, using default 0%")
                delta_co2_g = 0.0
            if delta_ch4_g is None:
                print(f"Warning: No CH4 delta found, using default 0%") 
                delta_ch4_g = 0.0
                
            try:
                delta_co2_val = float(delta_co2_g) / 100
                delta_ch4_val = float(delta_ch4_g) / 100
            except (ValueError, TypeError) as e:
                print(f"Skipping: Invalid greenhouse gas parameters - CO2: {delta_co2_g}, CH4: {delta_ch4_g}, Error: {e}")
                return None
            
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for diy_greenhouse")
                return None
                
            temp_val, _ = TOOL_MAP['diy_greenhouse'](
                longitude=params['longitude'], latitude=params['latitude'], year=year,
                setting=params['setting'], delta_CO2=delta_co2_val, delta_CH4=delta_ch4_val
            )
            temp = to_float(temp_val, 'diy_greenhouse')
            if temp is None: return None
            tool_results['greenhouse_temp'] = round(temp, 2)

        if 'is_land_or_sea' in tools_required:
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for is_land_or_sea")
                return None
                
            _, result_val = TOOL_MAP['is_land_or_sea'](lon=params['longitude'], lat=params['latitude'])
            tool_results['land_sea_str'] = "land" if result_val == 1 else "sea"

        if 'location_summary' in tools_required:
            # Validate required parameters
            if params.get('longitude') is None or params.get('latitude') is None:
                print(f"Skipping: Missing coordinates for location_summary")
                return None
                
            _, summary_val = TOOL_MAP['location_summary'](longitude=params['longitude'], latitude=params['latitude'])
            tool_results['location_summary_result'] = str(summary_val)

        # Final parameter assembly
        final_params = params.copy()

        # Check if we need to calculate diff_global_local and if future_temp is missing
        if 'diff_global_local' in answer_template and 'future_temp' not in tool_results and 'future_temperature' not in tools_required:
            print("DEBUG: Force-calling future_temperature to calculate diff_global_local.")
            year = params.get('year_future', params.get('year'))
            if year:
                val, _ = TOOL_MAP['future_temperature'](longitude=params['longitude'], latitude=params['latitude'], year=year, setting=params['setting'])
                temp = to_float(val, 'future_temperature')
                if temp is not None:
                    tool_results['future_temp'] = round(temp, 2)
                    print(f"DEBUG: force-called future_temperature returned: {tool_results['future_temp']}")
            else:
                print("Skipping force-call to future_temperature: year is missing.")


        # Direct mapping
        if 'hist_temp' in tool_results:
            final_params['hist_temp'] = tool_results['hist_temp']
        if 'future_temp' in tool_results:
            final_params['future_temp'] = tool_results['future_temp']
            final_params['local_temp_without_aerosols'] = tool_results['future_temp']
        if 'aerosol_temp' in tool_results:
            final_params['aerosol_temp'] = tool_results['aerosol_temp']
            final_params['local_temp_with_aerosols'] = tool_results['aerosol_temp']
            final_params['aerosol_adjusted_temp'] = tool_results['aerosol_temp']
        if 'global_temp_diff_val' in tool_results:
            final_params['global_temp_diff'] = tool_results['global_temp_diff_val']
            final_params['aerosol_temp_diff'] = tool_results['global_temp_diff_val']
            final_params['aerosol_temp_change'] = tool_results['global_temp_diff_val']
        if 'greenhouse_temp' in tool_results:
            final_params['greenhouse_temp'] = tool_results['greenhouse_temp']
        if 'land_sea_str' in tool_results:
            final_params['land_sea_result'] = tool_results['land_sea_str']

        # Derived values - ensure all calculations use float values
        if 'local_temp_with_aerosols' in final_params and 'local_temp_without_aerosols' in final_params:
            diff = float(final_params['local_temp_with_aerosols']) - float(final_params['local_temp_without_aerosols'])
            final_params['local_temp_diff'] = round(diff, 2)
            final_params['aerosol_temp_change'] = round(diff, 2)
            final_params['local_temp_difference'] = round(diff, 2)
            final_params['future_diy_temp_diff'] = round(diff, 2)
        if 'future_temp' in final_params and 'hist_temp' in final_params:
            diff = float(final_params['future_temp']) - float(final_params['hist_temp'])
            final_params['hist_future_temp_diff'] = round(diff, 2)
            final_params['future_hist_diff'] = round(diff, 2)
            final_params['future_temp_change'] = round(diff, 2)
        if 'aerosol_temp' in final_params and 'hist_temp' in final_params:
            diff = float(final_params['aerosol_temp']) - float(final_params['hist_temp'])
            final_params['modified_hist_future_temp_diff'] = round(diff, 2)
            final_params['modified_hist_diff'] = round(diff, 2)
        if 'greenhouse_temp' in final_params and 'hist_temp' in final_params:
            diff = float(final_params['greenhouse_temp']) - float(final_params['hist_temp'])
            final_params['modified_hist_future_temp_diff'] = round(diff, 2)
            final_params['modified_hist_diff'] = round(diff, 2)
        if 'modified_hist_future_temp_diff' in final_params and 'hist_future_temp_diff' in final_params:
            diff = float(final_params['modified_hist_future_temp_diff']) - float(final_params['hist_future_temp_diff'])
            final_params['net_temp_change'] = round(diff, 2)
        if 'greenhouse_temp' in final_params and 'aerosol_temp' in final_params:
            diff = float(final_params['greenhouse_temp']) - float(final_params['aerosol_temp'])
            final_params['temp_diff_aerosol_greenhouse'] = round(diff, 2)
        if 'local_temp_diff' in final_params and 'global_temp_diff' in final_params:
            diff = float(final_params['local_temp_diff']) - float(final_params['global_temp_diff'])
            final_params['diff_global_local'] = round(diff, 2)
        if 'greenhouse_temp' in final_params and 'aerosol_temp_diff' in final_params:
            diff = float(final_params['greenhouse_temp']) - float(final_params['aerosol_temp_diff'])
            final_params['total_temp_diff'] = round(diff, 2)
            
        # Add common placeholder aliases to handle LLM-generated variations
        final_params['description'] = "moderate warming trend"  # Default fallback for {description}
        
        # Add conditional text generation for complex LLM templates
        if 'temp_diff_aerosol_greenhouse' in final_params:
            if final_params['temp_diff_aerosol_greenhouse'] > 0:
                final_params['comparison_text'] = "the greenhouse gas intervention is more substantial"
            else:
                final_params['comparison_text'] = "the aerosol modification is more substantial"
        
        if 'aerosol_temp' in final_params and 'greenhouse_temp' in final_params:
            if float(final_params['aerosol_temp']) < float(final_params['greenhouse_temp']):
                final_params['cooling_warming'] = "cooling"
            else:
                final_params['cooling_warming'] = "warming"

        # Check for problematic Python code in templates (LLM API issue)
        if "if " in answer_template and " else " in answer_template and "'" in answer_template:
            print(f"Warning: LLM generated answer template contains Python code, which is invalid. Skipping...")
            return None
        
        # Standardize final parameters
        try:
            final_question = question_template.format_map(SafeDict(final_params))
            final_answer = answer_template.format_map(SafeDict(final_params))
        except (AttributeError, KeyError, ValueError) as e:
            print(f"Warning: Template formatting failed due to invalid template: {e}")
            return None

        if '{' in final_question or '}' in final_answer:
            print(f"Warning: Unfilled placeholders remain. Q: {final_question} A: {final_answer}")
            return None

        return {"quantitative_question": final_question, "quantitative_answer": final_answer}

    except Exception as e:
        print(f"Error during tool execution for city '{params.get('city_name', params.get('city', 'UNKNOWN'))}': {e}")
        traceback.print_exc()
        return None

# --- Main Execution Logic ---

def main():
    """
    Main function to process open questions, derive quantitative questions,
    and run emulators to get answers.
    """
    # Load environment and API key
    dotenv_path = os.path.join(os.path.dirname(__file__), '..', '..', '.env')
    load_dotenv(dotenv_path=dotenv_path)
    api_key = os.getenv("Google_API_KEY")
    if not api_key:
        raise ValueError("Google_API_KEY not found in .env file.")
    
    genai.configure(api_key=api_key)
    model = genai.GenerativeModel('gemini-2.5-flash')

    # Load open questions
    try:
        with open("open_questions.json", 'r', encoding='utf-8') as f:
            open_questions = json.load(f)
    except FileNotFoundError:
        print("Error: open_questions.json not found. Please run upsample_open.py first.")
        return

    derived_qa_pairs = []
    for i, item in enumerate(open_questions):
        print(f"--- Processing open question {i+1}/{len(open_questions)} ---")
        open_q = item['open_question']
        
        max_attempts = 3
        quantitative_qa = None

        for attempt in range(max_attempts):
            print(f"-> Attempt {attempt + 1}/{max_attempts}...")
            
            # Step 1: Derive the quantitative template and values
            print("Step 1: Deriving quantitative template...")
            derived_template = derive_quantitative_template(model, open_q)
            if not derived_template:
                print("-> Failed to derive template. Retrying...")
                continue

            # Step 2: Extract placeholders and values
            print("Step 2: Extracting placeholders and values...")
            placeholders = derived_template.get("placeholders", [])
            values = derived_template.get("values", [])
            if not placeholders or not values or len(placeholders) != len(values):
                print("-> Placeholders/values mismatch or missing. Retrying...")
                continue
            params = dict(zip(placeholders, values))
            print(f"-> Success. Params: {params}")

            # Step 3: Fill the template by running emulators
            print("Step 3: Running emulators...")
            quantitative_qa = fill_derived_template(derived_template, params)
            
            if quantitative_qa:
                print("-> Success. Quantitative Q&A generated.")
                break  # Exit loop on success
            else:
                print(f"-> Failed to run emulators and fill template. Retrying...")
        
        if not quantitative_qa:
            print("-> Failed to generate a valid Q&A pair after all retries. Saving record with null values.")
            quantitative_qa = {
                "quantitative_question": None,
                "quantitative_answer": None
            }

        # Combine everything
        final_record = {
            "open_question": open_q,
            "original_open_answer": item['open_answer'],
            "derived_quantitative_question": quantitative_qa['quantitative_question'],
            "derived_quantitative_answer": quantitative_qa['quantitative_answer']
        }
        derived_qa_pairs.append(final_record)

    # Save the final output
    output_file = "derived_quantitative_questions.json"
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(derived_qa_pairs, f, ensure_ascii=False, indent=4)

    print(f"\nSuccessfully processed {len(derived_qa_pairs)} questions.")
    print(f"Output saved to {output_file}")

if __name__ == "__main__":
    main()