import os
import json
from dotenv import load_dotenv
import google.generativeai as genai
import sys
import time

# Add the src directory to the Python path to allow importing functions_climate
# This assumes the script is run from src/Climate
sys.path.append(os.path.join(os.path.dirname(__file__), '..'))
from functions.climate import functions_climate

def generate_topic(client):
    """
    Generates a question topic using the Gemini API based on available tools.
    The prompt is designed to generate complex questions by showing examples of combining tools.
    """
    # Use a detailed, hand-crafted tool description to ensure accuracy.
    tools_description = """
    [
        {
            "name": "query_lat_and_lon",
            "description": "Finds the latitude and longitude for a given city name.",
            "parameters": { "type": "object", "properties": { "city_name": { "type": "string" } }, "required": ["city_name"] }
        },
        {
            "name": "history_temperature",
            "description": "Retrieves historical average temperature for a specific location and year. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" } }, "required": ["latitude", "longitude", "year"] }
        },
        {
            "name": "future_temperature",
            "description": "Predicts future average temperature for a specific location and year under a given scenario. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" } }, "required": ["latitude", "longitude", "year", "setting"] }
        },
        {
            "name": "diy_greenhouse",
            "description": "Models the local temperature impact of custom CO2 and CH4 emission changes. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_CO2": { "type": "number" }, "delta_CH4": { "type": "number" } }, "required": ["latitude", "longitude", "year", "setting", "delta_CO2", "delta_CH4"] }
        },
        {
            "name": "diy_aerosol",
            "description": "Models the local temperature impact of custom SO2 and Black Carbon (BC) emission changes at specified points. Both delta_SO2 and delta_BC are required. Returns a tuple: (temperature, description).",
            "parameters": { "type": "object", "properties": { "latitude": { "type": "number" }, "longitude": { "type": "number" }, "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string", "description": "A string representing a list of (lon, lat) tuples, e.g., '[(-10, 20), (15, 25)]'." } }, "required": ["latitude", "longitude", "year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "diy_aerosol_mean",
            "description": "Models the global average temperature impact of custom SO2 and Black Carbon (BC) emission changes at specified points. Returns a tuple: (description, temperature).",
            "parameters": { "type": "object", "properties": { "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string", "description": "A string representing a list of (lon, lat) tuples, e.g., '[(-10, 20), (15, 25)]'." } }, "required": ["year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "is_land_or_sea",
            "description": "Determines if a coordinate is on land or sea. Returns a tuple: (description, value), where value is 1 for land and 0 for sea.",
            "parameters": { "type": "object", "properties": { "lon": { "type": "number" }, "lat": { "type": "number" } }, "required": ["lon", "lat"] }
        },
        {
            "name": "diff_diy_aerosol_mean",
            "description": "Calculates the difference in global average temperature caused by custom SO2 and Black Carbon (BC) emission changes. Returns a tuple: (0, temperature_difference).",
            "parameters": { "type": "object", "properties": { "year": { "type": "integer" }, "setting": { "type": "string" }, "delta_SO2": { "type": "number" }, "delta_BC": { "type": "number" }, "modify_points": { "type": "string" } }, "required": ["year", "setting", "delta_SO2", "delta_BC", "modify_points"] }
        },
        {
            "name": "diy_greenhouse_summary",
            "description": "Provides a summary of temperature predictions under different scenarios with custom CO2 and CH4 changes. Returns a string.",
            "parameters": { "type": "object", "properties": { "longitude": { "type": "number" }, "latitude": { "type": "number" }, "delta_CO2": { "type": "number" }, "delta_CH4": { "type": "number" } }, "required": ["longitude", "latitude", "delta_CO2", "delta_CH4"] }
        },
        {
            "name": "location_summary",
            "description": "Retrieves historical and future temperature data for a location. Returns a tuple: (data_dict, description_string).",
            "parameters": { "type": "object", "properties": { "longitude": { "type": "number" }, "latitude": { "type": "number" } }, "required": ["longitude", "latitude"] }
        }
    ]
    """

    prompt = f"""
    Your task is to generate a single question template in a JSON format. This question should require invoking one to three of the available climate simulator tools to be answered. The questions should be challenging and creative, but shouldn't exceed three tools in usage.

    **Constraint Checklist & Output Format:**
    1.  The output MUST be a single JSON object. Do not include any text or markdown formatting like ```json ... ``` outside the object.
    2.  The JSON object must contain exactly three keys: "question_template", "answer_template", and "tools_required".
    3.  `"tools_required"`: A list of the names of the tools needed to answer the question.
    4.  `"question_template"` & `"answer_template"`: Strings with placeholders.
    5.  **Placeholder-Tool Dependency**: You MUST ensure that for every tool listed in `"tools_required"`, all necessary input placeholders appear in the `"question_template"`.
    6.  **Answer Placeholders**: The `answer_template` MUST ONLY use placeholders from the "Available Placeholders" list below or exact same name as the below tools description's key name, you should not do any slight modification to the placeholder names. Like `city_name1` is forbidden, only `city_name` is allowed.

    **Available Tools & Their Descriptions:**
    ```json
    {tools_description}
    ```

    **Available Placeholders for Answer Templates(For non-mentioned please follow the tools description):**
    *   **Direct Tool Outputs**:
        *   `hist_temp`: Historical temperature from `history_temperature`.
        *   `future_temp`: Baseline future temperature from `future_temperature`.
        *   `aerosol_temp`: Local future temperature with aerosol changes from `diy_aerosol`.
        *   `greenhouse_temp`: Local future temperature with greenhouse gas changes from `diy_greenhouse`.
        *   `global_temp_diff`: Global average temperature difference from the second element of `diff_diy_aerosol_mean`'s tuple output.
        *   `land_sea_result`: "land" or "sea" from `is_land_or_sea`.
        *   `temp_summary`: Text summary from `diy_greenhouse_summary`.
    *   **Calculated/Derived Values**(Use only when two placeholders in the brackets all exist):
        *   `hist_future_temp_diff`: Difference between future and historical temperatures (`future_temp` - `hist_temp`).
        *   `local_temp_diff`: Difference in local temperature with and without aerosol changes (`aerosol_temp` - `future_temp`).
        *   `modified_hist_future_temp_diff`: Difference between modified future temperature and historical temperature (`aerosol_temp` - `hist_temp`).
        *   `temp_diff_aerosol_greenhouse`: Difference between temperature with greenhouse changes and with aerosol changes (`greenhouse_temp` - `aerosol_temp`).
        *   `net_temp_change`: `modified_hist_future_temp_diff` - `hist_future_temp_diff`.
        *   `diff_global_local`: `local_temp_diff` - `global_temp_diff`.
        *   `total_temp_diff`: `greenhouse_temp` - `global_temp_diff`.
        *.  `delta_CO2`: Percentage change in CO2 emissions.
        *   `delta_CH4`: Percentage change in CH4 emissions.
        *   `delta_SO2`: Percentage change in SO2 emissions.
        *   `delta_BC`: Percentage change in Black Carbon emissions.
        *   `modify_points`: A string representing a list of (lon, lat) tuples, e.g., '[(-10, 20), (15, 25)]'.

    **Examples of High-Quality Question Templates:**

    *   **Example 1: Historical vs. Future Comparison**
        ```json
        {{
          "question_template": "How is the temperature in {{{{city_name}}}} expected to change from the historical year {{{{year_hist}}}} to the future year {{{{year_future}}}} under the {{{{setting}}}} scenario?",
          "answer_template": "The temperature in {{{{city_name}}}} was {{{{hist_temp}}}}°C in {{{{year_hist}}}}. It is projected to be {{{{future_temp}}}}°C in {{{{year_future}}}} under the {{{{setting}}}} scenario, marking a change of {{{{hist_future_temp_diff}}}}°C.",
          "tools_required": ["query_lat_and_lon", "history_temperature", "future_temperature"]
        }}
        ```

    *   **Example 2: "What-If" Greenhouse Gas Scenario**
        ```json
        {{
          "question_template": "What is the baseline predicted temperature for {{{{city_name}}}} in the year {{{{year}}}} under the {{{{setting}}}} scenario? How would this prediction change if CO2 emissions increased by {{{{delta_CO2}}}}% and CH4 emissions increased by {{{{delta_CH4}}}}%?",
          "answer_template": "The baseline temperature for {{{{city_name}}}} in {{{{year}}}} ({{{{setting}}}}) is {{{{future_temp}}}}°C. With a {{{{delta_CO2}}}}% CO2 increase and {{{{delta_CH4}}}}% CH4 increase, the new temperature would be {{{{greenhouse_temp}}}}°C.",
          "tools_required": ["query_lat_and_lon", "future_temperature", "diy_greenhouse"]
        }}
        ```

    *   **Example 3: Location Analysis**
        ```json
        {{
          "question_template": "Is the city of {{{{city_name}}}} located on land or sea? What was its average temperature in the year {{{{year_hist}}}}?",
          "answer_template": "The city of {{{{city_name}}}} is located on {{{{land_sea_result}}}}. Its average temperature in {{{{year_hist}}}} was {{{{hist_temp}}}}°C.",
          "tools_required": ["query_lat_and_lon", "is_land_or_sea", "history_temperature"]
        }}
        ```

    *   **Example 4: Local vs Global Aerosol Impact**
        ```json
        {{
            "question_template": "For the area around {{{{city_name}}}}, if we reduce SO2 by {{{{delta_SO2}}}}% and BC by {{{{delta_BC}}}}% in {{{{year_future}}}} under the {{{{setting}}}} scenario, what is the expected local temperature versus the global average temperature change? The modification is applied to these points: {{{{modify_points}}}}.",
            "answer_template": "With the specified aerosol changes near {{{{city_name}}}}, the local temperature is predicted to be {{{{aerosol_temp}}}}°C, a change of {{{{local_temp_diff}}}}°C from the baseline {{{{future_temp}}}}°C. The global average temperature change is predicted to be {{{{global_temp_diff}}}}°C. The difference between the local and global impact is {{{{diff_global_local}}}}°C.",
            "tools_required": ["query_lat_and_lon", "future_temperature", "diy_aerosol", "diff_diy_aerosol_mean"]
        }}
        ```

    *   **Example 5: Land/Sea Aerosol Analysis**
        ```json
        {{
          "question_template": "We are considering an aerosol modification in {{{{year}}}} under the {{{{setting}}}} scenario, with a {{{{delta_SO2}}}}% change in SO2 and {{{{delta_BC}}}}% change in BC at coordinates {{{{modify_points}}}}. What is the predicted local temperature at {{{{city_name}}}} after this change, and is this location on land or sea?",
          "answer_template": "After the aerosol modification, the temperature at {{{{city_name}}}} is predicted to be {{{{aerosol_temp}}}}°C. The location is on {{{{land_sea_result}}}}.",
          "tools_required": ["query_lat_and_lon", "diy_aerosol", "is_land_or_sea"]
        }}
        ```
    *   **Example 6: Global Aerosol Impact**
        ```json
        {{
          "question_template": "What is the global average temperature difference if we implement an aerosol injection program in {{{{year}}}} under the {{{{setting}}}} scenario, with a {{{{delta_SO2}}}}% change in SO2 and {{{{delta_BC}}}}% change in BC at coordinates {{{{modify_points}}}}?",
          "answer_template": "The global average temperature difference from the aerosol injection program is {{{{global_temp_diff}}}}°C.",
          "tools_required": ["diff_diy_aerosol_mean"]
        }}
        ```
    *   **Example 7: Greenhouse Gas Summary**
        ```json
        {{
          "question_template": "Provide a summary of temperature predictions for {{{{city_name}}}} if CO2 emissions change by {{{{delta_CO2}}}}% and CH4 emissions change by {{{{delta_CH4}}}}%.",
          "answer_template": "{{{{temp_summary}}}}",
          "tools_required": ["query_lat_and_lon", "diy_greenhouse_summary"]
        }}
        ```

    Now, please generate a new, complex question template that is different from the examples, please ensure not use more than three tools.
    """

    try:
        response = client.generate_content(prompt)
        # Clean up the response to ensure it's valid JSON
        cleaned_response = response.text.strip().replace("```json", "").replace("```", "").strip()
        topic = json.loads(cleaned_response)
        return topic
    except (ValueError, json.JSONDecodeError) as e:
        print(f"Error decoding JSON from Gemini response: {e}")
        print(f"Raw response: {response.text}")
        return None
    except Exception as e:
        print(f"An unexpected error occurred: {e}")
        return None

def main():
    """
    Main function to generate and save question topics.
    """
    # Load environment variables from .env file in the current directory
    dotenv_path = os.path.join(os.path.dirname(__file__), '..', '..', '.env')
    load_dotenv(dotenv_path=dotenv_path)

    api_key = os.getenv("Google_API_KEY")
    if not api_key:
        raise ValueError("Google_API_KEY not found in environment variables. Please create a .env file in src/Climate/ with your key.")

    # Use the client from genai library as requested
    genai.configure(api_key=api_key)
    model = genai.GenerativeModel('gemini-2.5-flash')


    num_topics = 20
    topics = []
    print(f"Generating {num_topics} topics...")
    while len(topics) < num_topics:
        # Pass the generative model to the function
        topic = generate_topic(model)
        if topic and all(k in topic for k in ['question_template', 'answer_template', 'tools_required']):
            topics.append(topic)
            print(f"Generated topic {len(topics)}/{num_topics}")
        else:
            print("Failed to generate a valid topic, retrying in 30 seconds...")
            time.sleep(30)

    output_file = "topics.json"
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(topics, f, ensure_ascii=False, indent=4)

    print(f"Successfully generated and saved {len(topics)} topics to {output_file}")

if __name__ == "__main__":
    main()