import os
import json
from dotenv import load_dotenv
import google.generativeai as genai
import time

def upsample_question(model, original_question, original_answer):
    """
    Uses Gemini to rephrase a factual Q&A pair into a more open-ended one.
    """
    prompt = f"""
    You are an expert in designing evaluation tasks for advanced AI models. Your goal is to convert a quantitative, fact-based climate Q&A pair into a new, open-ended question and a corresponding qualitative answer. This new pair will test a model's ability to first infer and calculate necessary numerical data, and then use that data to perform logical reasoning and make qualitative judgments.

    **Original Fact:**
    - Question: "{original_question}"
    - Answer: "{original_answer}"

    **Strict Rules for Generating the New Q&A Pair:**
    1.  **Preserve All Inputs in the Question:** The new `open_question` MUST contain ALL quantitative inputs from the original question EXACTLY as they appear: city names, years, climate scenarios (e.g., ssp245), every percentage change with their EXACT signs (+ or -), and the full geographic coordinates for `modify_points`. These are the fixed context and MUST NOT be modified in any way. CRITICAL: Do not change any signs of percentage values - if the original says "16.62%" keep it as "16.62%", if it says "-16.62%" keep it as "-16.62%, and use changed or modified instead of decrease for -".
    2.  **Create an Inferential Question:** The `open_question` should ask for both qualitative assessments (e.g., warming/cooling trends, significance) and, where appropriate, specific quantitative values that require calculation (e.g., the difference between two temperatures), but do not directly include more than two values in the answer from original answer (limit of 2 such values).
    3.  **Create a Multi-claim Qualitative Answer:** The new `open_answer` MUST be a concise conclusion that is DIRECTLY and LOGICALLY dependent on the numbers in the original answer. It must contain at least three distinct, verifiable claims and must NOT contain any numbers itself.
    4.  **Define Magnitude in the Question:** When asking about the degree of change (e.g., negligible, significant), the `open_question` must explicitly define the numerical thresholds for those categories.
    5.  **Incorporate Limited Quantitative Questions:** The `open_question` is now permitted to ask for a specific numerical value (e.g., 'what is the temperature change?'), but **no more than two** such questions should be asked in a single prompt, when you asked for it you must answer it in the `open_answer` as below example.

    ---
    **Example 1:**
    - Original Question: "What is the projected baseline temperature for Karaj in the year 2085 under the ssp126 scenario? If we then introduce an aerosol intervention at specified points [(50.9489, 35.8272)], reducing SO2 by -9.83% and BC by 1.99%, what would be the new local temperature, and what is the difference from the baseline?"
    - Original Answer: "The baseline predicted temperature for Karaj in 2085 under the ssp126 scenario is 16.56°C. With the proposed aerosol intervention (-9.83% SO2 change, 1.99% BC change at [(50.9489, 35.8272)]), the local temperature would become 16.65°C. This represents a local change of 0.09°C."
    - **Your Output (JSON):**
      {{
        "open_question": "For Karaj in 2085 under the ssp126 scenario, an aerosol intervention (-9.83% SO2, 1.99% BC at [(50.9489, 35.8272)]) is proposed. Does this intervention cause a warming or cooling effect compared to the baseline? Quantitatively, what is the predicted local temperature change? Is this change considered negligible (less than 0.2°C), modest (0.2°C to 1.0°C), or significant (over 1.0°C)?",
        "open_answer": "The proposed aerosol intervention results in a local warming effect. The magnitude of this temperature increase is about 0.09°C, it is considered negligible based on the provided thresholds. Overall, the intervention has a minimal impact on the local climate under this scenario."
      }}

    **Example 2:**
    - Original Question: "What was the average temperature in Unity in 1991? How would this compare to the predicted temperature in 2082 if CO2 emissions were adjusted by 49.85% and CH4 emissions by 28.95% under the ssp585 scenario?"
    - Original Answer: "The average temperature in Unity in 1991 was 12.66°C. The predicted temperature for 2082 under the ssp585 scenario with 49.85% CO2 and 28.95% CH4 adjustment would be 15.0°C."
    - **Your Output (JSON):**
      {{
        "open_question": "Consider the city of Unity. What was its average temperature in 1991, and what is the projected temperature change in 2082 under the ssp585 scenario with a 49.85% CO2 and 28.95% CH4 adjustment? Does this represent a minor (< 1°C) or a substantial warming trend (> 1°C) for the city, and would the primary concern of it be related to its effectiveness or potential disruptions?",
        "open_answer": "The historical average temperature in Unity in 1991 was 12.66°C, after the adjustment, the total temperature increases 2.34°C, which indicates a major shift in the local climate. The city is projected to experience a substantial warming trend between the historical and future dates. The primary concern would be related to the potential for disruptions as the value change is significant."
      }}

    **Example 3:**
    - Original Question: "What was the historical average temperature for Bad Kissingen in 1983? Considering the ssp370 scenario, what would be the projected temperature for Bad Kissingen in 2051 if SO2 emissions were modified by -17.82% and Black Carbon by -16.11% at points [(10.0667, 50.2)]?"
    - Original Answer: "The historical average temperature for Bad Kissingen in 1983 was 6.94°C. With the specified SO2 and Black Carbon modifications at [(10.0667, 50.2)], the projected temperature for Bad Kissingen in 2051 under the ssp370 scenario would be 7.21°C. This represents a change of 0.27°C compared to the historical temperature."
    - **Your Output (JSON):**
      {{
      "open_question": "For Bad Kissingen, what is the total projected temperature change from its 1983 historical average to the 2051 forecast under the ssp370 scenario with an aerosol modification of -17.82% SO2 and -16.11% BC at [(10.0667, 50.2)]? Is this overall change considered negligible (less than 0.2°C), modest (0.2°C to 1.0°C), or significant (over 1.0°C) and is that a warming or cooling trend?",
      "open_answer": "The overall magnitude of this temperature change is an increase of 0.27°C, which is considered modest based on the provided thresholds. This indicates a not extreme change to the local climate under the specified scenario and aerosol intervention. There is a warming trend projected for Bad Kissingen between the historical and future dates."
      }}

    **Instructions:**
    - Your response must be a single, valid JSON object.
    - The JSON object must have exactly two keys: "open_question" and "open_answer".
    - Do not include any text, explanations, or markdown formatting outside of the JSON object.

    Now, please generate the open-ended Q&A pair based on the provided fact.
    """

    try:
        response = model.generate_content(prompt)
        # Clean up the response to ensure it's valid JSON
        cleaned_response = response.text.strip().replace("```json", "").replace("```", "").strip()
        upsampled_pair = json.loads(cleaned_response)
        return upsampled_pair
    except (ValueError, json.JSONDecodeError) as e:
        print(f"Error decoding JSON from Gemini response: {e}")
        print(f"Raw response: {response.text}")
        return None
    except Exception as e:
        print(f"An unexpected error occurred: {e}")
        return None

def main():
    """
    Main function to load generated questions and upsample them.
    """
    # Load environment variables from .env file in the current directory
    dotenv_path = os.path.join(os.path.dirname(__file__), '..', '..', '.env')
    load_dotenv(dotenv_path=dotenv_path)
    
    api_key = os.getenv("Google_API_KEY")
    if not api_key:
        raise ValueError("Google_API_KEY not found in environment variables. Please create a .env file in src/Climate/ with your key.")

    genai.configure(api_key=api_key)
    model = genai.GenerativeModel('gemini-2.5-pro')

    try:
        with open("generated_questions.json", 'r', encoding='utf-8') as f:
            original_questions = json.load(f)
    except FileNotFoundError:
        print("Error: generated_questions.json not found. Please run create_open.py first.")
        return

    # Take the first 100 questions and upsample each twice
    questions_to_process = original_questions[:100]
    upsampled_questions = []
    target_count = len(questions_to_process) * 2

    print(f"Upsampling {len(questions_to_process)} questions into {target_count} open-ended Q&A pairs...")

    for i, qa_pair in enumerate(questions_to_process):
        for j in range(2): # Upsample each question twice
            print(f"Processing question {i+1}/{len(questions_to_process)}, attempt {j+1}/2...")
            new_pair = None
            retries = 3
            for attempt in range(retries):
                new_pair = upsample_question(model, qa_pair['question'], qa_pair['answer'])
                if new_pair and 'open_question' in new_pair and 'open_answer' in new_pair:
                    upsampled_questions.append(new_pair)
                    print(f"  -> Success. Total generated: {len(upsampled_questions)}/{target_count}")
                    break 
                else:
                    print(f"  -> Failed to generate a valid pair (attempt {attempt+1}/{retries}), retrying in 30 seconds...")
                    time.sleep(30)
            if new_pair is None:
                print(f"  -> Could not generate a valid pair for question {i+1} after {retries} attempts. Skipping.")


    output_file = "open_questions.json"
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(upsampled_questions, f, ensure_ascii=False, indent=4)

    print(f"Successfully generated and saved {len(upsampled_questions)} open-ended Q&A pairs to {output_file}")

if __name__ == "__main__":
    main()
