"""
GLEAM-AI: Spatio-Temporal Neural Processes for Epidemiological Forecasting

A comprehensive framework for modeling and forecasting epidemiological data using
neural processes with spatial and temporal dependencies.
"""

# Suppress pkg_resources deprecation warning from PyTorch Lightning
# This warning is from a dependency, not our code, and will be fixed in future versions
import warnings
warnings.filterwarnings("ignore", message="pkg_resources is deprecated")

from .models import STNP, ModelRegistry

# High-level functions for easy model loading
def load_pretrained_model(model_name: str = None, config_path: str = None, device: str = "cpu"):
    """
    High-level function for loading pretrained GLEAM-AI models.
    
    This is the simplest way to get started with GLEAM-AI. It automatically:
    - Loads the configuration file
    - Gets model paths from the registry
    - Loads the model checkpoint
    - Loads and caches y statistics
    
    Args:
        model_name: Name of the model in the registry. If None, uses default model.
        config_path: Path to configuration file. If None, uses "config.yaml".
        device: Device to load the model on ("cpu" or "cuda")
        
    Returns:
        STNP: Loaded and configured model instance
        
    Example:
        # Load default model
        model = gleam_ai.load_pretrained_model()
        
        # Load specific model
        model = gleam_ai.load_pretrained_model("v1.0")
        
        # Load with custom config
        model = gleam_ai.load_pretrained_model(config_path="custom_config.yaml")
    """
    return STNP.from_pretrained(model_name, config_path, device)


def get_available_models(config_path: str = None):
    """
    Get information about all available models.
    
    Args:
        config_path: Path to configuration file. If None, uses "config.yaml".
        
    Returns:
        Dictionary mapping model names to their information
    """
    registry = ModelRegistry(config_path)
    return registry.get_available_models()


def get_default_model_name(config_path: str = None):
    """
    Get the name of the default model.
    
    Args:
        config_path: Path to configuration file. If None, uses "config.yaml".
        
    Returns:
        String name of the default model
    """
    registry = ModelRegistry(config_path)
    return registry.get_default_model_name()


__version__ = "1.0.0"

# Main exports
from .active_learning import ActiveLearner, ActiveLearningData, MeanStd, LatentInfoGain
from .data import FeatureDataset, PoolDataset, load_graph_data, get_datasets
from .utils import build_mlp, load_model, crps_normal, save_model_checkpoint
from .training import GLEAMTrainer, main, run_standard_training, run_active_learning_training

__all__ = [
    "STNP",
    "ActiveLearner",
    "ActiveLearningData",
    "MeanStd",
    "LatentInfoGain",
    "FeatureDataset",
    "PoolDataset",
    "load_graph_data",
    "get_datasets",
    "build_mlp",
    "load_model",  # Keep the original utility function
    "crps_normal",
    "save_model_checkpoint",
    "GLEAMTrainer",
    "main",
    "run_standard_training",
    "run_active_learning_training",
    "ModelRegistry", 
    "load_pretrained_model",  # Our new function
    "get_available_models",
    "get_default_model_name"
]
