import os
import json
from dotenv import load_dotenv
import google.generativeai as genai
import time


def generate_topic(client):
    """
    Generates a question topic using the GLEAM-AI influenza forecasting simulator's capabilities.
    Focuses on seasonality, starting date, and R0 parameters as specified by the epidemiology expert.
    """
    import random

    # A detailed description of the new simulation environment for the prompt.
    simulation_description = """
    The GLEAM-AI simulator models influenza epidemic dynamics across US states over multiple months (28+ weeks). It focuses on seasonal flu patterns and provides comprehensive epidemiological forecasting for hospitalization and infection trends.

    **Key Parameters (as specified by epidemiology expert):**
    - R0 (basic reproduction number): varies from 1.75 to 3.0
    - Seasonality level: "no seasonality" (1.0), "moderate seasonality" (0.75), "high seasonality" (0.5)
    - Prior Immunity (via frac_susceptible): The percentage of the population that is not susceptible at the start.
    - Starting date: restricted to specific 2-month range (e.g., September-November for flu season)

    **Available Health Metrics:**
    - Hospital incidence & prevalence
    - Latent incidence & prevalence

    **Key Analysis Targets:**
    - Rate trends over next 4 weeks
    - Timing and height of the seasonal peak
    """

    # New, revised prompt with more examples
    prompt = f"""
    Your task is to generate a single, comprehensive question-answer template in a JSON format for influenza forecasting scenarios. The question should ask for an "overall epidemiological outlook," which naturally covers multiple independent aspects like peak magnitude, timing, and trends.

    **CRITICAL REQUIREMENTS:**
    1.  Output MUST be a single JSON object with "question_template" and "answer_template" keys.
    2.  The question MUST ask for a comprehensive "outlook," "forecast," or "seasonal trajectory" under specific conditions. It must be a single, overarching question, not a list of sub-questions.
    3.  The question must be framed as a forecasting or scenario planning task, NOT a policy intervention task.
    4.  The answer template should simply be a placeholder for the structured simulation results.

    **Simulator Details:**
        {simulation_description}

    **REQUIRED PLACEHOLDERS:**
    - `{{{{r0_value}}}}`: R0 numerical value (e.g., "1.85", "2.4").
    - `{{{{seasonality_level}}}}`: Seasonality description ("no seasonality", "moderate seasonality", "high seasonality").
    - `{{{{starting_date}}}}`: Starting date for the simulation.
    - `{{{{prior_immunity_level}}}}`: The percentage of the population with prior immunity (e.g., "20%", "40%").
    - `{{{{target_states}}}}`: The US states being analyzed.
    - `{{{{target_metric}}}}`: The primary health metric being analyzed (focus on hospitalizations).
    - `{{{{simulation_outlook}}}}`: A structured JSON string containing the complete outlook (peak, timing, trends, etc.).

    ---
    **EXAMPLE TEMPLATES:**

    **Example 1: The Standard Outlook Question**
    ```json
    {{
    "question_template": "For an upcoming influenza season in {{{{target_states}}}} starting around {{{{starting_date}}}}, what is the overall epidemiological outlook for {{{{target_metric}}}} if we assume an R0 of {{{{r0_value}}}}, {{{{seasonality_level}}}}, and a {{{{prior_immunity_level}}}} prior immunity in the population?",
    "answer_template": "**Epidemiological Outlook for {{{{target_metric}}}}:**\\n{{{{simulation_outlook}}}}"
    }}

    **Example 2: The Planning-Focused Question**
    ```json
    {{
    "question_template": "To inform our hospital preparedness strategy for the upcoming flu season in {{{{target_states}}}}, provide a comprehensive forecast for {{{{target_metric}}}}. The scenario assumes a starting date of {{{{starting_date}}}}, an R0 of {{{{r0_value}}}}, {{{{seasonality_level}}}}, and that {{{{prior_immunity_level}}}} of the population has existing immunity.",
    "answer_template": "**Comprehensive Forecast for Hospital Preparedness:**\\n{{{{simulation_outlook}}}}"
    }}
    ```

    **Example 3: The Planning-Focused Question**
    ```json
    {{
    "question_template": "What is the projected seasonal trajectory of {{{{target_metric}}}} in {{{{target_states}}}} given the challenging combination of a high R0 of {{{{r0_value}}}} and a low prior immunity level of {{{{prior_immunity_level}}}}? The season is expected to begin on {{{{starting_date}}}} with {{{{seasonality_level}}}}.",
    "answer_template": "**Seasonal Trajectory Analysis Under High-Risk Conditions:**\\n{{{{simulation_outlook}}}}"
    }}
    ```

    Generate a NEW, unique question template that is different from the examples but follows the same core principles of asking for a comprehensive outlook in a single question.
    """

    try:
        response = client.generate_content(prompt)
        cleaned_response = response.text.strip().replace(
            "```json", "").replace("```", "").strip()
        topic = json.loads(cleaned_response)
        return topic
    except (ValueError, json.JSONDecodeError) as e:
        print(
            f"Error decoding JSON from Gemini response: {e}\nRaw response: {response.text}")
        return None
    except Exception as e:
        print(f"An unexpected error occurred: {e}")
        return None


def main():
    """
    Main function to generate and save question topics for the epidemiology simulator.
    """
    dotenv_path = os.path.join(os.path.dirname(__file__), '..', '..', '.env')
    load_dotenv(dotenv_path=dotenv_path)

    api_key = os.getenv("Google_API_KEY")
    if not api_key:
        raise ValueError(
            "Google_API_KEY not found. Please create a .env file in src/Epidemiology/ with your key.")

    genai.configure(api_key=api_key)
    model = genai.GenerativeModel('gemini-2.5-flash')

    num_topics = 10
    topics = []
    print(f"Generating {num_topics} topics for epidemiology scenarios...")
    while len(topics) < num_topics:
        topic = generate_topic(model)
        if topic and all(k in topic for k in ['question_template', 'answer_template']):
            topics.append(topic)
            print(f"Generated topic {len(topics)}/{num_topics}")
        else:
            print("Failed to generate a valid topic, retrying in 5 seconds...")
            time.sleep(5)

    output_file = "topics.json"
    with open(output_file, 'w', encoding='utf-8') as f:
        json.dump(topics, f, ensure_ascii=False, indent=4)

    print(
        f"Successfully generated and saved {len(topics)} topics to {output_file}")


if __name__ == "__main__":
    main()
