from typing import Protocol, runtime_checkable

from torch import Tensor

from problems.types import Suites


@runtime_checkable
class BudgetLimitedSpace(Protocol):
    @property
    def used_budget(self) -> int:
        raise NotImplementedError()

    @property
    def total_budget(self) -> int:
        raise NotImplementedError()

    def free_to_check_data(self, device: int) -> Tensor:
        raise NotImplementedError()

    def initialize(self):
        raise NotImplementedError()


@runtime_checkable
class EvaluatedSpace(Protocol):
    def __call__(self, data_from_input_space: Tensor, debug_mode: bool = False) -> Tensor:
        raise NotImplementedError()

    def is_goal_reached(self):
        raise NotImplementedError()


@runtime_checkable
class BoundedSpace(Protocol):
    @property
    def upper_bound(self) -> Tensor:
        raise NotImplementedError()

    @property
    def lower_bound(self) -> Tensor:
        raise NotImplementedError()

    def denormalize(self, data: Tensor) -> Tensor:
        """
        map data from (-1,1) to entire space
        """
        raise NotImplementedError()

    def normalize(self, data: Tensor) -> Tensor:
        raise NotImplementedError()


@runtime_checkable
class SamplerSpace(Protocol):
    """
    This class is a space that allows user to sample from it
    """

    def sample_for_optimum_points(
        self, budget: int, num_of_samples: int, device: int = None
    ) -> Tensor:
        """
        sample the space and find n best points (with optimum values)
        :param budget: The number of time to sample from space
        :param num_of_samples: Number of best sample to return
        :param device: In which device to return the samples
        """
        raise NotImplementedError()

    def sample_from_space(
        self, num_samples: int, device: int = None
    ) -> Tensor:
        """
        Sample from Omega space i.e. from the input space of the loss function f
        :param num_samples: number of samples to take from the space
        :param device: Which processor to use
        """
        raise NotImplementedError()

    def best_k_values(self, input_in_space: Tensor, k: int, debug_mode: bool = False) -> Tensor:
        """
        Find the k best points in the space
        :param input_in_space: list of points in the space
        :param k: The num of top result you want to get
        :param debug_mode: Weather or not this is used for debugging
        """
        raise NotImplementedError()

    def best_k_indices(
        self, input_in_space: Tensor, k: int, debug_mode: bool = False
    ) -> Tensor:
        """
        Find the k best points in the space, return the matching indices by ordered
        :param input_in_space: list of points in the space
        :param k: The num of top result you want to get
        :param debug_mode: Weather or not this is used for debugging
        """
        raise NotImplementedError()


@runtime_checkable
class IdentifiableSpace(Protocol):
    @property
    def suite(self) -> Suites:
        raise NotImplementedError()

    @property
    def func_id(self) -> int:
        raise NotImplementedError()

    @property
    def dimension(self) -> int:
        raise NotImplementedError()

    @property
    def func_instance(self) -> int:
        raise NotImplementedError()

    @property
    def name(self) -> str:
        raise NotImplementedError()
