import abc
import math
from abc import ABC
from typing import List, Dict

from aenum import Enum
from matplotlib.figure import Figure

from compute_result.typing import Run, BudgetRun


def is_perfect_log(num: int):
    if num == 0:
        return True
    log_num = math.log(num, 2)
    return round(log_num) == log_num


class AxisTypes(Enum):
    NORMAL = "normal"
    LOG = "log"


class OutputManager(ABC):
    def log_alg_progression(
        self,
        run: Run,
        results: List[float],
        axis_type: AxisTypes = AxisTypes.NORMAL,
        graph_name: str = "",
        plot_name: str = "",
    ):
        self.log_data_progression(
            run, list(enumerate(results)), axis_type, graph_name, plot_name
        )

    @abc.abstractmethod
    def log_normalize_progression(self, results: BudgetRun, plot_name: str, graph_name: str):
        raise NotImplementedError()

    def log_data_progression(
        self,
        run: Run,
        results: BudgetRun,
        axis_type: AxisTypes = AxisTypes.NORMAL,
        graph_name: str = "",
        plot_name: str = "",
    ):
        algorithm, run_name = run
        plot_name = plot_name or f"{algorithm.value}_{run_name}"
        graph_name = graph_name or run_name
        if axis_type == AxisTypes.LOG:
            results = [
                (math.log(budget, 2) + 1 if budget else 0, math.log(value, 2))
                for budget, value in results
                if is_perfect_log(budget)
            ]

        self.log_normalize_progression(results, plot_name, graph_name)

    @abc.abstractmethod
    def print_image(
        self,
        run: Run,
        fig: Figure,
        index: int = None,
        graph_name: str = "",
        plot_name: str = "",
    ):
        raise NotImplementedError()

    @abc.abstractmethod
    def finish(self):
        raise NotImplementedError()

    @abc.abstractmethod
    def print_bars(self, bars: Dict[str, float], std: List[float], graph_name: str = ""):
        raise NotImplementedError()
