import json
import logging
import math
import os
from logging import Logger
from pathlib import Path
from typing import List, Tuple

from pandas import DataFrame

from compute_result.result_store.base import ResultStore
from compute_result.typing import ProblemSpace, Run, RunResult, Point
from problems.types import Suites
from utils.algorithms_data import Algorithms

TEXT_SEP = "&&"
FILE_NAME_SPE = "-"


def create_min_max_file_name(problem_space: ProblemSpace):
    suite, func_id, func_dim, func_instance = problem_space
    return f"{suite.value}{FILE_NAME_SPE}{func_id}{FILE_NAME_SPE}{func_dim}{FILE_NAME_SPE}{func_instance}"


def min_max_problem_from_file_name(min_max_file_name: str) -> ProblemSpace:
    suite, func_id, dim, func_instance = min_max_file_name.split(FILE_NAME_SPE)
    return Suites(suite), int(func_id), int(dim), int(func_instance)


def create_result_file_name(run: Run, problem_space: ProblemSpace):
    alg, run_name = run
    suite, func_id, func_dim, func_instance = problem_space
    return (
        f"{run_name}{FILE_NAME_SPE}{alg.value}{FILE_NAME_SPE}{suite.value}"
        f"{FILE_NAME_SPE}{func_id}{FILE_NAME_SPE}{func_dim}{FILE_NAME_SPE}{func_instance}"
    )


def results_problem_from_file_name(min_max_file_name: str) -> Tuple[Run, ProblemSpace]:
    run_name, algorithm, suite, func_id, dim, func_instance = min_max_file_name.split(
        FILE_NAME_SPE
    )
    return (
        (Algorithms(algorithm), run_name),
        (Suites(suite), int(func_id), int(dim), int(func_instance)),
    )


def min_max_text(min_value: float, max_value: float, min_point: Point, max_point: Point):
    return f"{min_value}{TEXT_SEP}{max_value}{TEXT_SEP}{min_point}{TEXT_SEP}{max_point}"


def result_step_text(*args):
    result_log = TEXT_SEP.join([str(arg) for arg in args])
    return f"{os.linesep}{result_log}"


class FileStorage(ResultStore):
    def __init__(self, data_path: Path, logger: Logger = None):
        self.storage_base_path = data_path
        self.logger = logger or logging.getLogger(__name__)

    @property
    def min_max_path(self):
        min_max_path = self.storage_base_path / "min_max"
        min_max_path.mkdir(parents=True, exist_ok=True)
        return min_max_path

    @property
    def results_path(self):
        results_path = self.storage_base_path / "run_results"
        results_path.mkdir(parents=True, exist_ok=True)
        return results_path

    def storage_exists(self):
        return self.results_path.exists() and self.min_max_path.exists()

    def is_run_exists(self, run: Run, problem_space: ProblemSpace):
        run_file = self.results_path / create_result_file_name(run, problem_space)
        return run_file.exists()

    def update_min(self, problem_space: ProblemSpace, min_value: float, min_point: Point):
        self.logger.info(f"Updating min: {min_value} for {problem_space}")
        curr_min, max_value, curr_min_point, *additional_data = self.min_max_from_space(
            problem_space
        )
        if min_value < curr_min:
            self.logger.info(f"Point {min_point} has value {min_value} is a new min point!")
            min_max_file = self.min_max_path / create_min_max_file_name(problem_space)
            min_max_file.write_text(
                min_max_text(min_value, max_value, min_point, *additional_data)
            )

    def update_max(self, problem_space: ProblemSpace, max_value: float, max_point: Point):
        self.logger.info(f"Updating max: {max_value} for {problem_space}")
        (
            min_value,
            curr_max,
            min_point,
            curr_max_point,
        ) = self.min_max_from_space(problem_space)
        if curr_max < max_value:
            self.logger.info(f"Point {max_point} has value {max_value} is a new max point!")
            min_max_file = self.min_max_path / create_min_max_file_name(problem_space)
            min_max_file.write_text(min_max_text(min_value, max_value, min_point, max_point))

    def min_max_from_space(
        self, problem_space: ProblemSpace
    ) -> Tuple[float, float, Point, Point]:
        min_max_file = self.min_max_path / create_min_max_file_name(problem_space)
        if min_max_file.exists():
            min_value, max_value, min_point, max_point = min_max_file.read_text().split(
                TEXT_SEP
            )
            return (
                float(min_value),
                float(max_value),
                json.loads(min_point),
                json.loads(max_point),
            )
        return math.inf, -math.inf, [], []

    def store_step(
        self,
        run: Run,
        problem_space: ProblemSpace,
        budget: int,
        value: float,
        point: Point,
        algorithm_name: str,
    ):
        run_file = self.results_path / create_result_file_name(run, problem_space)
        with run_file.open("a") as f:
            f.write(result_step_text(budget, value, point, algorithm_name))

    def store_run(self, run: Run, problem_space: ProblemSpace, run_result: RunResult):
        self.logger.info(
            f"Storing run {run} with {len(run_result)} results in problem {problem_space}"
        )
        self.remove_run(run, problem_space)
        run_file = self.results_path / create_result_file_name(run, problem_space)
        run_file.write_text(os.linesep.join([result_step_text(*step) for step in run_result]))

    def run_result(self, run: Run, problem_space: ProblemSpace) -> RunResult:
        run_file = self.results_path / create_result_file_name(run, problem_space)
        run_data = [
            stripped_line.strip().split(TEXT_SEP)
            for line in run_file.read_text().splitlines()
            if (stripped_line := line.strip())
        ]
        return [
            (int(budget), float(value), json.loads(point), alg_name)
            for budget, value, point, alg_name in run_data
        ]

    def remove_run(self, run: Run, problem_space: ProblemSpace):
        run_file = self.results_path / create_result_file_name(run, problem_space)
        if run_file.exists():
            run_file.unlink()

    def list_min_max_problems(self) -> List[ProblemSpace]:
        return [
            min_max_problem_from_file_name(min_max_file.name)
            for min_max_file in self.min_max_path.iterdir()
        ]

    def list_runs(self) -> List[Tuple[Run, ProblemSpace]]:
        return [
            results_problem_from_file_name(result_file.name)
            for result_file in self.results_path.iterdir()
        ]

    def rename_run(self, run: Run, new_run_name: str):
        for problem_file in self.results_path.iterdir():
            (alg, run_name), *_ = results_problem_from_file_name(problem_file.name)
            if (alg, run_name) != run:
                continue
            new_file_name = problem_file.name.replace(run_name, new_run_name)
            problem_file.rename(problem_file.parent / new_file_name)

    def all_algorithms_in_run(self, run: Run, problems: List[ProblemSpace]) -> List[str]:
        names = [
            name for problem in problems for _, _, _, name in self.run_result(run, problem)
        ]
        return list(set(names))

    def __str__(self):
        return f"File storage at {self.storage_base_path}"

    def store_metric(self, metric_name: str, data: DataFrame):
        flattened_list = [
            ",".join(list(row)) for row in data.iterrows()
        ]
        data_to_write = "\n".join(flattened_list)
        test_losses_path = self.storage_base_path / "test_losses"
        test_losses_path.write_text(data_to_write)

    def get_loss_data(self, run: Run, problem: ProblemSpace):
        raise NotImplementedError()
