import logging
import os
from pathlib import Path
from typing import List, Tuple
from unittest.mock import MagicMock

import click
import torch

from compute_result.factory import StoreFactory, StoreTypes
from compute_result.output_manager.base import AxisTypes, OutputManager
from compute_result.result_manager import ResultManager
from compute_result.typing import (
    Run,
    StatisticalAnalysisOptions,
    RunWithPlotName,
    ProblemSpace,
    MetricManipulation,
    AxisAnalysis,
)
from problems.types import Suites, Benchmarks
from run_options import (
    RUN_NAME_AND_ALG_OPTION,
    SUITE_OPTION,
    GRAPH_NAME_OPTION,
    RUN_NAME_OPTION,
    BENCHMARK_OPTION,
    AXIS_TYPE_OPTION,
    STATISTICAL_ANALYSIS_OPTION,
    output_creator,
    RUN_WITH_NAME_OPTION,
    PART_OPTION,
    FUNCTION_NUMBER_OPTION,
    space_instance,
    RESULT_PATH_OPTION,
    FUNCTION_DIM_OPTION,
    DEVICE_OPTION,
    MIN_DIFF_OPTION,
    BINS_OPTION,
    PERCENTILE_OPTION,
    PLOT_IN_LINE_OPTION,
)
from utils.python import timestamp_file_signature


def extract_runs_and_plot_names(
    runs_and_plots: List[RunWithPlotName],
) -> Tuple[List[Run], List[str]]:
    runs_and_plots = (
        list(runs_and_plots)
        if isinstance(runs_and_plots[0], tuple)
        else [runs_and_plots]
    )
    runs = [(alg, run_name) for alg, run_name, _ in runs_and_plots]
    plot_names = [plot_name for _, _, plot_name in runs_and_plots]
    return runs, plot_names


def extract_runs(runs: List[Run]) -> List[Run]:
    return list(runs) if isinstance(runs[0], tuple) else [runs]


@click.group()
def cli():
    pass


MIN_MAX_PATH_OPTION = click.option(
    "-m",
    "--min_max_path",
    type=Path,
    default=Path() / "results" / "min_max",
    help="The location of the min and max result of the space",
)
INCLUDE_UNFINISHED_OPTION = click.option(
    "--include_unfinished",
    type=bool,
    is_flag=True,
    default=False,
    help="Include all runs that are not close to use all budget",
)
PROBLEM_OPTION = lambda multiple=False: click.option(
    "--problem",
    type=click.Tuple([Suites, int, int, int]),
    help="Pick suite, function number ,dimension and instance  ",
    multiple=multiple,
)


@cli.command("converge_graph")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@PROBLEM_OPTION(True)
@click.option(
    "-s",
    "--selected_dims",
    type=click.Tuple([int, int]),
    default=(0, 1),
    help="The dims to print",
)
@click.option("--step_size", type=int, default=None)
@click.option("-p", "--total_points", type=int, default=None)
@click.option("-l", "--legend_size", type=int, default=40)
@PLOT_IN_LINE_OPTION
@output_creator
def print_convergence_graph(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    problem: List[ProblemSpace],
    selected_dims: List[int],
    step_size: int,
    total_points: int,
    legend_size: int,
    num_plot_in_line: int,
    output_type: OutputManager,
):
    run, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.draw_counter_map_of_run(
        run,
        problem,
        plot_names,
        dims=selected_dims,
        step_size=step_size,
        num_of_points=total_points,
        legend_size=legend_size,
        plots_in_row=num_plot_in_line,
    )


@cli.command("compare_by_dim")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@AXIS_TYPE_OPTION
@SUITE_OPTION
@STATISTICAL_ANALYSIS_OPTION
@click.option("--seperated", type=bool, is_flag=True, default=False)
@click.option("--single_plot", type=bool, is_flag=True, default=False)
@FUNCTION_DIM_OPTION()
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    axis_type: AxisTypes,
    suite: Suites,
    analysis: StatisticalAnalysisOptions,
    seperated: bool,
    single_plot: bool,
    func_dim: Tuple[int],
    graph_name: str,
    output_type: OutputManager,
):
    func_dim = list(func_dim) if func_dim else None
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    if seperated:
        result_manager.compare_run_by_dim(
            runs, suite, axis_type, analysis, graph_name, plot_names
        )
    elif single_plot:
        result_manager.plot_run_convergence_by_dim(runs, graph_name, plot_names)
    else:
        result_manager.compare_by_dim_in_canvas(
            runs, analysis, func_dim, graph_name, plot_names
        )


@cli.command("compare")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@AXIS_TYPE_OPTION
@SUITE_OPTION
@STATISTICAL_ANALYSIS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    axis_type: AxisTypes,
    analysis: StatisticalAnalysisOptions,
    suite: Suites,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    if analysis == StatisticalAnalysisOptions.FULL:
        result_manager.full_graph_compare(runs, suite, graph_name, plot_names)
    else:
        result_manager.compare_runs(
            runs, suite, axis_type, analysis, graph_name, plot_names
        )


@cli.command("compare_by_func")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@AXIS_TYPE_OPTION
@SUITE_OPTION
@STATISTICAL_ANALYSIS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    axis_type: AxisTypes,
    suite: Suites,
    analysis: StatisticalAnalysisOptions,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_run_by_func(
        runs, suite, axis_type, analysis, graph_name, plot_names
    )


@cli.command("compare_distance")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@STATISTICAL_ANALYSIS_OPTION
@SUITE_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    analysis: StatisticalAnalysisOptions,
    suite: Suites,
    graph_name: str,
    output_type: OutputManager,
):
    run, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_distance_from_best(run, suite, analysis, graph_name, plot_names)


@cli.command("compare_distance_by_func")
@RUN_NAME_AND_ALG_OPTION()
@RESULT_PATH_OPTION
@STATISTICAL_ANALYSIS_OPTION
@SUITE_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: List[Run],
    results_path: Tuple[Path, StoreTypes],
    analysis: StatisticalAnalysisOptions,
    suite: Suites,
    graph_name: str,
    output_type: OutputManager,
):
    run = list(run) if isinstance(run[0], tuple) else [run]
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_distance_from_best_by_func(run, suite, analysis, graph_name)


@cli.command("compare_all_distance")
@RUN_NAME_AND_ALG_OPTION(False)
@RESULT_PATH_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_alg_result(
    run: Run,
    results_path: Tuple[Path, StoreTypes],
    graph_name: str,
    output_type: OutputManager,
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.print_distance_from_best(run, graph_name)


@cli.command("update")
@click.option(
    "-c",
    "--curr_result",
    default=(Path() / "results", StoreTypes.SQLITE_HIERARCHY),
    type=click.Tuple([Path, StoreTypes]),
    help="The location of the algorithm result",
)
@click.option(
    "-n",
    "--new_result",
    type=click.Tuple([Path, StoreTypes]),
    default=(Path() / "new_results.db", StoreTypes.SQLITE),
    help="The location of the new best result",
)
def update_result_data(curr_result: Path, new_result: Path):
    result_manager = ResultManager(
        StoreFactory.get_store(
            store_type=StoreTypes(curr_result[1]), data_path=curr_result[0]
        ),
        None,
    )
    result_manager.update_from_new_result(
        StoreFactory.get_store(
            store_type=StoreTypes(new_result[1]), data_path=new_result[0]
        ),
    )


@cli.command("print_alg")
@RUN_NAME_AND_ALG_OPTION(True)
@SUITE_OPTION
@RESULT_PATH_OPTION
@GRAPH_NAME_OPTION
@output_creator
def map_alg(
    run: List[Run],
    suite: Suites,
    results_path: Tuple[Path, StoreTypes],
    graph_name: str,
    output_type: OutputManager,
):
    run = list(run) if isinstance(run[0], tuple) else [run]
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.print_run(run, suite, graph_name)


@cli.command("rename_run")
@RUN_NAME_AND_ALG_OPTION(False)
@RUN_NAME_OPTION
@RESULT_PATH_OPTION
def rename_run(run: Run, run_name: str, results_path: Path):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    result_manager.rename_run_name(run, run_name)


@cli.command("fill_benchmark")
@RUN_NAME_AND_ALG_OPTION(False)
@BENCHMARK_OPTION()
@RESULT_PATH_OPTION
@click.option(
    "--stack_indexes",
    is_flag=True,
    type=bool,
    default=False,
    help="Should we stack the index for simpler command (might result in additional runs)",
)
@INCLUDE_UNFINISHED_OPTION
@PART_OPTION
@FUNCTION_NUMBER_OPTION()
@FUNCTION_DIM_OPTION()
def command_for_missing_benchmark(
    run: Run,
    benchmark: Benchmarks,
    results_path: Tuple[Path, StoreTypes],
    stack_indexes: bool,
    include_unfinished: bool,
    part: Tuple[int, int],
    func_num: List[int],
    func_dim: List[int],
):
    parts = list(part) if isinstance(part[0], tuple) else [part]
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    commands = result_manager.command_line_for_missing_benchmark_run(
        run, benchmark, parts, func_num, func_dim, stack_indexes, include_unfinished
    )
    print(" && ".join(commands))


@cli.command("missing_problems")
@RUN_NAME_AND_ALG_OPTION(False)
@BENCHMARK_OPTION()
@RESULT_PATH_OPTION
@INCLUDE_UNFINISHED_OPTION
@PART_OPTION
@FUNCTION_NUMBER_OPTION
def missing_problems_in_benchmark(
    run: Run,
    benchmark: Benchmarks,
    results_path: Tuple[Path, StoreTypes],
    include_unfinished: bool,
    part: Tuple[int, int],
    func_num: List[int],
):
    parts = list(part) if isinstance(part[0], tuple) else [part]
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    missing_problems = result_manager.missing_parts_of_benchmark_run(
        run, benchmark, include_unfinished, parts, func_num
    )
    print(missing_problems)


@cli.command("alg_use")
@RUN_NAME_AND_ALG_OPTION(False)
@RESULT_PATH_OPTION
@output_creator
def how_much_each_algorithm_uses(
    run: Run, results_path: Tuple[Path, StoreTypes], output_type: OutputManager
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.average_alg_progression(run)


@cli.command("find_far_points")
@RUN_NAME_AND_ALG_OPTION(True)
@RESULT_PATH_OPTION
@output_creator
def far_points(
    run: List[Run], results_path: Tuple[Path, StoreTypes], output_type: OutputManager
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.is_point_distance(run[0], run[1], 15)


@cli.command("distance_between_runs")
@RUN_NAME_AND_ALG_OPTION(True)
@RESULT_PATH_OPTION
@STATISTICAL_ANALYSIS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def print_distance_between_runs(
    run: List[Run],
    results_path: Tuple[Path, StoreTypes],
    analysis: StatisticalAnalysisOptions,
    graph_name: str,
    output_type: OutputManager,
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.distance_between_runs(run[0], run[1], analysis, graph_name)


@cli.command("remove_run")
@RUN_NAME_AND_ALG_OPTION(False)
@RESULT_PATH_OPTION
def remove_run(run: Run, results_path: Path):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    result_manager.remove_run(run)


@cli.command("list_runs")
@RESULT_PATH_OPTION
def list_runs(results_path: Path):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    runs = result_manager.list_runs()
    print(os.linesep.join([f"{alg} run - {run}" for alg, run in runs]))


@cli.command("solved_problems")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@MIN_DIFF_OPTION
@output_creator
def print_solved_problems_bar(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    min_diff: float,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.print_bar_problems_solved(runs, min_diff, plot_names)


@cli.command("unsolved_problems")
@RUN_NAME_AND_ALG_OPTION()
@RESULT_PATH_OPTION
@FUNCTION_NUMBER_OPTION(False)
@click.option(
    "--show_func",
    default=False,
    is_flag=True,
    type=bool,
    help="Should print all losing function",
)
@output_creator
def find_where_problems_are_not_solved(
    run: List[Run],
    results_path: Tuple[Path, StoreTypes],
    func_num: int,
    show_func: bool,
    output_type: OutputManager,
):
    run1, run2 = run
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    unsolved_problems = result_manager.where_am_i_worse(
        run1, run2, func_number=func_num
    )
    for (
        func_num,
        (
            problems,
            distance_value,
            worst_problem,
            worst_distance,
            distant_problem,
            distance,
        ),
    ) in unsolved_problems.items():
        print(
            f"Func: {func_num}, has {distance_value} total in the problems {problems if show_func else len(problems)} "
            f"The worst one is {worst_problem} with distance {worst_distance}, Further point problem {distant_problem} "
            f"with distance {distance}"
        )


@cli.command("problems_distance_from_min")
@RUN_NAME_AND_ALG_OPTION(False)
@RESULT_PATH_OPTION
@output_creator
def find_where_problems_are_not_solved(
    run: Run,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    unsolved_problems = result_manager.where_am_i_far_from_best(run)
    for (
        func_num,
        (num_of_problems, distance_value, problems),
    ) in unsolved_problems.items():
        print(
            f"Func: {func_num}, has {distance_value} total in {num_of_problems} problems"
        )


@cli.command("budget_use")
@RUN_NAME_AND_ALG_OPTION()
@RESULT_PATH_OPTION
@output_creator
def how_much_budget_was_used(
    run: List[Run], results_path: Tuple[Path, StoreTypes], output_type: OutputManager
):
    run = extract_runs(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.budget_use_for_runs(run)


@cli.command("between_pest_points")
@RUN_NAME_AND_ALG_OPTION()
@PROBLEM_OPTION()
@RESULT_PATH_OPTION
@output_creator
def log_points_between_runs(
    run: List[Run],
    problem,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
):
    run = extract_runs(run)
    assert len(run) == 2, f"You must chose 2 algorithm to compare, not {len(run)}"
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.plot_difference_between_alg_best_points(run[0], run[1], problem)


@cli.command("final_result")
@RUN_NAME_AND_ALG_OPTION(False)
@PROBLEM_OPTION()
@RESULT_PATH_OPTION
def get_final_run_result(
    run: Run, problem: ProblemSpace, results_path: Tuple[Path, StoreTypes]
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    print(result_manager.final_point_of_run(run, problem))


@cli.command("merge_logs")
@click.option(
    "-l",
    "--logs_path",
    default=Path(),
    type=Path,
    help="The location of the logs",
)
def merge_logs(logs_path: Path):
    total_files = logs_path / f"total_logs_{timestamp_file_signature()}.logs"
    for log_file in logs_path.iterdir():
        with total_files.open("a") as f:
            f.write(log_file.read_text())


@cli.command("early_stopping_graph")
@RUN_NAME_AND_ALG_OPTION(False)
@PROBLEM_OPTION()
@RESULT_PATH_OPTION
def get_final_run_result(
    run: Run, problem: ProblemSpace, results_path: Tuple[Path, StoreTypes]
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        None,
    )
    result_manager.early_stopping_for_run(run, problem)


@cli.command("plot_line")
@space_instance
@output_creator
def plot_between_points(output_type: OutputManager, space, **kwargs):
    mid = torch.tensor(
        [
            -0.6882043094673458,
            0.6174536095299965,
            0.6426181973646479,
            -0.22928467454167478,
            -0.629497450232301,
            -0.7079984596519235,
            0.4240044681834698,
            0.6463972873535093,
            -1.383794612991986,
            -0.3817401591426073,
        ]
    )
    end = torch.tensor(
        [
            -0.7706699706082727,
            -0.03873238859687511,
            -0.13584942842317282,
            -0.2238977984397188,
            -0.2104829286124401,
            -1.1397592111263792,
            1.8692454755974799,
            0.12722436748634625,
            0.36911722905390043,
            -0.7625491742020065,
        ]
    )
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=MagicMock(), data_path=MagicMock()),
        output_type,
    )
    direction = end - mid
    result_manager.plot_vector_line(
        space, mid.tolist(), direction, "between to min", f"Graph {space}", 20, 20
    )


@cli.command("plot_pca")
@space_instance
@output_creator
def plot_pca_contour(output_type: OutputManager, space, **kwargs):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=MagicMock(), data_path=MagicMock()),
        output_type,
    )
    result_manager.contour_pca_space((MagicMock(value="mock"), "Test"), space)


@cli.command("plot_metric")
@click.option("-m", "--metric", type=str)
@RESULT_PATH_OPTION
@output_creator
def get_final_run_result(
    metric: str,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
    **kwargs,
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.print_metric(metric, metric)


@cli.command("compare_metric")
@click.option("-m", "--metric", type=str)
@click.option(
    "--manipulation", type=MetricManipulation, default=MetricManipulation.NONE
)
@RESULT_PATH_OPTION
@output_creator
def get_final_run_result(
    metric: str,
    manipulation: MetricManipulation,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
    **kwargs,
):
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_metric_by_budget(metric, metric, manipulation)


@cli.command("compare_dim_effect")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@SUITE_OPTION
@STATISTICAL_ANALYSIS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_dim_effect(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    suite: Suites,
    analysis: StatisticalAnalysisOptions,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_best_by_dim(runs, suite, analysis, graph_name, plot_names)


@cli.command("compare_step_distance")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@FUNCTION_NUMBER_OPTION(False)
@FUNCTION_DIM_OPTION(False)
@STATISTICAL_ANALYSIS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def compare_dim_effect(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    func_num: int,
    func_dim: int,
    analysis: StatisticalAnalysisOptions,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.plot_steps_len(
        runs, analysis, func_dim, func_num, graph_name, plot_names
    )


@cli.command("plot_terrain_difficulties")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@FUNCTION_NUMBER_OPTION(False)
@FUNCTION_DIM_OPTION(False)
@STATISTICAL_ANALYSIS_OPTION
@click.option(
    "--terrain_analysis",
    type=StatisticalAnalysisOptions,
    default=StatisticalAnalysisOptions.QUANTILE_90,
    help="How to measure the values between steps",
)
@GRAPH_NAME_OPTION
@output_creator
@DEVICE_OPTION
def compare_dim_effect(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    func_num: int,
    func_dim: int,
    analysis: StatisticalAnalysisOptions,
    terrain_analysis: StatisticalAnalysisOptions,
    graph_name: str,
    output_type: OutputManager,
    device: int,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.plot_steps_terrains(
        runs,
        func_num,
        func_dim,
        analysis,
        terrain_analysis,
        graph_name,
        plot_names,
        device,
    )


@cli.command("moving_vs_shrinking")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@STATISTICAL_ANALYSIS_OPTION
@click.option(
    "--log_base",
    default=Path(),
    type=Path,
)
@BINS_OPTION
@GRAPH_NAME_OPTION
@output_creator
@DEVICE_OPTION
def compare_dim_effect(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    analysis: StatisticalAnalysisOptions,
    log_base: Path,
    bins: int,
    graph_name: str,
    output_type: OutputManager,
    device: int,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.shrink_vs_move_plot(
        runs, log_base, bins, analysis, graph_name, plot_names, device
    )


@cli.command("problem_solved_by_dim")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@MIN_DIFF_OPTION
@click.option("--bar_width", type=float, default=0.2)
@GRAPH_NAME_OPTION
@output_creator
def compare_dim_effect(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    min_diff: float,
    bar_width: float,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_dim_solving_bar(
        runs, min_diff, bar_width, graph_name=graph_name, plot_names=plot_names
    )


@cli.command("error_hist")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@BINS_OPTION
@GRAPH_NAME_OPTION
@output_creator
def create_runs_hist(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    bins: int,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.runs_distribution(
        runs, bins, graph_name=graph_name, plot_names=plot_names
    )


@cli.command("epoch_dist")
@RUN_WITH_NAME_OPTION()
@RESULT_PATH_OPTION
@BINS_OPTION
@PERCENTILE_OPTION
@GRAPH_NAME_OPTION
@output_creator
def create_runs_hist(
    run: List[RunWithPlotName],
    results_path: Tuple[Path, StoreTypes],
    bins: int,
    percentile: float,
    graph_name: str,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    dist_desc = result_manager.budget_progression_distributions(
        runs, bins, percentile, graph_name=graph_name, plot_names=plot_names
    )
    for desc in dist_desc:
        print(desc)


@cli.command("dist_desc")
@RUN_WITH_NAME_OPTION()
@MIN_DIFF_OPTION
@BINS_OPTION
@GRAPH_NAME_OPTION
@RESULT_PATH_OPTION
@output_creator
def create_runs_hist(
    run: List[RunWithPlotName],
    min_diff: float,
    bins: int,
    graph_name: str,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    dist_desc = result_manager.distributions_description_of_runs(
        runs, min_diff, bins, plot_names, graph_name
    )
    for desc in dist_desc:
        print(desc)


@cli.command("axis_compare")
@RUN_WITH_NAME_OPTION()
@click.option(
    "--axis",
    multiple=True,
    type=click.Tuple([AxisAnalysis, AxisAnalysis]),
    default=[(AxisAnalysis.MEAN, AxisAnalysis.STD)],
)
@PERCENTILE_OPTION
@GRAPH_NAME_OPTION
@RESULT_PATH_OPTION
@MIN_DIFF_OPTION
@PLOT_IN_LINE_OPTION
@output_creator
def compare_axis(
    run: List[RunWithPlotName],
    axis: List[Tuple[AxisAnalysis, AxisAnalysis]],
    percentile: float,
    graph_name: str,
    results_path: Tuple[Path, StoreTypes],
    min_diff: float,
    num_plot_in_line: int,
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_runs_on_axis(
        runs,
        axis,
        plot_names,
        graph_name,
        percentile=percentile,
        offset=min_diff,
        plots_in_rows=num_plot_in_line,
    )


@cli.command("progress_consistency")
@RUN_WITH_NAME_OPTION()
@GRAPH_NAME_OPTION
@RESULT_PATH_OPTION
@output_creator
def compare_axis(
    run: List[RunWithPlotName],
    graph_name: str,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_steps_per_budget(runs, graph_name, plot_names)


@cli.command("progress_consistency_overtime")
@RUN_WITH_NAME_OPTION()
@GRAPH_NAME_OPTION
@RESULT_PATH_OPTION
@output_creator
def compare_axis(
    run: List[RunWithPlotName],
    graph_name: str,
    results_path: Tuple[Path, StoreTypes],
    output_type: OutputManager,
):
    runs, plot_names = extract_runs_and_plot_names(run)
    result_manager = ResultManager(
        StoreFactory.get_store(store_type=results_path[1], data_path=results_path[0]),
        output_type,
    )
    result_manager.compare_steps_per_budget_overtime(runs, graph_name, plot_names)


if __name__ == "__main__":
    logging.basicConfig(format="%(levelname)s - %(message)s", level=logging.INFO)
    cli()
