import math

from algorithms.convergence_algorithms.protocols import AlgorithmOnEnvironment
from algorithms.stopping_condition.base import AlgorithmStopCondition


class EnvGoalIsReached(AlgorithmStopCondition):
    REASON = "{alg} reached the goal in {env} - {best_point}"

    def should_stop(self, alg: AlgorithmOnEnvironment, **kwargs) -> bool:
        return alg.environment.is_goal_reached()


class NoMoreBudget(AlgorithmStopCondition):
    REASON = "{alg} has no more budget"

    def should_stop(self, alg: AlgorithmOnEnvironment, **kwargs) -> bool:
        return alg.environment.used_budget > alg.environment.total_budget


class LogarithmicOptimization(AlgorithmStopCondition):
    REASON = "{alg} improved logarithmically for too long"

    def __init__(self, num_of_iteration_for_log_improvement: int):
        self.num_of_iteration_for_log_improvement = num_of_iteration_for_log_improvement
        self.last_point_value = None

    def should_stop(self, alg: AlgorithmOnEnvironment, **kwargs) -> bool:
        best_point_value = alg.environment(alg.best_point_until_now, debug_mode=True)
        if self.last_point_value and math.log(self.last_point_value - best_point_value) < 1:
            return False  # TODO


class EarlyBudgetStop(AlgorithmStopCondition):
    REASON = "{alg} reached its early budget"

    def __init__(self, early_budget: int):
        self.early_budget = early_budget

    def should_stop(self, alg: AlgorithmOnEnvironment, **kwargs) -> bool:
        return alg.environment.used_budget >= self.early_budget
