import logging
from logging import Logger
from pathlib import Path
from typing import List, Tuple

from pandas import DataFrame

from compute_result.result_store.base import ResultStore
from compute_result.result_store.sqlite_utils import (
    list_run_result,
    TABLE_NAME_BREAK,
    is_table_exist,
    min_max_from_space,
    update_min,
    update_max,
    remove_table,
    store_step,
    store_run,
    list_min_max_problems,
    list_tables,
    find_algorithms_in_table,
    last_record_from_run_result,
    store_df,
    extract_df,
)
from compute_result.typing import Run, ProblemSpace, RunResult, Point, ResultRow
from problems.types import Suites
from utils.algorithms_data import Algorithms


def find_path_to_run(base_path: Path, run: Run):
    algorithm, run_name = run
    return base_path / algorithm.value / f"{run_name}.db"


def construct_table_name(problem: ProblemSpace) -> str:
    suite, func_num, func_id, func_instance = problem
    return (
        f"{suite.value}{TABLE_NAME_BREAK}{func_num}{TABLE_NAME_BREAK}"
        f"{func_id}{TABLE_NAME_BREAK}{func_instance}"
    )


def deconstruct_problem_from_table_name(table_name: str) -> ProblemSpace:
    suite, func_num, func_id, func_instance = table_name.split(TABLE_NAME_BREAK)
    return Suites(suite), int(func_num), int(func_id), int(func_instance)


def find_min_max_db(base_path: Path) -> Path:
    return base_path / "min_max.db"


class SQLHierarchyStorage(ResultStore):
    def __init__(self, data_path: Path, logger: Logger = None):
        self.base_path = data_path
        self.logger = logger or logging.getLogger(__name__)

    def storage_exists(self):
        return self.base_path.exists() and find_min_max_db(self.base_path).exists()

    def run_result(self, run: Run, problem_space: ProblemSpace) -> RunResult:
        db_path = find_path_to_run(self.base_path, run)
        table_name = construct_table_name(problem_space)
        return list_run_result(db_path, table_name)

    def is_run_exists(self, run: Run, problem_space: ProblemSpace):
        run_path = find_path_to_run(self.base_path, run)
        if not run_path.exists():
            return False
        return is_table_exist(run_path, construct_table_name(problem_space))

    def min_max_from_space(
        self, problem_space: ProblemSpace
    ) -> Tuple[float, float, Point, Point]:
        return min_max_from_space(find_min_max_db(self.base_path), problem_space)

    def update_min(self, problem_space: ProblemSpace, min_value: float, min_point: Point):
        return update_min(
            find_min_max_db(self.base_path), self.logger, problem_space, min_value, min_point
        )

    def update_max(self, problem_space: ProblemSpace, max_value: float, max_point: Point):
        return update_max(
            find_min_max_db(self.base_path), self.logger, problem_space, max_value, max_point
        )

    def remove_run(self, run: Run, problem_space: ProblemSpace):
        run_path = find_path_to_run(self.base_path, run)
        if not run_path.exists():
            return
        table_name = construct_table_name(problem_space)
        return remove_table(run_path, self.logger, table_name)

    def store_step(
        self,
        run: Run,
        problem_space: ProblemSpace,
        budget: int,
        value: float,
        point: Point,
        algorithm_name: str,
    ):
        run_path = find_path_to_run(self.base_path, run)
        table_name = construct_table_name(problem_space)
        return store_step(run_path, table_name, budget, value, point, algorithm_name)

    def store_run(self, run: Run, problem_space: ProblemSpace, run_result: RunResult):
        self.logger.info(f"Storing run {run}, {problem_space} {len(run_result)}")
        run_result_df = self._filter_multiple_budget(run_result, return_df=True)
        self.logger.info(f"Filter {run}, {problem_space} to {len(run_result_df)}")

        run_path = find_path_to_run(self.base_path, run)
        table_name = construct_table_name(problem_space)
        return store_run(run_path, table_name, run_result_df)

    def list_min_max_problems(self) -> List[ProblemSpace]:
        min_max_path = find_min_max_db(self.base_path)
        return list_min_max_problems(min_max_path)

    def list_runs(self) -> List[Tuple[Run, ProblemSpace]]:
        return [
            (
                (Algorithms(alg_path.name), run_path.stem),
                deconstruct_problem_from_table_name(table_name),
            )
            for alg_path in self.base_path.iterdir()
            if alg_path.is_dir()
            for run_path in alg_path.iterdir()
            for table_name in list_tables(run_path)
        ]

    def rename_run(self, run: Run, new_run_name: str):
        run_path = find_path_to_run(self.base_path, run)
        run_path.rename(f"{new_run_name}.db")

    def all_algorithms_in_run(self, run: Run, problems: List[ProblemSpace]) -> List[str]:
        return find_algorithms_in_table(
            find_path_to_run(self.base_path, run),
            run,
            lambda table_name: (run, deconstruct_problem_from_table_name(table_name)),
        )

    def problem_final_value_for_run(self, run: Run, problem: ProblemSpace) -> ResultRow:
        return last_record_from_run_result(
            find_path_to_run(self.base_path, run), construct_table_name(problem)
        )

    def store_metric(self, metric_name: str, data: DataFrame):
        store_df(data, self.base_path / "metrics.sqlite", metric_name)

    def get_metric(self, metric_name: str) -> DataFrame:
        return extract_df(self.base_path / "metrics.sqlite", metric_name)

    def get_metrics_types(self, metric_initial: str) -> List[str]:
        metric_path = self.base_path / "metrics.sqlite"
        tables = list_tables(metric_path)
        return [table for table in tables if table.startswith(metric_initial)]
