import importlib
import inspect
import os
import pkgutil
from abc import ABC
from types import ModuleType
from typing import List, Dict, Any, TypeVar, Tuple, Type, Callable

TBase = TypeVar("TBase")


class Configurable(ABC):
    def __str__(self):
        return (
            f"The class {self.__class__.__name__} is using "
            f"{os.linesep.join([f'{param}={value}' for param, value in self.__dict__.items()])}"
        )

    @classmethod
    def class_default_values(cls) -> Dict[str, Callable]:
        base_values = cls.object_default_values()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                base_values = klass.class_default_values() | base_values
        return base_values

    @classmethod
    def object_default_values(cls) -> Dict[str, Callable]:
        return {}

    @classmethod
    def ignored_params(cls) -> List[str]:
        ignored = cls._ignored_params()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                ignored += klass.ignored_params()
        return ignored

    @classmethod
    def _ignored_params(cls) -> List[str]:
        return [
            "self",
            "callback_handlers",
            "stopping_conditions",
            "space",
            "env",
            "environment",
        ]

    @classmethod
    def default_types(cls) -> Dict[str, type]:
        default_types = cls._default_types()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                default_types = klass.default_types() | default_types
        return default_types

    @classmethod
    def _default_types(cls) -> Dict[str, type]:
        return {}

    @classmethod
    def additional_configs(cls) -> Dict[str, Dict[str, Any]]:
        additional_config = cls._additional_configs()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                additional_config = klass.additional_configs() | additional_config
        return additional_config

    @classmethod
    def _additional_configs(cls) -> Dict[str, Dict[str, Any]]:
        return {}

    @classmethod
    def manipulate_parameters(cls) -> Dict[str, Callable]:
        parameter_manipulation = cls._manipulate_parameters()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                parameter_manipulation = klass.manipulate_parameters() | parameter_manipulation
        return parameter_manipulation

    @classmethod
    def _manipulate_parameters(cls) -> Dict[str, Callable]:
        return {}

    @classmethod
    def additional_parameters(cls) -> Dict[str, Any]:
        parameters = cls._additional_parameters()
        for klass in cls.__bases__:
            if issubclass(klass, Configurable):
                parameters = klass.additional_parameters() | parameters
        return parameters

    @classmethod
    def _additional_parameters(cls) -> Dict[str, Any]:
        return {}


def need_params_for_signature(obj: Any) -> bool:
    return inspect.isclass(obj) and issubclass(obj, Configurable)


def find_subclasses(module: ModuleType, base_class: TBase) -> List[TBase]:
    subclasses = []

    # Iterate over all modules in the package
    for loader, name, is_pkg in pkgutil.walk_packages(module.__path__):
        # Import the module
        full_name = module.__name__ + "." + name
        sub_module = importlib.import_module(full_name)
        if is_pkg:
            subclasses.extend(find_subclasses(sub_module, base_class))
        else:
            # Iterate over all objects in the module
            for obj_name, obj in inspect.getmembers(sub_module):
                # Check if the object is a class, is a subclass of the base class,
                # and is not the base class itself
                if (
                    inspect.isclass(obj)
                    and issubclass(obj, base_class)
                    and not inspect.isabstract(obj)
                ):
                    subclasses.append(obj)
    return subclasses


def find_class_by_name(module: ModuleType, class_name: str) -> type:
    for loader, name, is_pkg in pkgutil.walk_packages(module.__path__):
        full_name = module.__name__ + "." + name
        sub_module = importlib.import_module(full_name)
        if is_pkg:
            if klass := find_class_by_name(sub_module, class_name):
                return klass
        else:
            # Iterate over all objects in the module
            for obj_name, obj in inspect.getmembers(sub_module):
                # Check if the object is a class, is a subclass of the base class,
                # and is not the base class itself
                if (
                    inspect.isclass(obj)
                    and not inspect.isabstract(obj)
                    and class_name == obj_name
                ):
                    return obj
    return None


def get_full_signature_parameters(
    signature: type, base_class: type
) -> Dict[str, inspect.Parameter]:
    parameters = {}
    for parent_class in getattr(signature, "__bases__", []):
        if issubclass(parent_class, base_class):
            parameters.update(get_full_signature_parameters(parent_class, base_class))
    parameters.update(inspect.signature(signature).parameters)
    return parameters


def get_signature_parameters(
    signature: type, signature_class: Type[Configurable], base_class: type
) -> Dict[str, Tuple[type, Any]]:
    parameters = {}
    default_types = signature_class.default_types()
    for param, value in get_full_signature_parameters(signature, base_class).items():
        if (
            value.kind == inspect.Parameter.VAR_POSITIONAL
            or value.kind == inspect.Parameter.VAR_KEYWORD
        ):
            continue
        default_type = default_types.get(param, None) or value.annotation
        if need_params_for_signature(default_type):
            klass_parameters = get_signature_parameters(
                default_type, signature_class, Configurable
            )
            parameters.update(klass_parameters)
        if value.default == inspect.Parameter.empty:
            parameters[param] = (value.annotation, None)
        else:
            parameters[param] = (
                type(value.default) if value.default else default_type,
                value.default,
            )
    return parameters
