import threading
import traceback

from ros_action.utils import exec_safe, merge_dicts, get_robot_action_list
from ros_action.constants import RobotExecConstants

def handle_exec_robot_code(robot_api, code_str):
    """Handle all things related to executing robot subtask

    This is the main function that gets called by the main function. 
    It handles:
    - Start executing a subtask in a thread
    - Stop the current subtask

    Args:
        robot_api (RobotActionAPI): A singleton instance of a robot's action api
        observation_model (Observation): A singleton instance of the observation model
        DAG (DAG): A DAG representing the current recipe
        logger (Logger): A singleton instance of the logger
    """
    if robot_is_idle(robot_api):
        t = threading.Thread(
            target=_start_a_subtask_in_thread, args=(robot_api, code_str))
        t.start()


def _start_a_subtask_in_thread(robot_api, code_str):
    """Start executing a subtask
    
    It follows this workflow:
    - Assigns the first of the robot's subtask queue to be the robot's current task
    - Generate code for the robot
    - Execute the generated code

    Args:
        robot_api (RobotActionAPI): A singleton instance of a robot's action api
        observation_model (Observation): A singleton instance of the observation model
        DAG (DAG): A DAG representing the current recipe
        logger (Logger): A singleton instance of the logger
    """
    robot_api.reset()

    robot_api.curr_code = code_str
    robot_name = robot_api.robot_name

    print(f"\n************ executing {robot_name}'s code *******************")
    print(code_str)
    exec_robot_code(robot_api)
    print(f"========{robot_name} finished ===========")


def exec_robot_code(robot_api):
    """Helper function to actually execute the code in robot_api.curr_code

    Pre-condition:
        - The robot_api.curr_code is set with the code to execute

    Post-condition:
        - In the robot_api,
            - robot_api.curr_code gets clear out (reset to "")
            - If the interruption flag gets triggered during this subtask,
                update past_completed_action_list to keep track of what action has been executed
            - Reset the interruption flag
            - Reset the curr_completed_action_list

    Args:
        robot_api (RobotActionAPI): A singleton instance of a robot's action api
    """
    # Get the environment constant that the code has access to
    constants = {
        c.name: c.value for c in RobotExecConstants
    }
    
    # Get the robot's action that the the code has access to
    robot_api_fn = {
        f: getattr(robot_api, f) for f in get_robot_action_list(robot_api)
    }

    all_imported_vars = merge_dicts([constants, robot_api_fn])

    try:
        exec_safe(robot_api.curr_code, gvars=all_imported_vars, lvars={})
    except Exception:
        traceback.print_exc()

    # After everything finishes executing, we can reset the robot_api
    robot_api.reset()

def robot_is_idle(robot_api):
    return robot_api.curr_code == ""