import time
import threading
import random
from copy import deepcopy

import rospy
from APRICOT_main.srv import HalPick, HalPlace, HalMove

RUNNING = -1
SUCCESS = 1
FAILED = 0

# If we are not using ros action service, 
# sleep for this long to pretend the robot is doing the subtask
MIN_SLEEP_TIME = 1
MAX_SLEEP_TIME = 5


class RobotActionAPI:
    """Base class of action APIs for all robots"""

    def __new__(cls):
        """Create a singleton instance of the RobotActionAPI model.

        Returns:
            RobotActionAPI: The singleton instance of the RobotActionAPI model.
        """
        if not hasattr(cls, 'instance'):
            cls.instance = super(RobotActionAPI, cls).__new__(cls)
        return cls.instance

    def _initialize(self, use_ros_service):
        """Initialize the RobotActionAPI singleton model.

        Args:
            use_ros_service (bool): True if we are using ROS action service to control the robots. 
        """
        self.stop_executing = threading.Event()  # For flagging and interrupting action execution
        self.curr_completed_action_list = []
        self.past_completed_action_list = []
        self.curr_code = ""

        self._use_ros_service = use_ros_service

    def stop(self):
        """Turn on the flag to interrupt robot subtask execution.

        After this flag gets triggered, all subsequent action defined in self.curr_code will 
        be skipped. 
        """
        self.stop_executing.set()

    def reset(self):
        """Reset the action api. 

        Reset the interruption flag. If the previous subtask has triggered the interrupt flag, 
        deepcopy what action have been completed in the previous subtask, so that the current 
        subtask's code generation can take into account of what action has happened. 
        """
        if self.stop_executing.is_set():
            # Previous subtask has experienced interruption
            if self.past_completed_action_list == []:
                # Keep track of what action has happened in previous subtask
                self.past_completed_action_list = deepcopy(self.curr_completed_action_list)
            
            # Reset the interruption flag
            self.stop_executing.clear()
        
        self.curr_code = ""
        self.curr_completed_action_list = []

    def safe_append_completed_action(self, fn_str):
        """Safely keep track of completed action.

        Only keep track of the action function if the interruption flag is not set. 

        Args:
            fn_str (str) - String representing the action function with its arguments that have
            just been executed.
        """
        if not self.stop_executing.is_set():
            if fn_str not in self.curr_completed_action_list:
                self.curr_completed_action_list.append(fn_str)
    


def service_client_template(func_args, 
                            func_name, server_name,
                            service_msg):
    """Send request to robot's action service server

    Given a robot action function, send an action service request, which 
    has type server_msg, with the arguments (func_args) via ROS
    to the action server called server_name

    Paramters:
        func_args (tuple): Arguments for the function that we are making request for
        func_name (str): Name of the function
        server_name (str): Name of the ROS action service server to send request to
        service_msg: Service message used to send request
    """
    print(f"Requesting {func_name}({func_args}) at server={server_name}")

    rospy.wait_for_service(server_name)

    try:
        action_request = rospy.ServiceProxy(server_name, service_msg)

        # Send the request differently based on the number of arguments
        if len(func_args) == 0:
            response = action_request()
        elif len(func_args) == 1:
            response = action_request(func_args[0])
        elif len(func_args) == 2:
            response = action_request(func_args[0], func_args[1])
        elif len(func_args) == 3:
            response = action_request(func_args[0], func_args[1], func_args[2])
        else:
            raise NotImplementedError("Currently, only support functions with at most 3 parameters")

        return response.status
    except rospy.ServiceException as e:
        print("Service call failed: %s" % e)


def function_wrapper(stop_exec_event, use_ros_service, func, *args):
    """A wrapper to wait for the robot to finish its action

    Args:
        stop_exec_event (Threading.Event): An interruption flag on whether to stop executing the actions
        use_ros_service (bool): True if we are using ROS action service to control the robots. 
        func: A python function corresponding to a robot's low-level action
        args: Argument to this function
    """
    if not stop_exec_event.is_set():
        if use_ros_service:
            # Actually sending action service request to robots via ROS
            if args[0] == None:
                # When there is no argument expected for the function
                # args is a Tuple, so we get the first element and check if it is none
                # see dora_stir() for an example
                func_status = func()
            else:
                func_status = func(args)
            # Keep checking the status of the action until it finishes running
            while func_status == RUNNING:
                time.sleep(0.1)
                func_status = func(args)  # Update the function status again
                print(f"Current status: {func_status}")

            return func_status
        else:
            # Pretending to run the action by sleeping
            pretend_to_run_time = random.randint(MIN_SLEEP_TIME, MAX_SLEEP_TIME)
            print(f"     - pretending to run for {pretend_to_run_time} seconds")
            time.sleep(pretend_to_run_time)

            return SUCCESS
    else:
        # Because the interruption flag is set, we skip this function
        print("[[[[SKIPPING]]]]")
        return FAILED



class HalActionAPI(RobotActionAPI):
    """A singleton model of the RobotActionAPI for the robot HAL"""
    def initialize(self, use_ros_service):
        """Initialize the HALActionAPI singleton model. 

        Set self.robot_name. Mainly using the self._initialize function in parent class (RobotActionAPI)

        Args:
            use_ros_service (bool): True if we are using ROS action service to control the robots. 
        """
        self.robot_name = "HAL"
        self._initialize(use_ros_service)


    """Low level action workflow

    To send action request via ROS service, we define the low level action following these rules
    - def action_name(self, ...) is what gets called when we execute the code generated by the LLM
    - def action_name(self, ...) wraps two functions using this layering
        def action_name(self, ...) contains
            function_wrapper(...), which contains
                service_client_template
    - function_wrapper contains the logic of how the action gets executed
        - If interruption flag (self.stop_executing) is not set AND we are using ROS_action_service,
            we actually call service_client_template to send action service request to robot via ROS
        - If interruption flag is not set AND we are not using ROS_action_service,
            we will pretend to be the robot taking time to do this action by sleeping
        - If interruption flag is set,
            we skip this action immediately
    - service_client_template contains the logic to send action service request to actual robot via ROS
    

    Low level action function should always return an action status defined at the top of this file:
        RUNNING = -1
        SUCCESS = 1
        FAILED = 0

    Tips for adding new low level action:
    - You should follow the same format as existing low level action
    - The name of the action function matters! This is what the LLM will see to generate code for a subtask
        - The name of the action function should not contain the robot's name
        - Multiple robots can have the same name for an action as long as they are semantically performing similar action
            (see pick_up_item in HalActionAPI v.s. pick_up_item in DoraActionAPI)
    """

    def go_to(self, target_loc):
        print(f"HAL go to {target_loc}")
        self.safe_append_completed_action(f"go_to({target_loc})")

        def _go_to(args):
            service_client_template(args,
                                    func_name="hal_go_to", server_name="HalMove",
                                    service_msg=HalMove)

        return function_wrapper(self.stop_executing, self._use_ros_service,
                                _go_to, target_loc)
 

    def pick_up_item(self, target_item):
        print(f"HAL pick up {target_item}")
        self.safe_append_completed_action(f"pick_up_item({target_item})")

        def _pick_up_item(args):
            service_client_template(args,
                                    func_name="hal_pick_up_item", server_name="HalPick",
                                    service_msg=HalPick)

        return function_wrapper(self.stop_executing, self._use_ros_service,
                                _pick_up_item, target_item)


    def place_item_at(self, target_x, target_y, target_z):
        print(f"HAL place item at {(target_x, target_y, target_z)}")
        self.safe_append_completed_action(f"place_item_at({(target_x, target_y, target_z)})")

        def _place_item_at(args):
            service_client_template(args,
                                    func_name="hal_place_item_at", server_name="HalPlace",
                                    service_msg=HalPlace)

        return function_wrapper(self.stop_executing, self._use_ros_service,
                                _place_item_at, target_x, target_y, target_z)
    


