## YAML CONFIG FILES ## 
import yaml

""" Wrapper for config dict to access it using dot notation
"""
class DictConfig(dict):

    def __getattr__(self, name):
        value = self[name]
        if isinstance(value, dict):
            value = DictConfig(value)
        return value

    def get_dict(self):
        return super()


"""" Recursively unpacks the includes in a config
"""
def unpack_config_rec(config):
    
    # Unpack includes
    if isinstance(config, str) and config.split(":")[0] == "include":
        config = yaml.safe_load(open(config.split(":")[1],"r"))
    
    if isinstance(config, dict):
        for field in config:
            config[field] = unpack_config_rec(config[field])

    return config



"""" Recursively update the entries of the new_config dict wit the entries of the config dict
"""
def update_config_rec(new_config, config):
    
    if isinstance(config, dict):
        # Force new fields in new_config to update with fields from config
        if not isinstance(new_config,dict):
            print("Created new subdict")
            new_config = {}
        for field in config:
            if not field in new_config:
                # print(f"Creating new field {field}")
                new_config[field] = {}
            new_config[field] = update_config_rec(new_config[field], config[field])
    else:
        # Assign leaf
        new_config = config

    return new_config


""" Update values in default_config from config, adding the missing keys if needed. 
    If config is None, the default config is returned (with all the includes unpacked).
    Configs can also be a path to the config file.
"""
def update_config(default_config, config = None):

    if isinstance(default_config, str):
        default_config = yaml.safe_load(open(default_config,"r"))

    # If no config is provided, we iterate using the same config to make sure that the includes
    # are unpacked
    config = default_config if config is None else config

    if isinstance(config, str):
        config = yaml.safe_load(open(config,"r"))

    # Go down the tree to unpack the includes
    unpacked_default_config = unpack_config_rec(default_config)
    unpacked_config = unpack_config_rec(config)

    return DictConfig(update_config_rec(unpacked_default_config, unpacked_config))



## COMMAND LINE KWARGS ##
import argparse

""" Parse command line kwargs to dict
"""
class ParseKwargs(argparse.Action):
    def __call__(self, parser, namespace, values, option_string=None):
        setattr(namespace, self.dest, dict())
        for value in values:
            key, value = value.split('=')
            getattr(namespace, self.dest)[key] = value


""" Convert string flags to adequate dtype
"""
def convert_to_dtype(value):

    value = value.strip()

    # Catch list
    if value[0] == "[" and value[-1] == "]":
        value = [convert_to_dtype(v) for v in value[1:-1].split(",")]
    # Catch None
    elif value == "null" or value == "None" or value == "none":
        value = None
    # Catch bool
    elif value == "true" or value == "True":
        value = True
    elif value == "false" or value == "False":
        value = False
    # Catch int
    elif value.isdigit() or value.replace("-","").isdigit():
        value = int(value)
    # Catch float
    else:
        try:
            value = float(value)
        except Exception:   
            pass
    return value
            
""" Parse flat kwargs dict with dot notation keys to nested dict
    TO DO: parse lists
"""
def config_from_kwargs(kwargs):
    
    config = {}
    
    if kwargs is not None:
        for key, value in kwargs.items():
            
            # Froms string to aproppriate dtype
            value = convert_to_dtype(value)
            
            # Go iteratively to the leaf
            cur = config
            for sub_key in key.split(".")[:-1]:
                if not sub_key in cur:
                    cur[sub_key] = {}
                cur = cur[sub_key]
            cur[key.split(".")[-1]] = value

    return DictConfig(config)
