from __future__ import absolute_import

import torch.nn as nn
from models_baseline.gcn.sem_graph_conv import SemGraphConv
import torch
from functools import reduce

'''
this folder and code is modified base on SemGCN code,
https://github.com/garyzhao/SemGCN
the SemGCN model.
'''

class _GraphConv(nn.Module):
    def __init__(self, adj, input_dim, output_dim, p_dropout=None):
        super(_GraphConv, self).__init__()

        self.gconv = SemGraphConv(input_dim, output_dim, adj)
        self.bn = nn.BatchNorm1d(output_dim)
        self.relu = nn.ReLU()

        if p_dropout is not None:
            self.dropout = nn.Dropout(p_dropout)
        else:
            self.dropout = None

    def forward(self, x):
        x = self.gconv(x).transpose(1, 2)
        x = self.bn(x).transpose(1, 2)
        if self.dropout is not None:
            x = self.dropout(self.relu(x))

        x = self.relu(x)
        return x


class _ResGraphConv(nn.Module):
    def __init__(self, adj, input_dim, output_dim, hid_dim, p_dropout):
        super(_ResGraphConv, self).__init__()

        self.gconv1 = _GraphConv(adj, input_dim, hid_dim, p_dropout)
        self.gconv2 = _GraphConv(adj, hid_dim, output_dim, p_dropout)

    def forward(self, x):
        residual = x
        out = self.gconv1(x)
        out = self.gconv2(out)
        return residual + out


class _GraphNonLocal(nn.Module):
    def __init__(self, hid_dim, grouped_order, restored_order, group_size):
        super(_GraphNonLocal, self).__init__()

        # self.nonlocal = GraphNonLocal(hid_dim, sub_sample=group_size) # keywords nonlocal
        self.grouped_order = grouped_order
        self.restored_order = restored_order

    def forward(self, x):
        out = x[:, self.grouped_order, :]
        # out = self.nonlocal(out.transpose(1, 2)).transpose(1, 2)
        out = out[:, self.restored_order, :]
        return out


class SemGCN(nn.Module):
    def __init__(self, adj, hid_dim, coords_dim=(2, 3), num_layers=4, nodes_group=None, p_dropout=None, num_branches=5):
        super(SemGCN, self).__init__()

        self.num_branches = num_branches
        _gconv_input = [_GraphConv(adj, coords_dim[0], hid_dim, p_dropout=p_dropout)]
        _gconv_layers = []

        if nodes_group is None:
            for i in range(num_layers):
                _gconv_layers.append(_ResGraphConv(adj, hid_dim, hid_dim, hid_dim, p_dropout=p_dropout))
        else:
            group_size = len(nodes_group[0])
            assert group_size > 1

            grouped_order = list(reduce(lambda x, y: x + y, nodes_group))
            restored_order = [0] * len(grouped_order)
            for i in range(len(restored_order)):
                for j in range(len(grouped_order)):
                    if grouped_order[j] == i:
                        restored_order[i] = j
                        break

            _gconv_input.append(_GraphNonLocal(hid_dim, grouped_order, restored_order, group_size))
            for i in range(num_layers):
                _gconv_layers.append(_ResGraphConv(adj, hid_dim, hid_dim, hid_dim, p_dropout=p_dropout))
                _gconv_layers.append(_GraphNonLocal(hid_dim, grouped_order, restored_order, group_size))

        self.gconv_input = nn.Sequential(*_gconv_input)
        self.gconv_layers = nn.Sequential(*_gconv_layers)

        # regression
        self.regression_stages = []
        self.regression_scale_stages = []
        for rl in range(self.num_branches):
            self.regression_stages.append(SemGraphConv(hid_dim, coords_dim[1], adj))
            self.regression_scale_stages.append(nn.Linear(16*128, 1))

        self.regression_stages = nn.ModuleList(self.regression_stages)
        self.regression_scale_stages = nn.ModuleList(self.regression_scale_stages)
        self.relu = nn.ReLU()

    def forward(self, x):
        """
        input: bx16x2 / bx32
        output: bx16x3
        """
        if len(x.shape) == 2:
            x = x.view(x.shape[0], 16, 2)
        out = self.gconv_input(x)
        outs = self.gconv_layers(out)

        poses = []
        scales = []
        for i in range(self.num_branches):
            out = self.regression_stages[i](outs)
            out_scale = self.regression_scale_stages[i](outs.reshape(outs.shape[0], -1))
            poses.append(out.reshape(-1,16*3))
            scales.append(self.relu(out_scale)+1)
        
        scaled_poses = []
        for i in range(self.num_branches):
            scaled_poses.append(poses[i]/(scales[i]))
        
        scaled_poses = torch.stack(scaled_poses, dim=0)     # [num_branch, #b, 48]   

        return scaled_poses