import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np

from models.LinearModel import KronLinear


class LeNet(nn.Module):
    def __init__(self):
        super(LeNet, self).__init__()
        self.conv1 = nn.Conv2d(1, 6, 5)
        self.conv2 = nn.Conv2d(6, 16, 5)
        self.fc1 = nn.Linear(16 * 4 * 4, 120)
        self.fc2 = nn.Linear(120, 84)
        self.fc3 = nn.Linear(84, 10)
        
    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2)
        x = x.view(-1, self.num_flat_features(x))
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = self.fc3(x)
        return x
    
class KPDLeNet(nn.Module):
    def __init__(self, patterns):
        super(KPDLeNet, self).__init__()
        self.patterns = patterns
        self.conv1 = nn.Conv2d(1, 6, 5)
        self.conv2 = nn.Conv2d(6, 16, 5)
        
        self.kron_fc1 = KronLinear(16 * 4 * 4, 120, patterns[0], structured_sparse=True, rank=5)
        self.kron_fc2 = KronLinear(120, 84, patterns[1], structured_sparse=True, rank=5)
        self.kron_fc3 = KronLinear(84, 10, patterns[2], structured_sparse=True, rank=5)
        
        
        
    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = F.max_pool2d(x, 2)
        x = F.relu(self.conv2(x))
        x = F.max_pool2d(x, 2)
        x = x.view(x.size(0), -1)
        x = F.relu(self.kron_fc1(x))
        x = F.relu(self.kron_fc2(x))
        x = self.kron_fc3(x)
        return x
    
    def get_group_lasso(self):
        group_lasso = 0
        s_num = 0
        for name, module in self._modules.items():
            if isinstance(module, KronLinear):
                group_lasso += torch.norm(module.s, p=2) ** 2
                s_num += module.s.flatten().shape[0]
                # print(group_lasso, s_num)
        return torch.sqrt(group_lasso/s_num)

    def get_s_norm(self):
        s_norm = 0
        s_num = 0
        for name, module in self._modules.items():
            if isinstance(module, KronLinear):
                s_norm += torch.norm(module.s, p=1)
                s_num += module.s.flatten().shape[0]
        return s_norm/s_num
    
    
    