import torch
import torch.utils.data
from torchvision import models


class Server(object):
    def __init__(self, conf, eval_dataset):  # 构造函数
        self.conf = conf
        self.global_model = eval('models.{}()'.format(conf['model_name']))  # 从配置文件获得模型名称并创建服务器模型
        self.eval_loader = torch.utils.data.DataLoader(eval_dataset, batch_size=self.conf["batch_size"],
                                                       shuffle=True)  # 创建测试集加载器用于测试最终的聚合模型
        if torch.cuda.is_available():
            self.global_model = self.global_model.cuda()

    def model_aggregrate(self, weight_accumulator):  # 聚合函数，更新全局模型用
        # 其中weight_accumulator存放客户端上传参数的变化值, 即更新前全局模型和本地更新后模型的参数变化L_t+1-G_t
        for name, data in self.global_model.state_dict().items():  # 获得全局模型的变量名和参数值
            update_per_layer = weight_accumulator[name] * ( 1 / self.conf["k"])
            # update_per_layer = weight_accumulator[name] * self.conf["lambda"]  # 乘以系数λ（更新步长）
            if data.type() != update_per_layer.type():  # 如果数据类型不符
                data.add_(update_per_layer.to(torch.int64))  # 进行数据转换后再累加
            else:
                data.add_(update_per_layer)  # 直接进行累加

    def model_eval(self):  # 训练结束后，对全局模型进行评估的函数
        self.global_model.eval()  # 标记进入测试模式，模型参数不发生变化

        total_loss = 0.0  # 记录损失和，计算平均损失用
        correct = 0  # 记录正确数目
        dataset_size = 0  # 测试数据总数
        for batch_id, batch in enumerate(self.eval_loader):  # 对测试数据进行编号和按batch提取数据
            data, target = batch  # 解包数据和标签
            dataset_size += data.size()[0]  # 获得当前batch的数据量，进行累加
            if torch.cuda.is_available():
                data, target = data.cuda(), target.cuda()  # 如果pytorch支持GPU，则使用cuda计算
            output = self.global_model(data)  # 获得预测结果
            total_loss += torch.nn.functional.cross_entropy(output, target, reduction='sum').item()  # 把计算的损失进行累加
            pred = output.data.max(1)[1]  # 取预测值最大的类索引
            correct += pred.eq(target.data.view_as(pred)).cpu().sum().item()  # 获得该batch中预测正确的数目
        acc = 100.0 * (float(correct) / float(dataset_size))  # 得到准确率的百分值
        total_l = total_loss / dataset_size  # 计算平均损失值
        return acc, total_l


