from abc import ABC, abstractmethod
import torch

from SwiLLaVA.llava.model.multimodal_encoder.builder import build_vision_tower
from SwiLLaVA.llava.model.multimodal_projector.builder import build_vision_projector

from SwiLLaVA.llava.constants import IGNORE_INDEX, IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_PATCH_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN, OBJECT_TOKEN_INDEX


class LlavaSearchMetaModel:

    def __init__(self, config):
        super(LlavaSearchMetaModel, self).__init__(config)

        if hasattr(config, "mm_vision_tower"):
            self.vision_tower = build_vision_tower(config, delay_load=True)
            self.mm_projector = build_vision_projector(config)
            self.mm_projector_object = build_vision_projector(config, object_projector=True)
            
    def get_vision_tower(self):
        vision_tower = getattr(self, 'vision_tower', None)
        if type(vision_tower) is list:
            vision_tower = vision_tower[0]
        return vision_tower

    def initialize_vision_modules(self, model_args, fsdp=None):
        vision_tower = model_args.vision_tower
        mm_vision_select_layer = model_args.mm_vision_select_layer
        mm_vision_select_feature = model_args.mm_vision_select_feature
        pretrain_mm_mlp_adapter = model_args.pretrain_mm_mlp_adapter
        pretrain_mm_perceiver_adapter = model_args.pretrain_mm_perceiver_adapter

        self.config.mm_vision_tower = vision_tower

        if self.get_vision_tower() is None:
            vision_tower = build_vision_tower(model_args)

            if fsdp is not None and len(fsdp) > 0:
                self.vision_tower = [vision_tower]
            else:
                self.vision_tower = vision_tower
        else:
            if fsdp is not None and len(fsdp) > 0:
                vision_tower = self.vision_tower[0]
            else:
                vision_tower = self.vision_tower
            vision_tower.load_model()

        self.config.use_mm_proj = True
        self.config.mm_projector_type = getattr(model_args, 'mm_projector_type', 'linear')
        self.config.object_mm_projector_type = getattr(model_args, 'object_mm_projector_type', 'perceiver')
        self.config.mm_hidden_size = vision_tower.hidden_size
        self.config.mm_vision_select_layer = mm_vision_select_layer
        self.config.mm_vision_select_feature = mm_vision_select_feature

        if getattr(self, 'mm_projector', None) is None:
            self.mm_projector = build_vision_projector(self.config)
            self.mm_projector_object = build_vision_projector(self.config, object_projector=True)


        if pretrain_mm_mlp_adapter is not None:
            mm_projector_weights = torch.load(pretrain_mm_mlp_adapter, map_location='cpu')
            def get_w(weights, keyword):
                return {k.split(keyword + '.')[1]: v for k, v in weights.items() if keyword in k}
            self.mm_projector.load_state_dict(get_w(mm_projector_weights, 'mm_projector'))

        if pretrain_mm_perceiver_adapter is not None:
            mm_projector_weights = torch.load(pretrain_mm_perceiver_adapter, map_location='cpu')
            def get_w(weights, keyword):
                return {k.split(keyword + '.')[1]: v for k, v in weights.items() if keyword in k}
            self.mm_projector_object.load_state_dict(get_w(mm_projector_weights, 'mm_projector'))


class LlavaSearchMetaForCausalLM(ABC):

    @abstractmethod
    def get_model(self):
        pass

    def get_vision_tower(self):
        return self.get_model().get_vision_tower()

    def encode_images(self, images):
        image_features = self.get_model().get_vision_tower()(images)
        image_features_long = self.get_model().mm_projector(image_features)
        image_features_short = self.get_model().mm_projector_object(image_features)
        return image_features_long, image_features_short
    
    def project_features(self, object_features):
        object_features = self.get_model().get_vision_tower()(object_features)
        image_features_long = self.get_model().mm_projector(object_features)
        object_features_short = self.get_model().mm_projector_object(object_features)
        return image_features_long, object_features_short
    
    def prepare_inputs_labels_for_multimodal(
        self, input_ids, attention_mask, past_key_values, labels, images, object_features, images_long=None, objects_long=None
    ):  
        vision_tower = self.get_vision_tower()
        if vision_tower is None or images is None or input_ids.shape[1] == 1:
            if past_key_values is not None and vision_tower is not None and images is not None and input_ids.shape[1] == 1:
                attention_mask = torch.ones((attention_mask.shape[0], past_key_values[-1][-1].shape[-2] + 1), dtype=attention_mask.dtype, device=attention_mask.device)
            return input_ids, attention_mask, past_key_values, None, labels
        if type(images) is list or images.ndim == 5:
            concat_images = torch.cat([image for image in images], dim=0)
            image_features = self.encode_images(concat_images)
            split_sizes = [image.shape[0] for image in images]
            image_features = torch.split(image_features, split_sizes, dim=0)
            image_features = [x.flatten(0, 1) for x in image_features]
        else:
            image_features_long, image_features_short = self.encode_images(images)

        if object_features is not None and len(object_features) > 0:
            projected_object_features_long, projected_object_features_short = self.project_features(object_features)

        new_input_embeds = []
        new_labels = [] if labels is not None else None
        new_attention_mask = [] if attention_mask is not None else None
        cur_image_idx = 0
        cur_object_idx = 0
        for batch_idx, cur_input_ids in enumerate(input_ids):
            if (cur_input_ids == IMAGE_TOKEN_INDEX).sum() == 0:
                # multimodal LLM, but the current sample is not multimodal
                half_len = cur_input_ids.shape[0] // 2
                cur_object_features = projected_object_features_short[cur_object_idx]
                cur_input_embeds_1 = self.get_model().embed_tokens(cur_input_ids[:half_len])
                cur_input_embeds_2 = self.get_model().embed_tokens(cur_input_ids[half_len:])
                cat_list = [cur_input_embeds_1, image_features_short[cur_image_idx][0:0], image_features_long[cur_image_idx][0:0]]
                for _ in range(3):
                    cat_list.extend([projected_object_features_short[cur_object_idx][0:0], projected_object_features_long[cur_object_idx][0:0]])
                    cur_object_idx += 1
                cat_list.append(cur_input_embeds_2)
                cur_input_embeds = torch.cat(cat_list, dim=0)
                new_input_embeds.append(cur_input_embeds)
                if labels is not None:
                    new_labels.append(labels[batch_idx])
                cur_image_idx += 1
                new_attention_mask.append(attention_mask[batch_idx])
                continue
            image_token_indices = torch.where(cur_input_ids == IMAGE_TOKEN_INDEX)[0]
            cur_new_input_embeds = []
            if labels is not None:
                cur_labels = labels[batch_idx]
                cur_new_labels = []
                assert cur_labels.shape == cur_input_ids.shape
            if attention_mask is not None:
                cur_attention_mask = attention_mask[batch_idx]
                cur_new_attention_mask = []
                assert cur_attention_mask.shape == cur_input_ids.shape
            while image_token_indices.numel() > 0:
                if images_long is None or images_long[cur_image_idx]:
                    cur_image_features = torch.cat([image_features_short[cur_image_idx][0:0], image_features_long[cur_image_idx]])
                else:
                    cur_image_features = torch.cat([image_features_short[cur_image_idx], image_features_long[cur_image_idx][0:0]])
                image_token_start = image_token_indices[0]
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[:image_token_start-1]).detach())
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[image_token_start-1:image_token_start]))
                    cur_new_input_embeds.append(cur_image_features)
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[image_token_start+1:image_token_start+2]))
                    if labels is not None:
                        cur_new_labels.append(cur_labels[:image_token_start])
                        cur_new_labels.append(torch.full((cur_image_features.shape[0],), IGNORE_INDEX, device=labels.device, dtype=labels.dtype))
                        cur_new_labels.append(cur_labels[image_token_start:image_token_start+1])
                        cur_labels = cur_labels[image_token_start+2:]
                    if attention_mask is not None:
                        cur_new_attention_mask.append(cur_attention_mask[:image_token_start])
                        if cur_attention_mask[image_token_start]:
                            cur_new_attention_mask.append(torch.full((cur_image_features.shape[0],), True, device=attention_mask.device, dtype=attention_mask.dtype))
                        else:
                            cur_new_attention_mask.append(torch.full((cur_image_features.shape[0],), False, device=attention_mask.device, dtype=attention_mask.dtype))
                        cur_new_attention_mask.append(cur_attention_mask[image_token_start:image_token_start+1])
                        cur_attention_mask = cur_attention_mask[image_token_start+2:]
                else:
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[:image_token_start]))
                    cur_new_input_embeds.append(cur_image_features)
                    if labels is not None:
                        cur_new_labels.append(cur_labels[:image_token_start])
                        cur_new_labels.append(torch.full((cur_image_features.shape[0],), IGNORE_INDEX, device=labels.device, dtype=labels.dtype))
                        cur_labels = cur_labels[image_token_start+1:]
                    if attention_mask is not None:
                        cur_new_attention_mask.append(cur_attention_mask[:image_token_start])
                        if cur_attention_mask[image_token_start]:
                            cur_new_attention_mask.append(torch.full((cur_image_features.shape[0],), True, device=attention_mask.device, dtype=attention_mask.dtype))
                        else:
                            cur_new_attention_mask.append(torch.full((cur_image_features.shape[0],), False, device=attention_mask.device, dtype=attention_mask.dtype))
                        cur_attention_mask = cur_attention_mask[image_token_start+1:]
                cur_image_idx += 1
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    cur_input_ids = cur_input_ids[image_token_start+2:]
                else:
                    cur_input_ids = cur_input_ids[image_token_start+1:]
                image_token_indices = torch.where(cur_input_ids == IMAGE_TOKEN_INDEX)[0]
            object_token_indices = torch.where(cur_input_ids == OBJECT_TOKEN_INDEX)[0]
            cur_object_num = object_token_indices.numel()
            while object_token_indices.numel() > 0:
                if objects_long is None or not objects_long[cur_object_idx]:
                    cur_object_features = torch.cat([projected_object_features_short[cur_object_idx], projected_object_features_long[cur_object_idx][0:0]]) 
                else:
                    cur_object_features = torch.cat([projected_object_features_short[cur_object_idx][0:0],projected_object_features_long[cur_object_idx]])
                object_token_start = object_token_indices[0]
                cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids[:object_token_start]))
                cur_new_input_embeds.append(cur_object_features)
                if labels is not None:
                    cur_new_labels.append(cur_labels[:object_token_start])
                    cur_new_labels.append(torch.full((cur_object_features.shape[0],), IGNORE_INDEX, device=labels.device, dtype=labels.dtype))
                    cur_labels = cur_labels[object_token_start+1:]
                if attention_mask is not None:
                    cur_new_attention_mask.append(cur_attention_mask[:object_token_start])
                    if cur_attention_mask[object_token_start]:
                        cur_new_attention_mask.append(torch.full((cur_object_features.shape[0],), True, device=attention_mask.device, dtype=attention_mask.dtype))
                    else:
                        cur_new_attention_mask.append(torch.full((cur_object_features.shape[0],), False, device=attention_mask.device, dtype=attention_mask.dtype))
                    cur_attention_mask = cur_attention_mask[object_token_start+1:]
                cur_object_idx += 1
                cur_input_ids = cur_input_ids[object_token_start+1:]
                object_token_indices = torch.where(cur_input_ids == OBJECT_TOKEN_INDEX)[0]
            
            if cur_input_ids.numel() > 0:
                if getattr(self.config, 'tune_mm_mlp_adapter', False) and getattr(self.config, 'mm_use_im_start_end', False):
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids).detach())
                else:
                    cur_new_input_embeds.append(self.get_model().embed_tokens(cur_input_ids))
                if labels is not None:
                    cur_new_labels.append(cur_labels)
                if attention_mask is not None:
                    cur_new_attention_mask.append(cur_attention_mask)
            cur_new_input_embeds = [x.to(device=self.device) for x in cur_new_input_embeds]
            cur_new_input_embeds = torch.cat(cur_new_input_embeds, dim=0)
            new_input_embeds.append(cur_new_input_embeds)
            if labels is not None:
                cur_new_labels = torch.cat(cur_new_labels, dim=0)
                new_labels.append(cur_new_labels)
            if attention_mask is not None:
                cur_new_attention_mask = torch.cat(cur_new_attention_mask, dim=0)
                new_attention_mask.append(cur_new_attention_mask)

        need_padding = False
        for i in range(len(new_input_embeds)):
            for j in range(i+1, len(new_input_embeds)):
                if new_input_embeds[i].shape != new_input_embeds[j].shape:
                    need_padding = True
                    break
            if need_padding:
                break
        if need_padding:
            max_len = max(x.shape[0] for x in new_input_embeds)

            new_input_embeds_align = []
            for cur_new_embed in new_input_embeds:
                cur_new_embed = torch.cat((cur_new_embed, torch.zeros((max_len - cur_new_embed.shape[0], cur_new_embed.shape[1]), dtype=cur_new_embed.dtype, device=cur_new_embed.device)), dim=0)
                new_input_embeds_align.append(cur_new_embed)
            new_input_embeds = torch.stack(new_input_embeds_align, dim=0)

            if labels is not None:
                new_labels_align = []
                for cur_new_label in new_labels:
                    cur_new_label = torch.cat((cur_new_label, torch.full((max_len - cur_new_label.shape[0],), IGNORE_INDEX, dtype=cur_new_label.dtype, device=cur_new_label.device)), dim=0)
                    new_labels_align.append(cur_new_label)
                new_labels = torch.stack(new_labels_align, dim=0)

            if attention_mask is not None:
                new_attention_mask_align = []
                for cur_new_attention_mask in new_attention_mask:
                    new_attn_mask_pad_right = torch.full((max_len - cur_new_attention_mask.shape[0],), False, dtype=attention_mask.dtype, device=attention_mask.device)
                    cur_new_attention_mask = torch.cat((cur_new_attention_mask, new_attn_mask_pad_right), dim=0)
                    new_attention_mask.append(cur_new_attention_mask)
                attention_mask = torch.stack(new_attention_mask, dim=0)
                assert attention_mask.shape == new_labels.shape

        else:
            new_input_embeds = torch.stack(new_input_embeds, dim=0)
            if labels is not None:
                new_labels  = torch.stack(new_labels, dim=0)
            if new_attention_mask is not None and len(new_attention_mask):
                new_attention_mask  = torch.stack(new_attention_mask, dim=0)
                attention_mask = new_attention_mask
                assert attention_mask.shape == new_input_embeds.shape[:2]
        
        return None, attention_mask, past_key_values, new_input_embeds, new_labels

    def initialize_vision_tokenizer(self, model_args, tokenizer):
        if model_args.mm_use_im_patch_token:
            tokenizer.add_tokens([DEFAULT_IMAGE_PATCH_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

        if model_args.mm_use_im_start_end:
            num_new_tokens = tokenizer.add_tokens([DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN], special_tokens=True)
            self.resize_token_embeddings(len(tokenizer))

            if num_new_tokens > 0:
                input_embeddings = self.get_input_embeddings().weight.data
                output_embeddings = self.get_output_embeddings().weight.data

                input_embeddings_avg = input_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)
                output_embeddings_avg = output_embeddings[:-num_new_tokens].mean(
                    dim=0, keepdim=True)

                input_embeddings[-num_new_tokens:] = input_embeddings_avg
                output_embeddings[-num_new_tokens:] = output_embeddings_avg

            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = True
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False

            if model_args.pretrain_mm_mlp_adapter:
                mm_projector_weights = torch.load(model_args.pretrain_mm_mlp_adapter, map_location='cpu')
                embed_tokens_weight = mm_projector_weights['model.embed_tokens.weight']
                assert num_new_tokens == 2
                if input_embeddings.shape == embed_tokens_weight.shape:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight[-num_new_tokens:]
                elif embed_tokens_weight.shape[0] == num_new_tokens:
                    input_embeddings[-num_new_tokens:] = embed_tokens_weight
                else:
                    raise ValueError(f"Unexpected embed_tokens_weight shape. Pretrained: {embed_tokens_weight.shape}. Current: {input_embeddings.shape}. Numer of new tokens: {num_new_tokens}.")
        elif model_args.mm_use_im_patch_token:
            if model_args.tune_mm_mlp_adapter:
                for p in self.get_input_embeddings().parameters():
                    p.requires_grad = False
                for p in self.get_output_embeddings().parameters():
                    p.requires_grad = False
