from typing import List, Optional, Tuple
import argparse
from pkg_resources import require

from seism.simulation_power import do_simulation
from seism.analysis import do_analysis



class UsageFormatter(argparse.RawDescriptionHelpFormatter):
    """Usage formatter with more columns."""
    def __init__(self,prog):
        super().__init__(prog, max_help_position = 60, width = 120)
################################################################################
# Simulation mode

def simulation_parser(parser: argparse.ArgumentParser):
    simulated_dataset_definition = parser.add_argument_group("Parameters for the simulated dataset")
    simulated_dataset_definition.add_argument(
        "--is-not-calibration", dest="is_calibration",
        action = 'store_false',
        help = "Use this toggle if you want to be under H1"
    )
    simulated_dataset_definition.add_argument(
        "--nb-motifs", dest="nb_motifs",
        type = int, default = 1,
        help = "Number of motifs to search for (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--nb-samples", dest="nb_samples",
        type = int, default = 30,
        help = "Number of samples in the simulated dataset (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--model-noise", dest="model_noise",
        type = float, default = 1,
        help = "Variance of the gaussian noise (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--sequences-length", dest="sequences_length",
        type = int, default = 30,
        help = "Length of the sequences in the simulated dataset (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--min-motifs-length", dest="min_motifs_length",
        type = int, default = 8,
        help = "Minimum length of the motifs to search for (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--max-motifs-length", dest="max_motifs_length",
        type = int, default = 8,
        help = "Maximum length of the motifs to search for (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--nb-true-motifs", dest="nb_true_motifs",
        type = int, default = 1,
        help = "Number of motifs to include in the sequences (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--true-motifs-length", dest="true_motifs_length",
        type = int, default = 8,
        help = "Length of the motifs to include in the sequences (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--true-motifs-information", dest="true_motifs_information",
        type = int, default = 10,
        help = "Information (bits) of the motifs to include in the sequences (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--association-score", dest="association_score",
        choices = ["ridge", "hsic"], default = "ridge",
        help = "Choice of the association score (default: %(default)s)"
    )
    simulated_dataset_definition.add_argument(
        "--ridge-lambda", dest="ridge_lambda",
        type = float, default = 1e-4,
        help = "Penalty to use if using a ridge score (default: %(default)s)"
    )


    inference_definition = parser.add_argument_group("Main choices for the inference method")
    inference_definition.add_argument(
        "--inferer-type", dest="inferer_type",
        choices = ["hit_and_run", "rejection", "data_split"] , default = "hit_and_run",
        help = "Method for inference to be used (default: %(default)s)"
    )
    inference_definition.add_argument(
        "--number-of-threads", dest="nb_threads",
        type = int , default = 1,
        help = "Number of threads allowed for parallelization (default: %(default)s)"
    )

    hit_and_run_definition = parser.add_argument_group("Parameters for inference using a Hit and Run sampler")
    hit_and_run_definition.add_argument(
        "--hr-nb-burn-in", dest = "hr_nb_burn_in",
        type = int, default = 10000,
        help = "Number of burn in iterations (default: %(default)s)"
    )
    hit_and_run_definition.add_argument(
        "--hr-nb-replicates", dest = "hr_nb_replicates",
        type = int, default = 50000,
        help = "Number of replicates (default: %(default)s)"
    )
    hit_and_run_definition.add_argument(
        "--hr-mesh-size", dest = "hr_mesh_size",
        type = float, default = 0.5,
        help = "Size of the meshes for the motif space (default: %(default)s)"
    )

    rejection_definition = parser.add_argument_group("Parameters for inference using a rejection sampler")
    rejection_definition.add_argument(
        "--rs-nb-replicates", dest = "rs_nb_replicates",
        type = int, default = 1000,
        help = "Number of replicates (default: %(default)s)"
    )
    rejection_definition.add_argument(
        "--rs-mesh-size", dest = "rs_mesh_size",
        type = float, default = 0.5,
        help = "Size of the meshes for the motif space (default: %(default)s)"
    )

    data_split_definition = parser.add_argument_group("Parameters for inference using data splitting strategies")
    data_split_definition.add_argument(
        "--ds-nb-replicates", dest = "ds_nb_replicates",
        type = int, default = 1000,
        help = "Number of replicates (default: %(default)s)"
    )
    data_split_definition.add_argument(
        "--ds-split-ratio", dest = "ds_split_ratio",
        type = float, default = 0.8,
        help = "Balance train/test datasets (default: %(default)s)"
    )

def setup_simulation_parser(parser: argparse.ArgumentParser):
    parser.add_argument(
        "output", type = str,
        help = "Location to store the results of the simulation"
    )
    simulation_parser(parser)

def do_power_simulation(argument_values: argparse.Namespace):
    do_simulation(argument_values)

################################################################################
# File analysis mode

def analysis_parser(parser: argparse.ArgumentParser):
    selection_parameters_defininition = parser.add_argument_group("Parameters for the motif discovery")
    selection_parameters_defininition.add_argument(
        "--nb-motifs", dest="nb_motifs",
        type = int, default = 1,
        help = "Number of motifs to search for (default: %(default)s)"
    )
    selection_parameters_defininition.add_argument(
        "--min-motifs-length", dest="min_motifs_length",
        type = int, default = 8,
        help = "Minimum authorized length for the motifs (default: %(default)s)"
    )
    selection_parameters_defininition.add_argument(
        "--max-motifs-length", dest="max_motifs_length",
        type = int, default = 8,
        help = "Maximum authorized length for the motifs(default: %(default)s)"
    )
    selection_parameters_defininition.add_argument(
        "--association-score", dest="association_score",
        choices = ["ridge", "hsic"], default = "ridge",
        help = "Choice of the association score (default: %(default)s)"
    )
    selection_parameters_defininition.add_argument(
        "--ridge-lambda", dest="ridge_lambda",
        type = float, default = 1e-4,
        help = "Penalty to use if using a ridge score (default: %(default)s)"
    )
    inference_parameters_definition = parser.add_argument_group("Parameters for the inference step")
    inference_parameters_definition.add_argument(
        "--do-not-perform-inference", dest="perform_inference",
        action = 'store_false',
        help = "Use this toggle if you do NOT want to test the selected motifs"
    )
    inference_parameters_definition.add_argument(
        "--inferer-type", dest="inferer_type",
        choices = ["hit_and_run", "rejection", "data_split"] , default = "hit_and_run",
        help = "Method for inference to be used (default: %(default)s) | Please do not use rejection, only used for tests."
    )
    inference_parameters_definition.add_argument(
        "--number-of-threads", dest="nb_threads",
        type = int , default = 1,
        help = "Number of threads allowed for parallelization (default: %(default)s)"
    )
    hit_and_run_definition = parser.add_argument_group("Parameters for inference using a Hit and Run sampler")
    hit_and_run_definition.add_argument(
        "--hr-nb-burn-in", dest = "hr_nb_burn_in",
        type = int, default = 10000,
        help = "Number of burn in iterations (default: %(default)s)"
    )
    hit_and_run_definition.add_argument(
        "--hr-nb-replicates", dest = "hr_nb_replicates",
        type = int, default = 50000,
        help = "Number of replicates (default: %(default)s)"
    )
    hit_and_run_definition.add_argument(
        "--hr-mesh-size", dest = "hr_mesh_size",
        type = float, default = 0.5,
        help = "Size of the meshes for the motif space (default: %(default)s)"
    )

    rejection_definition = parser.add_argument_group("Parameters for inference using a rejection sampler")
    rejection_definition.add_argument(
        "--rs-nb-replicates", dest = "rs_nb_replicates",
        type = int, default = 1000,
        help = "Number of replicates (default: %(default)s)"
    )
    rejection_definition.add_argument(
        "--rs-mesh-size", dest = "rs_mesh_size",
        type = float, default = 0.5,
        help = "Size of the meshes for the motif space (default: %(default)s)"
    )

    data_split_definition = parser.add_argument_group("Parameters for inference using data splitting strategies")
    data_split_definition.add_argument(
        "--ds-nb-replicates", dest = "ds_nb_replicates",
        type = int, default = 1000,
        help = "Number of replicates (default: %(default)s)"
    )
    data_split_definition.add_argument(
        "--ds-split-ratio", dest = "ds_split_ratio",
        type = float, default = 0.8,
        help = "Balance train/test datasets (default: %(default)s)"
    )


def setup_analysis_parser(parser: argparse.ArgumentParser):
    parser.add_argument(
        "input", type = str,
        help = "Location of the input fasta file"
    )
    parser.add_argument(
        "output", type = str,
        help = "Location to store the results of the analysis"
    )
    analysis_parser(parser)

def do_cli_analysis(argument_values: argparse.ArgumentParser):
    do_analysis(argument_values)


################################################################################
# To implement (model)

def not_implemented_parser(parser: argparse.ArgumentParser):
    parameters_definition = parser.add_argument_group("Parameters definition")
    parameters_definition.add_argument(
        "--results-location", dest="results_location",
        type = str, required = True,
        help = "Location to store the results"
    )

def setup_not_implemented_parser(parser: argparse.ArgumentParser):
    parser.add_argument(
        "output", type = argparse.FileType("wb"),
        help = ""
    )
    not_implemented_parser(parser)

def do_not_implemented(argument_values: argparse.Namespace):
    pass

################################################################################
# Entry point

def argument_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog = "seism",
        description = "SEISM implementation",
        formatter_class = UsageFormatter
    )
    modes = parser.add_subparsers(dest = "mode")
    modes.required = True # Can be specified in add_subparsers since python 3.7

    setup_simulation_parser(modes.add_parser(
        "simulation",
        help = "Perform a simulation procedure",
        formatter_class = UsageFormatter
    ))
    setup_analysis_parser(modes.add_parser(
        "analysis",
        help = "Perform analysis on a fasta file",
        formatter_class = UsageFormatter
    ))
    setup_not_implemented_parser(modes.add_parser(
        "to be implemented",
        help = "to be implemented",
        formatter_class = UsageFormatter
    ))
    return parser

def main():
    parser = argument_parser()
    args = parser.parse_args()

    if args.mode == "simulation":
        do_power_simulation(args)
    elif args.mode == "analysis":
        do_cli_analysis(args)
    else:
        assert False, "unknown mode: {}".format(args.mode)

if __name__ == '__main__':
    main()
