import argparse
from portfolio_environment import environment
import numpy as np
from itertools import count

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
from torch.distributions import Categorical


parser = argparse.ArgumentParser(description='PyTorch REINFORCE example')
parser.add_argument('--seed', type=int, default=543, metavar='N',
                    help='random seed (default: 543)')
parser.add_argument('--render', action='store_true',
                    help='render the environment')
parser.add_argument('--log-interval', type=int, default=10, metavar='N',
                    help='interval between training status logs (default: 10)')
parser.add_argument('--log_path', type=str, default='results/temp',
                    help='log path')
parser.add_argument('--trials', type=int, default=100,
                    help='num trials')
args = parser.parse_args()

target_var = 100


env = environment()
torch.manual_seed(args.seed)


class Policy(nn.Module):
    def __init__(self):
        super(Policy, self).__init__()
        self.affine1 = nn.Linear(env.W+2, env.W+2)
        self.affine2 = nn.Linear(env.W+2, 2)

        self.saved_log_probs = []
        self.rewards = []

    def forward(self, x):
        x = self.affine1(x)
        x = F.relu(x)
        action_scores = self.affine2(x)
        return F.softmax(action_scores, dim=1)


def select_action(state, policy):
    state = torch.from_numpy(state).float().unsqueeze(0)
    probs = policy(state)
    m = Categorical(probs)
    action = m.sample()
    policy.saved_log_probs.append(m.log_prob(action))
    return action.item()


def finish_episode(model, optimizer, eps, i_episode, J, V):
    """
    Training code. Calculates actor and critic loss and performs backprop.
    """

    param1 = 1/(i_episode + 1)**(1/100)
    param2 = 1/(i_episode + 1)**(1/10)

    R = 0
    policy_loss = []
    for r in model.rewards:
        R = r + R

    J += 1e-2*param1*(R - J)
    V += 1e-2*param1*(R**2 - J**2 - V)

    objective = np.max([0, V - target_var]) * (R**2 - 2 * J)

    for log_prob in model.saved_log_probs:
        policy_loss.append(-log_prob * (R - objective))
    optimizer.zero_grad()
    policy_loss = param2*torch.cat(policy_loss).sum()
    policy_loss.backward()
    optimizer.step()
    del model.rewards[:]
    del model.saved_log_probs[:]

    return J, V


def main():
    ep_reward_list_train = []
    ep_reward_list_test = []

    for tr in range(1000):

        running_ep_reward = []

        policy = Policy()
        optimizer = optim.Adam(policy.parameters(), lr=1e-2)
        eps = np.finfo(np.float32).eps.item()

        J = 0
        V = 0

        for i_episode in range(1000):
            state, ep_reward = env.reset(), 0
            for t in range(1, 50):  # Don't infinite loop while learning
                action = select_action(state, policy)
                state, reward = env.step(action)
                policy.rewards.append(reward)
                ep_reward += reward

            running_ep_reward.append(ep_reward)
            J, V = finish_episode(policy, optimizer, eps, i_episode, J, V)

            # log results
            if i_episode % args.log_interval == 0:
                print('Episode {}\tLast reward: {:.2f}'.format(
                    i_episode, ep_reward))

        ep_reward_list_train.append(running_ep_reward)
        np.savetxt('results/equm_tamar_train%2f.csv' % target_var,
                   ep_reward_list_train)

        ep_reward_list_test_temp = []

        for i_episode in range(100):

            state = env.reset()
            ep_reward = 0

            for t in range(1, 50):

                # select action from policy
                action = select_action(state, policy)

                # take the action
                state, reward = env.step(action)
                policy.rewards.append(reward)
                ep_reward += reward

            ep_reward_list_test_temp.append(ep_reward)

        ep_reward_list_test.append(ep_reward_list_test_temp)
        np.savetxt('results/equm_tamar_test%2f.csv' %
                   target_var, ep_reward_list_test)


if __name__ == '__main__':
    main()
