# Code is adapted from: https://github.com/DonGovi/pyramid-detection-3D

import math

import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.autograd import Variable


__all__ = ['ResNet3D', 'resnet18', 'resnet34', 'resnet50', 'resnet101',
           'resnet152']


model_urls = {
    'resnet18': 'https://download.pytorch.org/models/resnet18-5c106cde.pth',
    'resnet34': 'https://download.pytorch.org/models/resnet34-333f7ec4.pth',
    'resnet50': 'https://download.pytorch.org/models/resnet50-19c8e357.pth',
    'resnet101': 'https://download.pytorch.org/models/resnet101-5d3b4d8f.pth',
    'resnet152': 'https://download.pytorch.org/models/resnet152-b121ed2d.pth',
}



def conv3x3x3(in_planes, out_planes, stride=1):
    """3x3x3 convolution with padding"""
    return nn.Conv3d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=1, bias=False)
'''
def downsample_basic_block(x, planes, stride):
    out = F.avg_pool3d(x, kernel_size=1, stride=stride)
    zero_pads = torch.Tensor(
        out.size(0), planes - out.size(1), out.size(2), out.size(3),
        out.size(4)).zero_()
    if isinstance(out.data, torch.cuda.FloatTensor):
        zero_pads = zero_pads.cuda()

    out = Variable(torch.cat([out.data, zero_pads], dim=1))

    return out
'''

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, downsample=None):
        super(BasicBlock, self).__init__()
        self.conv1 = conv3x3x3(in_planes, planes, stride)
        self.bn1 = nn.BatchNorm3d(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3x3(planes, planes)
        self.bn2 = nn.BatchNorm3d(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        #conv2_rep = out
        out = self.bn2(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1, downsample=None):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv3d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm3d(planes)
        self.conv2 = nn.Conv3d(planes, planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm3d(planes)
        self.conv3 = nn.Conv3d(planes, planes * 4, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm3d(planes * 4)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        residual = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        #conv3_rep = out
        out = self.bn3(out)

        if self.downsample is not None:
            residual = self.downsample(x)

        out += residual
        out = self.relu(out)

        return out


class ResNet3D(nn.Module):
    def __init__(self, in_channels, block, layers, grid_step=None, N_features=1024):
        self.in_planes = 64
        super(ResNet3D, self).__init__()
        self.conv1 = nn.Conv3d(in_channels, 64, kernel_size=5, stride=2, padding=2, bias=False)  # 128 -> 64
        self.bn1 = nn.BatchNorm3d(64)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool3d(kernel_size=3, stride=2, padding=1)    # 64 -> 32
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2)    # 32 -> 16
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2)    # 16 -> 8

        assert N_features in [1024, 2048], 'N_features should be 1024 or 2048'
        self.N_features = N_features    
        if self.N_features == 2048:
            self.layer4 = self._make_layer(block, 512, layers[3], stride=2)    # 8 -> 4

        # self.layer5 = self._make_layer(block, 256, layers[3], stride=2)    # 4 -> 2

        if grid_step is None:
            grid_step = 1/128

        if grid_step >= 1/64 - 1/512:
            print('grid_size == 1/64')
            if self.N_features == 2048:
                self.avgpool = nn.AvgPool3d(2, stride=1)
            else:
                self.avgpool = nn.AvgPool3d(4, stride=1)
        elif grid_step >= 1/128 - 1/512:
            print('grid_size == 1/128')
            if self.N_features == 2048:
                self.avgpool = nn.AvgPool3d(4, stride=1)
            else:
                self.avgpool = nn.AvgPool3d(8, stride=1)
        else:
            print('grid_size == 1/256')
            if self.N_features == 2048:
                self.avgpool = nn.AvgPool3d(8, stride=1)
            else:
                self.avgpool = nn.AvgPool3d(16, stride=1)
        '''
        self.fc = nn.Linear(512 * block.expansion, num_classes)
        '''

        for m in self.modules():
            if isinstance(m, nn.Conv3d):
                m.weight = nn.init.xavier_normal_(m.weight)
            elif isinstance(m, nn.BatchNorm3d):
                m.weight.data.fill_(1)
                m.bias.data.zero_()

    def _make_layer(self, block, planes, blocks, stride=1):
        downsample = None
        if stride != 1 or self.in_planes != planes * block.expansion:
            downsample = nn.Sequential(
                nn.Conv3d(self.in_planes, planes * block.expansion,
                          kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm3d(planes * block.expansion)
            )

        layers = []
        layers.append(block(self.in_planes, planes, stride, downsample))
        self.in_planes = planes * block.expansion
        for i in range(1, blocks):
            layers.append(block(self.in_planes, planes))

        return nn.Sequential(*layers)

    def forward(self, x):       # 128 
        c1 = self.conv1(x)      # 64 --> 8 anchor_area
        c1 = self.bn1(c1)         
        c1 = self.relu(c1)
        c2 = self.maxpool(c1)   # 32

        c2 = self.layer1(c2)  # 32 --> 16 anchor_area
        c3 = self.layer2(c2)  # 16 --> 32 anchor_area
        c4 = self.layer3(c3)  # 8
        if self.N_features == 2048:
            c5 = self.layer4(c4) # 4
            out = self.avgpool(c5)
        else: 
            # 1024 features
            out = self.avgpool(c4)
        
        # c6 = self.layer5(c5)  # 4
        return out
        '''
        x = x.view(x.size(0), -1)
        x = self.fc(x)
        '''
        # return c1, c2, c3, c4, c5, c6
        # return c1, c2, c3, c4, c5


def resnet18(in_channels=3, pretrained=False, grid_step=None,N_features=None, **kwargs):
    """Constructs a ResNet-18 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet3D(in_channels, BasicBlock, [2, 2, 2, 2], grid_step=grid_step, N_features=N_features, **kwargs)
    # if pretrained:
    #     model.load_state_dict(model_zoo.load_url(model_urls['resnet18']))
    return model


def resnet34(in_channels=3, pretrained=False, grid_step=None, N_features=None, **kwargs):
    """Constructs a ResNet-34 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet3D(in_channels, BasicBlock, [3, 4, 6, 3], grid_step=grid_step,N_features=N_features, **kwargs)
    # if pretrained:
    #     model.load_state_dict(model_zoo.load_url(model_urls['resnet34']))
    return model


def resnet50(in_channels=3, pretrained=False, grid_step=None, N_features=None, **kwargs):
    """Constructs a ResNet-50 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet3D(in_channels, Bottleneck, [3, 4, 6, 3], grid_step=grid_step,N_features=N_features, **kwargs)
    if pretrained:
        print('Pre-trained')
        model.load_state_dict(torch.utils.model_zoo.load_url(model_urls['resnet50']))
    return model


def resnet101(in_channels=3, pretrained=False, grid_step=None, N_features=None, **kwargs):
    """Constructs a ResNet-101 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet3D(in_channels, Bottleneck, [3, 4, 23, 3], grid_step = grid_step, N_features=N_features, **kwargs)
    # if pretrained:
    #     model.load_state_dict(model_zoo.load_url(model_urls['resnet101']))
    return model


def resnet152(in_channels=3, pretrained=False, grid_step = None, N_features=None, **kwargs):
    """Constructs a ResNet-152 model.
    Args:
        pretrained (bool): If True, returns a model pre-trained on ImageNet
    """
    model = ResNet3D(in_channels, Bottleneck, [3, 8, 36, 3], grid_step = grid_step, N_features=N_features,  **kwargs)
    # if pretrained:
    #     model.load_state_dict(model_zoo.load_url(model_urls['resnet152']))
    return model


class ResNet3D_helper(nn.Module):
    backbones = {
            'resnet18': resnet18,
            'resnet34': resnet34,
            'resnet50': resnet50,
            'resnet101': resnet101,
            'resnet152': resnet152
        }
    def __init__(self, in_channels=3, backbone='resnet50', pretrained=False, grid_step=None, N_features=1024):
        super(ResNet3D_helper, self).__init__()

        self.backbone_net = ResNet3D_helper.backbones[backbone](
            in_channels=in_channels,
            pretrained=pretrained,
            grid_step=grid_step, 
            N_features=N_features
        )	

    def forward(self, x):
        return self.backbone_net(x)
        # return resnet_feature_1, resnet_feature_2, resnet_feature_3, resnet_feature_4, resnet_feature_5, resnet_feature_6
