"""
    ShuffleNet for ImageNet-1K, implemented in PyTorch.
    Original paper: 'ShuffleNet: An Extremely Efficient Convolutional Neural Network for Mobile Devices,'
    https://arxiv.org/abs/1707.01083.
    Adapted from https://github.com/osmr/imgclsmob/blob/master/pytorch/pytorchcv/models/shufflenet.py
"""

import os
import torch
import torch.nn as nn
import torch.nn.init as init
from Layers.layers import conv1x1, conv3x3, conv1x1_block, conv3x3_block, Conv2d, Linear, BatchNorm2d, ChannelShuffle, sparse_initialize
from Models.imagenet_shufflenet import ShuffleUnit


# change in_size and num_classes accordingly
class ShuffleNet(nn.Module):
    def __init__(self, channels, init_block_channels, groups, in_channels=3, in_size=(32, 32), num_classes=10):
        super(ShuffleNet, self).__init__()
        self.in_size = in_size
        self.num_classes = num_classes

        self.features = nn.Sequential()
        # change ShuffleInit Block to conv1x1_block
        # maybe we want to try conv3x3 block instead
        self.features.add_module("init_block", conv3x3_block(in_channels=in_channels, out_channels=init_block_channels))
        in_channels = init_block_channels
        for i, channels_per_stage in enumerate(channels):
            stage = nn.Sequential()
            for j, out_channels in enumerate(channels_per_stage):
                downsample = (j == 0)
                ignore_group = (i == 0) and (j == 0)
                stage.add_module(
                    "unit{}".format(j + 1),
                    ShuffleUnit(in_channels=in_channels, out_channels=out_channels, groups=groups, downsample=downsample, ignore_group=ignore_group))
                in_channels = out_channels
            self.features.add_module("stage{}".format(i + 1), stage)
        # change AvgPool2d(7) to AvgPool2d(4)
        self.features.add_module("final_pool", nn.AvgPool2d(kernel_size=4, stride=1))

        self.output = Linear(in_features=in_channels, out_features=num_classes)

        self._initialize_weights(sparse_init=False)
        self._set_prune_types()

    def _initialize_weights(self, sparse_init=True):
        if sparse_init:
            sparse_initialize(self)
        else:
            for name, module in self.named_modules():
                if isinstance(module, Conv2d):
                    init.kaiming_uniform_(module.weight)
                    if module.bias is not None:
                        init.constant_(module.bias, 0)

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.output(x)
        return x

    def _set_prune_types(self):
        self.pruned_types = {}
        unit_list = []
        # non_sequential_conv = []
        for name, module in self.named_modules():
            if isinstance(module, ShuffleUnit):
                unit_list.append(module)
        length = len(unit_list)
        for index, unit in enumerate(unit_list):
            self.pruned_types[unit.compress_conv1] = 'in'
            self.pruned_types[unit.expand_conv3] = 'vgg_out'

    def clean(self):
        with torch.no_grad():
            self.cuda()
            self._initialize_weights()
            self(torch.randn(64, 3, 32, 32).cuda())


def get_shufflenet(groups, width_scale, channels_per_layers, **kwargs):

    init_block_channels = 24
    layers = [4, 8, 4]

    channels = [[ci] * li for (ci, li) in zip(channels_per_layers, layers)]

    if width_scale != 1.0:
        channels = [[int(cij * width_scale) for cij in ci] for ci in channels]
        init_block_channels = int(init_block_channels * width_scale)

    net = ShuffleNet(channels=channels, init_block_channels=init_block_channels, groups=groups, **kwargs)

    return net


def shufflenet_g1(groups, width_factor=1, **kwargs):
    """
    ShuffleNet 1x (g=1) model from 'ShuffleNet: An Extremely Efficient Convolutional Neural Network for Mobile Devices,'
    """
    channels_per_layers = [144, 288, 576]
    return get_shufflenet(groups=groups, width_scale=width_factor, channels_per_layers=channels_per_layers, **kwargs)


def shufflenet_g3(groups, width_factor=1, **kwargs):
    """
    ShuffleNet 1x (g=3) model from 'ShuffleNet: An Extremely Efficient Convolutional Neural Network for Mobile Devices,'
    """
    channels_per_layers = [240, 480, 960]
    return get_shufflenet(groups=groups, width_scale=width_factor, channels_per_layers=channels_per_layers, **kwargs)


def shufflenet_g8(groups, width_factor=1, **kwargs):
    """
    ShuffleNet 1x (g=8) model from 'ShuffleNet: An Extremely Efficient Convolutional Neural Network for Mobile Devices,'
    """
    channels_per_layers = [384, 768, 1536]
    return get_shufflenet(groups=groups, width_scale=width_factor, channels_per_layers=channels_per_layers, **kwargs)


def _test():
    import torch

    models = [
        shufflenet_g1_w1,
        shufflenet_g3_w1,
        shufflenet_g8_w1,
    ]

    for model in models:

        net = model(groups=1)

        # net.train()
        net.eval()
        x = torch.randn(1, 3, 32, 32)
        y = net(x)
        y.sum().backward()
        assert (tuple(y.size()) == (1, 10))


if __name__ == "__main__":
    _test()
