"""
    ResNet for ImageNet-1K, implemented in PyTorch.
    Original paper: 'Deep Residual Learning for Image Recognition,' https://arxiv.org/abs/1512.03385.
    Adapted from https://github.com/osmr/imgclsmob/blob/master/pytorch/pytorchcv/models/efficientnet.py
"""

import os
import torch
import torch.nn as nn
import torch.nn.init as init
from Layers.layers import conv1x1_block, conv3x3_block, conv7x7_block, Conv2d, Linear, sparse_initialize, PaddedIdentityAdd


class ResBlock(nn.Module):
    def __init__(self, in_channels, out_channels, stride, bias=False, use_bn=True):
        super(ResBlock, self).__init__()
        assert not bias
        assert use_bn
        self.conv1 = conv3x3_block(in_channels=in_channels, out_channels=out_channels, stride=stride, shuffle=True)
        self.conv2 = conv3x3_block(in_channels=out_channels, out_channels=out_channels, activation=None)
        self.groups = 1

    def forward(self, x):
        x = self.conv1(x)
        x = self.conv2(x)
        return x


class ResBottleneck(nn.Module):
    def __init__(self, in_channels, out_channels, stride, padding=1, dilation=1, conv1_stride=False, bottleneck_factor=4):
        super(ResBottleneck, self).__init__()
        mid_channels = out_channels // bottleneck_factor

        self.conv1 = conv1x1_block(in_channels=in_channels, out_channels=mid_channels, stride=(stride if conv1_stride else 1))
        self.conv2 = conv3x3_block(in_channels=mid_channels, out_channels=mid_channels, stride=(1 if conv1_stride else stride), padding=padding)
        self.conv3 = conv1x1_block(in_channels=mid_channels, out_channels=out_channels, activation=None)

    def forward(self, x):
        x = self.conv1(x)
        x = self.conv2(x)
        x = self.conv3(x)
        return x


class ResUnit(nn.Module):
    def __init__(self, in_channels, out_channels, stride, padding=1, dilation=1, bias=False, use_bn=True, bottleneck=True, conv1_stride=False):
        super(ResUnit, self).__init__()
        self.resize_identity = (in_channels != out_channels) or (stride != 1)

        assert use_bn
        # assert not bottleneck
        assert not bias
        if bottleneck:
            self.body = ResBottleneck(in_channels=in_channels,
                                      out_channels=out_channels,
                                      stride=stride,
                                      padding=padding,
                                      dilation=dilation,
                                      conv1_stride=conv1_stride)
        else:
            self.body = ResBlock(in_channels=in_channels, out_channels=out_channels, stride=stride, bias=bias)
        if self.resize_identity:
            self.identity_conv = conv1x1_block(in_channels=in_channels, out_channels=out_channels, stride=stride, bias=bias, activation=None)
        self.activ = nn.ReLU(inplace=True)
        self.identity = PaddedIdentityAdd()

    def forward(self, x):
        if self.resize_identity:
            identity = self.identity_conv(x)
        else:
            identity = x
        x = self.body(x)
        # x = x + identity
        x = self.identity(x, identity)
        x = self.activ(x)
        return x


class ResInitBlock(nn.Module):
    def __init__(self, in_channels, out_channels):
        super(ResInitBlock, self).__init__()
        self.conv = conv7x7_block(in_channels=in_channels, out_channels=out_channels, stride=2)
        self.pool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)

    def forward(self, x):
        x = self.conv(x)
        x = self.pool(x)
        return x


class ResNet(nn.Module):
    def __init__(self, channels, init_block_channels, bottleneck, conv1_stride, in_channels=3, in_size=(224, 224), num_classes=1000):
        super(ResNet, self).__init__()
        self.in_size = in_size
        self.num_classes = num_classes
        self.bottleneck = bottleneck

        self.features = nn.Sequential()
        self.features.add_module("init_block", ResInitBlock(in_channels=in_channels, out_channels=init_block_channels))
        in_channels = init_block_channels
        for i, channels_per_stage in enumerate(channels):
            stage = nn.Sequential()
            for j, out_channels in enumerate(channels_per_stage):
                stride = 2 if (j == 0) and (i != 0) else 1
                stage.add_module(
                    "unit{}".format(j + 1),
                    ResUnit(in_channels=in_channels, out_channels=out_channels, stride=stride, bottleneck=bottleneck, conv1_stride=conv1_stride))
                in_channels = out_channels
            self.features.add_module("stage{}".format(i + 1), stage)
        self.features.add_module("final_pool", nn.AvgPool2d(kernel_size=7, stride=1))

        self.output = Linear(in_features=in_channels, out_features=num_classes)
        self._initialize_weights(sparse_init=False)
        self.set_prune_types()

    def _initialize_weights(self, sparse_init=True):
        if sparse_init:
            sparse_initialize(self)
        else:
            for name, module in self.named_modules():
                if isinstance(module, Conv2d):
                    init.kaiming_uniform_(module.weight)
                    if module.bias is not None:
                        init.constant_(module.bias, 0)

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.output(x)
        return x

    def set_prune_types(self):
        self.pruned_types = {}
        unit_list = []
        for name, module in self.named_modules():
            if isinstance(module, ResUnit):
                unit_list.append(module)
        length = len(unit_list)
        self.pruned_types[self.features.init_block.conv] = 'out'
        for index, unit in enumerate(unit_list):
            self.pruned_types[unit.body.conv1] = 'in'
            self.pruned_types[unit.body.conv2] = 'vgg_out'
            if self.bottleneck:
                self.pruned_types[unit.body.conv3] = 'vgg_out'
            if unit.resize_identity:
                self.pruned_types[unit.identity_conv] = 'skip'

    def clean(self):
        with torch.no_grad():
            self.cuda()
            self._initialize_weights()
            self(torch.randn(64, 3, 224, 224).cuda())


def get_resnet(blocks, bottleneck=None, conv1_stride=True, width_scale=1.0, **kwargs):
    if bottleneck is None:
        bottleneck = (blocks >= 50)

    if blocks == 18:
        layers = [2, 2, 2, 2]
    elif blocks == 34:
        layers = [3, 4, 6, 3]
    elif blocks == 50:
        layers = [3, 4, 6, 3]
    else:
        raise ValueError("Unsupported ResNet with number of blocks: {}".format(blocks))

    if bottleneck:
        assert (sum(layers) * 3 + 2 == blocks)
    else:
        assert (sum(layers) * 2 + 2 == blocks)

    init_block_channels = 64
    channels_per_layers = [64, 128, 256, 512]

    if bottleneck:
        bottleneck_factor = 4
        channels_per_layers = [ci * bottleneck_factor for ci in channels_per_layers]

    channels = [[ci] * li for (ci, li) in zip(channels_per_layers, layers)]

    if width_scale != 1.0:
        channels = [[int(cij * width_scale) if (i != len(channels) - 1) or (j != len(ci) - 1) else cij for j, cij in enumerate(ci)]
                    for i, ci in enumerate(channels)]
        init_block_channels = int(init_block_channels * width_scale)

    net = ResNet(channels=channels, init_block_channels=init_block_channels, bottleneck=bottleneck, conv1_stride=conv1_stride, **kwargs)

    return net


def resnet18(groups=1, width_factor=1.0, **kwargs):
    assert groups == 1
    return get_resnet(blocks=18, width_scale=width_factor, **kwargs)


def resnet34(groups=1, width_factor=1.0, **kwargs):
    assert groups == 1
    return get_resnet(blocks=34, width_scale=width_factor, **kwargs)


def resnet50(groups=1, width_factor=1.0, **kwargs):
    assert groups == 1
    return get_resnet(blocks=50, width_scale=width_factor, **kwargs)


def _test():
    import torch

    models = [
        resnet18,
        resnet34,
        resnet50,
    ]

    for model in models:

        net = model()

        # net.train()
        net.eval()
        batch = 4
        x = torch.randn(batch, 3, 224, 224)
        y = net(x)
        y.sum().backward()
        print(y.shape)
        assert (tuple(y.size()) == (batch, 1000))


if __name__ == "__main__":
    _test()
