#
# Copyright (C) 2023, Inria
# GRAPHDECO research group, https://team.inria.fr/graphdeco
# All rights reserved.
#
# This software is free for non-commercial, research and evaluation use 
# under the terms of the LICENSE.md file.
#
# For inquiries contact  george.drettakis@inria.fr
#

import torch
from scene import Scene, DeformModel, DeformEnvModel
import os
from tqdm import tqdm
from os import makedirs
from gaussian_renderer import render, render_lighting
import torchvision
from utils.general_utils import safe_state, safe_normalize, reflect
from argparse import ArgumentParser
from arguments import ModelParams, PipelineParams, get_combined_args, OptimizationParams
from gaussian_renderer import GaussianModel
from scene.NVDIFFREC.util import save_image_raw
import imageio
import numpy as np

def render_lightings(model_path, name, iteration, gaussians, sample_num):
    lighting_path = os.path.join(model_path, name, "ours_{}".format(iteration))
    makedirs(lighting_path, exist_ok=True)    
    # sampled_indicies = torch.randperm(gaussians.get_xyz.shape[0])[:sample_num]
    sampled_indicies = torch.arange(gaussians.get_xyz.shape[0], dtype=torch.long)[:sample_num]
    for sampled_index in tqdm(sampled_indicies, desc="Rendering lighting progress"):
        lighting = render_lighting(gaussians, resolution=(gaussians.brdf_envmap_res, gaussians.brdf_envmap_res * 2), sampled_index=sampled_index)
        torchvision.utils.save_image(lighting, os.path.join(lighting_path, '{0:05d}'.format(sampled_index) + ".png"))
        # save_image_raw(os.path.join(lighting_path, '{0:05d}'.format(sampled_index) + ".hdr"), lighting.permute(1,2,0).detach().cpu().numpy())


def render_set(model_path, opt, name, iteration, views, gaussians, pipeline, background, deform, deform_env):
    render_path = os.path.join(model_path, name, "ours_{}".format(iteration), "renders")
    gts_path = os.path.join(model_path, name, "ours_{}".format(iteration), "gt")

    # lighting_path = os.path.join(model_path, name, "ours_{}".format(iteration),"lighting")
    # makedirs(lighting_path, exist_ok=True) 
   

    makedirs(render_path, exist_ok=True)
    makedirs(gts_path, exist_ok=True)
    tensor_list = []

    for idx, view in enumerate(tqdm(views, desc="Rendering progress")):
        view.load2device()
        fid = view.fid
        xyz = gaussians.get_xyz
        time_input = fid.unsqueeze(0).expand(xyz.shape[0], -1)
        d_xyz, d_rotation, d_scaling = deform.step(xyz.detach(), time_input)
        gb_pos = gaussians.get_xyz + d_xyz # (N, 3) 
        view_pos = view.camera_center.repeat(gaussians.get_opacity.shape[0], 1) # (N, 3) 
        d_viewdir_normalized = safe_normalize(view_pos - gb_pos)
        normal, deform_delta_normal = gaussians.get_normal(gaussians.get_scaling, gaussians.get_rotation, d_scaling, d_rotation, d_viewdir_normalized) # (N, 3)
        reflvec = safe_normalize(reflect(d_viewdir_normalized, normal))
        d_reflvec = deform_env.step(reflvec.detach(), time_input)        
        # tensor_list.append(deform_cube)        
        results = render(view, gaussians, pipeline, background, d_xyz, d_rotation, d_scaling, d_reflvec, iteration, opt)
        rendering = results["render"]
        for k in results.keys():
            if results[k].dim()<3 or k=="render" or k=="delta_normal_norm":
                continue
            save_path = os.path.join(model_path, name, "ours_{}".format(iteration), k)
            makedirs(save_path, exist_ok=True)
            if "normal" in k:
                results[k] = 0.5 + (0.5*results[k])
            torchvision.utils.save_image(results[k], os.path.join(save_path, '{0:05d}'.format(idx) + ".png"))

        gt = view.original_image[0:3, :, :]
        torchvision.utils.save_image(rendering, os.path.join(render_path, '{0:05d}'.format(idx) + ".png"))
        torchvision.utils.save_image(gt, os.path.join(gts_path, '{0:05d}'.format(idx) + ".png"))

    # different_indices = []
    # reference_tensor = tensor_list[0]
    # for i, tensor in enumerate(tensor_list[1:], start=1):
    #     if not torch.equal(reference_tensor, tensor):
    #         different_indices.append(i)

    # 輸出結果
    # if different_indices:
    #     print(f"以下索引的張量與第一個張量不同: {different_indices}")
    # else:
    #     print("所有張量都相等")

            

def render_sets(dataset: ModelParams, opt: OptimizationParams, iteration: int, pipeline: PipelineParams, skip_train: bool, skip_test: bool,
                mode: str):
    with torch.no_grad():
        gaussians = GaussianModel(dataset.sh_degree, dataset.brdf_envmap_res)
        scene = Scene(dataset, gaussians, load_iteration=iteration, shuffle=False)
        deform = DeformModel()
        deform.load_weights(dataset.model_path)
        deform_env = DeformEnvModel()
        deform_env.load_weights(dataset.model_path)

        bg_color = [1, 1, 1] if dataset.white_background else [0, 0, 0]
        background = torch.tensor(bg_color, dtype=torch.float32, device="cuda")

        if mode == "render":
            render_func = render_set

        if not skip_train:
            render_func(dataset.model_path, opt, "train", scene.loaded_iter,
                        scene.getTrainCameras(), gaussians, pipeline,
                        background, deform, deform_env)

        if not skip_test:
            render_func(dataset.model_path, opt, "test", scene.loaded_iter,
                        scene.getTestCameras(), gaussians, pipeline,
                        background, deform, deform_env)
            
        render_lightings(dataset.model_path, "lighting", scene.loaded_iter, gaussians, sample_num=1)

if __name__ == "__main__":
    # Set up command line argument parser
    parser = ArgumentParser(description="Testing script parameters")
    model = ModelParams(parser, sentinel=True)
    op = OptimizationParams(parser)
    pipeline = PipelineParams(parser)
    parser.add_argument("--iteration", default=-1, type=int)
    parser.add_argument("--skip_train", action="store_true")
    parser.add_argument("--skip_test", action="store_true")
    parser.add_argument("--quiet", action="store_true")
    parser.add_argument("--mode", default='render', choices=['render', 'time', 'view', 'all', 'pose', 'original'])
    args = get_combined_args(parser)
    print("Rendering " + args.model_path)

    # Initialize system state (RNG)
    safe_state(args.quiet)

    render_sets(model.extract(args), op.extract(args), args.iteration, pipeline.extract(args), args.skip_train, args.skip_test, args.mode)
