"""
Configuration management for Bedrock Claude client.
"""

import os
from typing import Optional, Dict, Any
from dataclasses import dataclass
from dotenv import load_dotenv

from .exceptions import BedrockConfigError


@dataclass
class BedrockConfig:
    """Configuration class for Bedrock Claude client."""
    
    # AWS Configuration
    aws_region: str = "us-east-1"
    aws_access_key_id: Optional[str] = None
    aws_secret_access_key: Optional[str] = None
    aws_session_token: Optional[str] = None
    aws_profile: Optional[str] = None
    
    # Model Configuration
    default_model_id: str = "anthropic.claude-sonnet-4-20250514-v1:0" #"anthropic.claude-3-5-sonnet-20240620-v1:0"
    default_max_tokens: int = 4096
    default_temperature: float = 0.7
    default_top_p: float = 0.9
    
    # Client Configuration
    timeout: int = 300  # 5 minutes
    retry_attempts: int = 3
    enable_logging: bool = True
    
    @classmethod
    def from_env(cls, env_file: Optional[str] = None) -> "BedrockConfig":
        """
        Create configuration from environment variables.
        
        Args:
            env_file: Path to .env file (optional)
            
        Returns:
            BedrockConfig instance
        """
        if env_file:
            load_dotenv(env_file)
        else:
            load_dotenv()
            
        return cls(
            aws_region=os.getenv("AWS_REGION", "us-east-1"),
            aws_access_key_id=os.getenv("AWS_ACCESS_KEY_ID"),
            aws_secret_access_key=os.getenv("AWS_SECRET_ACCESS_KEY"),
            aws_session_token=os.getenv("AWS_SESSION_TOKEN"),
            aws_profile=os.getenv("AWS_PROFILE"),
            default_model_id=os.getenv("BEDROCK_DEFAULT_MODEL","us.anthropic.claude-3-7-sonnet-20250219-v1:0"), # "anthropic.claude-3-5-sonnet-20240620-v1:0"), #anthropic.claude-sonnet-4-20250514-v1:0
            default_max_tokens=int(os.getenv("BEDROCK_MAX_TOKENS", "4096")),
            default_temperature=float(os.getenv("BEDROCK_TEMPERATURE", "0.7")),
            default_top_p=float(os.getenv("BEDROCK_TOP_P", "0.9")),
            timeout=int(os.getenv("BEDROCK_TIMEOUT", "300")),
            retry_attempts=int(os.getenv("BEDROCK_RETRY_ATTEMPTS", "3")),
            enable_logging=os.getenv("BEDROCK_ENABLE_LOGGING", "true").lower() == "true"
        )
    
    def validate(self) -> None:
        """
        Validate configuration settings.
        
        Raises:
            BedrockConfigError: If configuration is invalid
        """
        if not self.aws_region:
            raise BedrockConfigError("AWS region must be specified")
            
        if self.default_max_tokens <= 0:
            raise BedrockConfigError("Max tokens must be positive")
            
        if not 0 <= self.default_temperature <= 1:
            raise BedrockConfigError("Temperature must be between 0 and 1")
            
        if not 0 <= self.default_top_p <= 1:
            raise BedrockConfigError("Top_p must be between 0 and 1")
            
        if self.timeout <= 0:
            raise BedrockConfigError("Timeout must be positive")
            
        if self.retry_attempts < 0:
            raise BedrockConfigError("Retry attempts must be non-negative")
    
    def get_aws_session_kwargs(self) -> Dict[str, Any]:
        """
        Get AWS session configuration as keyword arguments.
        
        Returns:
            Dictionary of AWS session configuration
        """
        kwargs = {"region_name": self.aws_region}
        
        if self.aws_access_key_id and self.aws_secret_access_key:
            kwargs.update({
                "aws_access_key_id": self.aws_access_key_id,
                "aws_secret_access_key": self.aws_secret_access_key
            })
            if self.aws_session_token:
                kwargs["aws_session_token"] = self.aws_session_token
        elif self.aws_profile:
            kwargs["profile_name"] = self.aws_profile
            
        return kwargs


# Available Claude models in Bedrock (using on-demand compatible IDs)
CLAUDE_MODELS = {
    "claude-3-5-sonnet": "anthropic.claude-3-5-sonnet-20240620-v1:0",
    "claude-3-5-haiku": "anthropic.claude-3-5-haiku-20241022-v1:0",
    "claude-3-opus": "anthropic.claude-3-opus-20240229-v1:0",
    "claude-3-sonnet": "anthropic.claude-3-sonnet-20240229-v1:0",
    "claude-3-haiku": "anthropic.claude-3-haiku-20240307-v1:0",
    "claude-3-7-sonnet": "us.anthropic.claude-3-7-sonnet-20250219-v1:0",
    "claude-4": "anthropic.claude-sonnet-4-20250514-v1:0"
}

# Models that support thinking parameters
THINKING_CAPABLE_MODELS = {
    "us.anthropic.claude-3-7-sonnet-20250219-v1:0",
    "anthropic.claude-sonnet-4-20250514-v1:0",
    # Add future thinking-capable models here
}

# Valid thinking modes
VALID_THINKING_MODES = {
    "reasoning", "analysis", "exploration", "verification", "planning"
}

def supports_thinking(model_id: str) -> bool:
    """
    Check if a model supports thinking parameters.
    
    Args:
        model_id: The model identifier
        
    Returns:
        True if the model supports thinking parameters
    """
    return model_id in THINKING_CAPABLE_MODELS

# Default system prompts for different use cases
DEFAULT_SYSTEM_PROMPTS = {
    "assistant": "You are Claude, a helpful AI assistant created by Anthropic. You are thoughtful, harmless, and honest.",
    "coding": "You are Claude, an AI assistant specialized in software development. You write clean, efficient, and well-documented code.",
    "analysis": "You are Claude, an AI assistant specialized in analysis and research. You provide thorough, accurate, and well-structured analysis."
}
