#!/bin/bash

# Whisper ASR Evaluation - Quick Test Script
# This script runs a quick test of all model variants with 10 samples each

echo "🚀 Starting Whisper ASR Evaluation Quick Test"
echo "=============================================="

# Check if we're in the right directory
if [ ! -d "decode_scripts" ]; then
    echo "❌ Error: Please run this script from the reproducibility/ directory"
    echo "   cd reproducibility && ./run_test.sh"
    exit 1
fi

# Check if Python is available
if ! command -v python &> /dev/null; then
    echo "❌ Error: Python not found. Please install Python 3.8+"
    exit 1
fi

# Check if virtual environment is activated (optional but recommended)
if [[ "$VIRTUAL_ENV" == "" ]]; then
    echo "⚠️  Warning: No virtual environment detected. Consider using:"
    echo "   python -m venv whisper_asr_env"
    echo "   source whisper_asr_env/bin/activate"
    echo "   pip install -r requirements.txt"
    echo ""
fi

echo "📋 Test Configuration:"
echo "   - Samples per model: 10"
echo "   - Models: Tiny, Base, Medium, Oracle"
echo "   - Slot-WER: Disabled (for speed)"
echo ""

# Array of models to test
models=("tiny" "base" "medium" "oracle")
model_descriptions=("Whisper-Tiny (fastest)" "Whisper-Base (balanced)" "Whisper-Medium (most accurate)" "Oracle Experiment (perfect prompts)")

# Test each model
for i in "${!models[@]}"; do
    model="${models[$i]}"
    description="${model_descriptions[$i]}"
    script="decode_scripts/comprehensive_asr_evaluation_improved_${model}.py"
    
    echo "🧪 Testing ${description}..."
    echo "   Script: ${script}"
    
    if [ ! -f "$script" ]; then
        echo "   ❌ Script not found: $script"
        continue
    fi
    
    # Run the test
    echo "   ▶️  Running test..."
    if python "$script" --test-run --max-samples 10; then
        echo "   ✅ ${description} test completed successfully"
    else
        echo "   ❌ ${description} test failed"
        echo "   💡 Check the error messages above for troubleshooting"
    fi
    echo ""
done

echo "🎉 Quick test completed!"
echo ""
echo "📊 Next Steps:"
echo "   1. Check the asr_experiments/ directory for results"
echo "   2. Run full evaluations: python decode_scripts/comprehensive_asr_evaluation_improved_medium.py"
echo "   3. Compare results across different model sizes"
echo ""
echo "📖 For more options, see: README.md"
