#include <fstream>
#include <cassert>
#include <iostream>
#include <string>
#include <cmath>

using namespace std;

const int MAX = 10000;
int graph[MAX][MAX];
double text_sim[MAX][MAX];
int text_weight[MAX][MAX];
const int HIST_BUCKETS = 100;
unsigned long long text_hist[HIST_BUCKETS];
int store[MAX];
int n, k, missing_edge_allowance;

// Check if vertices in store[0:b] are a clique
bool is_clique(int b) {
	int missing_edges = 0;
	for (int i = 0; i < b; i++) {
		for (int j = i + 1; j < b; j++) {
			if (graph[store[i]][store[j]] == 0) {
				missing_edges++;
			}
			if (graph[store[j]][store[i]] == 0) {
				missing_edges++;
			}
			if (missing_edges > missing_edge_allowance) {
				return false;
			}
		}
	}
	return true;
}

double clique_text_score(int b) {
	double total_sim = 0;
	int total_weight = 0;
	for (int i = 0; i < b; i++) {
		for (int j = i + 1; j < b; j++) {
			if (graph[store[i]][store[j]] == 1) {
				total_sim += text_sim[store[i]][store[j]];
				total_weight += text_weight[store[i]][store[j]];
			}
			if (graph[store[j]][store[i]] == 1) {
				total_sim += text_sim[store[j]][store[i]];
				total_weight += text_weight[store[j]][store[i]];
			}
		}
	}
	if (total_weight == 0) {
		return 0;
	}
	return total_sim / total_weight;
}

// Return the number of cliques containing vertices in store[0:l] by inserting vertices from [i, )
// i is the next available vertex to insert
// l is the current clique length
unsigned long long count_cliques(int i, int l) {
	if (l == k) {
		//assert(is_clique(k));
		double s = clique_text_score(k);
		assert(s >= 0 && s <= 1);
		int bucket = ceil(s * HIST_BUCKETS) - 1;
		if (bucket == -1) {
			assert(s == 0);
			bucket = 0;
		}
		assert(bucket >= 0 && bucket < HIST_BUCKETS);
		text_hist[bucket] += 1;
		return 1;
	}
	unsigned long long count = 0;
	for (int j = i; j <= n - (k - l); j++) {
		store[l] = j;
		if (is_clique(l + 1)) {
			count += count_cliques(j+1, l+1);
		}
	}
	return count;
}

void read_graph(string fname) {
	ifstream graph_file;
	graph_file.open(fname);

	string line;
	getline(graph_file, line);
	n = stoi(line);
	getline(graph_file, line);
	k = stoi(line);
	getline(graph_file, line);
	double gamma = stod(line);
	missing_edge_allowance = int((1 - gamma) * (k * (k-1)));

	assert(n <= MAX);

	while (getline(graph_file, line)) {
		size_t idx1 = line.find(' ');
		size_t idx2 = line.find(' ', idx1+1);
		int i = stoi(line.substr(0, idx1));
		int j = -1;
		double t = 0;
		int w = 1;
		if (idx2 == line.npos) {
			j = stoi(line.substr(idx1+1, line.length() - (idx1 + 1))); // idx1 + (len - idx1 - 1) + 1 = len
		} else { // counting by text-weight as well
			j = stoi(line.substr(idx1+1, idx2 - (idx1+1)));
			size_t idx3 = line.find(' ', idx2+1);
			t = stod(line.substr(idx2+1, idx3 - (idx2+1)));
			w = stoi(line.substr(idx3+1, line.length() - (idx3+1))); // idx1 + (idx2 - idx1-1) + (idx3 - idx2-1) + (len -idx3-1) + 3 = len
		}
		assert((i >= 0) && (i < n) && (j >= 0) && (j < n)); // 0-indexed
		graph[i][j] = 1;
		text_sim[i][j] = t;
		text_weight[i][j] = w;
	}
}


void write_result(unsigned long long count) {
	ofstream result_file;
	result_file.open("clique_result.txt");
	result_file << count << endl;
}

int main(int argc, char* argv[]) {
	assert(argc == 2);
	string graph_fname(argv[1]);
	read_graph(graph_fname);
	for (int i = 0; i < HIST_BUCKETS; i++) {
		text_hist[i] = 0;
	}
	unsigned long long count = count_cliques(0, 0);
	cout << count << endl;
	unsigned long long text_count = 0;
	for (int i = 0; i < HIST_BUCKETS; i++) {
		cout << text_hist[i] << endl;
		text_count += text_hist[i];
	}
	assert(count == text_count);
	return 0;
}
