import argparse
import os
import random
import time
from collections import deque
from distutils.util import strtobool
import copy

import envpool
import gym
import numpy as np
import torch
import torch.nn as nn
from torch.distributions.categorical import Categorical
from collections import Counter

import torchattacks
import matplotlib.pyplot as plt

import pickle as pk
from GNN import GCN_pooling_Net, GCN_pooling_Net_large, MyData
from MLP import MLP
from vanilla_vae import VanillaVAE

from torch.nn.functional import normalize

def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--save-dataset-name", type=str, default=None)
    parser.add_argument("--dataset-length", type=int, default=1000)

    parser.add_argument("--test-len", type=int, default=100)
    parser.add_argument("--load-agent-name", type=str, default=None,
                         help="filename of the saved agent model")
    parser.add_argument("--return-save-name", type=str, default='',
                        help="filename of the saved return")
    parser.add_argument("--fusion", type=str, default=None,
                        help="vote;gnn;mlp;gnn_nostate;mlp_nostate")
    parser.add_argument("--agg", type=str, default=None,
                        help="pn")

    parser.add_argument("--atk", type=str, default=None,
                        help="square, cw, onepixel, jitter")
    parser.add_argument("--eps", type=float, default=None,
                        help="eps for attack")

    parser.add_argument("--load-VAE-name", type=str, default=None)
    parser.add_argument("--load-MLP-name", type=str, default=None)
    parser.add_argument("--load-GNN-name", type=str, default=None)


    parser.add_argument("--env-id", type=str, default="RoadRunner-v5",
                        help="the id of the environment")
    parser.add_argument("--num-envs", type=int, default=8,
                        help="the number of parallel game environments") # rmb to set it to 1 while using gnn or mlp
    parser.add_argument("--seed", type=int, default=1,
                        help="seed of the experiment")


    parser.add_argument("--exp-name", type=str, default=os.path.basename(__file__).rstrip(".py"),
        help="the name of this experiment")
    parser.add_argument("--torch-deterministic", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="if toggled, `torch.backends.cudnn.deterministic=False`")
    parser.add_argument("--cuda", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="if toggled, cuda will be enabled by default")
    parser.add_argument("--track", type=lambda x: bool(strtobool(x)), default=False, nargs="?", const=True,
        help="if toggled, this experiment will be tracked with Weights and Biases")
    parser.add_argument("--wandb-project-name", type=str, default="cleanRL",
        help="the wandb's project name")
    parser.add_argument("--wandb-entity", type=str, default=None,
        help="the entity (team) of wandb's project")
    parser.add_argument("--capture-video", type=lambda x: bool(strtobool(x)), default=False, nargs="?", const=True,
        help="weather to capture videos of the agent performances (check out `videos` folder)")

    parser.add_argument("--total-timesteps", type=int, default=10000000,
        help="total timesteps of the experiments")
    parser.add_argument("--learning-rate", type=float, default=2.5e-4,
        help="the learning rate of the optimizer")
    parser.add_argument("--num_channel", type=int, default=4,
                        help="observation number channels")
    parser.add_argument("--num-steps", type=int, default=128,
        help="the number of steps to run in each environment per policy rollout")
    parser.add_argument("--anneal-lr", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggle learning rate annealing for policy and value networks")
    parser.add_argument("--gae", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Use GAE for advantage computation")
    parser.add_argument("--gamma", type=float, default=0.99,
        help="the discount factor gamma")
    parser.add_argument("--gae-lambda", type=float, default=0.95,
        help="the lambda for the general advantage estimation")
    parser.add_argument("--num-minibatches", type=int, default=4,
        help="the number of mini-batches")
    parser.add_argument("--update-epochs", type=int, default=4,
        help="the K epochs to update the policy")
    parser.add_argument("--norm-adv", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggles advantages normalization")
    parser.add_argument("--clip-coef", type=float, default=0.1,
        help="the surrogate clipping coefficient")
    parser.add_argument("--clip-vloss", type=lambda x: bool(strtobool(x)), default=True, nargs="?", const=True,
        help="Toggles whether or not to use a clipped loss for the value function, as per the paper.")
    parser.add_argument("--ent-coef", type=float, default=0.01,
        help="coefficient of the entropy")
    parser.add_argument("--vf-coef", type=float, default=0.5,
        help="coefficient of the value function")
    parser.add_argument("--max-grad-norm", type=float, default=0.5,
        help="the maximum norm for the gradient clipping")
    parser.add_argument("--target-kl", type=float, default=None,
        help="the target KL divergence threshold")
    args = parser.parse_args()
    args.batch_size = int(args.num_envs * args.num_steps)
    args.minibatch_size = int(args.batch_size // args.num_minibatches)
    # fmt: on
    return args


class RecordEpisodeStatistics(gym.Wrapper):
    def __init__(self, env, deque_size=100):
        super().__init__(env)
        self.num_envs = getattr(env, "num_envs", 1)
        self.episode_returns = None
        self.episode_lengths = None
        # get if the env has lives
        self.has_lives = False
        env.reset()
        info = env.step(np.zeros(self.num_envs, dtype=int))[-1]
        if info["lives"].sum() > 0:
            self.has_lives = True
            print("env has lives")

    def reset(self, **kwargs):
        observations = super().reset(**kwargs)
        self.episode_returns = np.zeros(self.num_envs, dtype=np.float32)
        self.episode_lengths = np.zeros(self.num_envs, dtype=np.int32)
        self.lives = np.zeros(self.num_envs, dtype=np.int32)
        self.returned_episode_returns = np.zeros(self.num_envs, dtype=np.float32)
        self.returned_episode_lengths = np.zeros(self.num_envs, dtype=np.int32)
        return observations

    def step(self, action):
        observations, rewards, dones, infos = super().step(action)
        self.episode_returns += infos["reward"]
        self.episode_lengths += 1
        self.returned_episode_returns[:] = self.episode_returns
        self.returned_episode_lengths[:] = self.episode_lengths
        all_lives_exhausted = infos["lives"] == 0
        if self.has_lives:
            self.episode_returns *= 1 - all_lives_exhausted
            self.episode_lengths *= 1 - all_lives_exhausted
        else:
            self.episode_returns *= 1 - dones
            self.episode_lengths *= 1 - dones
        infos["r"] = self.returned_episode_returns
        infos["l"] = self.returned_episode_lengths
        return (
            observations,
            rewards,
            dones,
            infos,
        )


def layer_init(layer, std=np.sqrt(2), bias_const=0.0):
    torch.nn.init.orthogonal_(layer.weight, std)
    torch.nn.init.constant_(layer.bias, bias_const)
    return layer

def agg_vote(actions):
    c = Counter(actions)
    value, count = c.most_common()[0]
    return value
def agg_pn(p_actions, n_actions):
    p = Counter(p_actions)
    n = Counter(n_actions)
    for i in p:
        if i in n:
            p[i] -= n[i]
    value, count = p.most_common()[0]
    return value
def agg_final(logits):
    main_probs = torch.nn.functional.softmax(logits[0], dim = 1)
    entropies = []
    best = []
    worst = []
    alpha = beta = 1/envs.single_action_space.n

    for i in range(1,len(logits)):
        entropies.append(Categorical(logits = logits[i].detach()).entropy().data.cpu().numpy()[0])
    for k in sorted(range(len(entropies)), key=lambda i: entropies[i])[-3:]:
        _, bb = torch.max(logits[k+1], 1)
        _, ww = torch.min(logits[k + 1], 1)
        best.append(bb.data[0])
        worst.append(ww.data[0])
    temp = torch.ones(main_probs.shape).to(device)
    for b in best:
        temp[0][b] += alpha
    for w in worst:
        temp[0][w] += beta
    _, action = torch.max(main_probs+temp, 1)
    return action, main_probs



class Agent(nn.Module):
    def __init__(self, envs, args, hidden_dims=[32, 64, 64, 64 * 7 * 7, 8, 4, 3, 512]):
        super().__init__()
        self.network = nn.Sequential(
            layer_init(nn.Conv2d(args.num_channel, hidden_dims[0], hidden_dims[4], stride=4)),
            nn.ReLU(),
            layer_init(nn.Conv2d(hidden_dims[0], hidden_dims[1], hidden_dims[5], stride=2)),
            nn.ReLU(),
            layer_init(nn.Conv2d(hidden_dims[1], hidden_dims[2], hidden_dims[6], stride=1)),
            nn.ReLU(),
            nn.Flatten(),
            layer_init(nn.Linear(hidden_dims[3], hidden_dims[7])),
            nn.ReLU(),
        )
        self.actor = layer_init(nn.Linear(512, envs.single_action_space.n), std=0.01)
        self.critic = layer_init(nn.Linear(512, 1), std=1)

    def get_value(self, x):
        if torch.max(x) > 1.0:
            x_scale = x/255.0
        else:
            x_scale = x
        return self.critic(self.network(x_scale))

    def get_action_and_value(self, x, action=None):
        if torch.max(x) > 1.0:
            x_scale = x/255.0
        else:
            x_scale = x
        hidden = self.network(x_scale)
        logits = self.actor(hidden)
        probs = Categorical(logits=logits)
        if action is None:
            action = probs.sample()
        return action, probs.log_prob(action), probs.entropy(), self.critic(hidden), logits
    def forward(self, x):
        if torch.max(x) <= 1.0:
            x_scale = x/255.0
        else:
            x_scale = x
        hidden = self.network(x_scale)
        logits = self.actor(hidden)
        return logits

if __name__ == "__main__":
    args = parse_args()
    if args.load_VAE_name is None:
        args.load_VAE_name = args.env_id + '_'
    if args.load_MLP_name is None:
        args.load_MLP_name = args.env_id + '_'
    if args.load_GNN_name is None:
        args.load_GNN_name = args.env_id + '_'

    prefix = ''
    if args.fusion is not None:
        prefix += args.fusion + '_'
    else:
        prefix += args.load_agent_name
    if args.agg is not None:
        prefix += args.agg + '_'
    if args.atk is not None:
        prefix += args.atk + '_'
    args.return_save_name = prefix + args.return_save_name

    if args.eps is None:
        if args.env_id == 'RoadRunner-v5':
            args.eps = 0.002
        elif args.env_id =='Riverraid-v5':
            args.eps = 0.01
        elif args.env_id == 'SpaceInvaders-v5':
            args.eps = 0.002
        else:
            raise TypeError ('env-id not recognized')
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    if args.track:
        import wandb

        wandb.init(
            project=args.wandb_project_name,
            entity=args.wandb_entity,
            sync_tensorboard=True,
            config=vars(args),
            name=run_name,
            monitor_gym=True,
            save_code=True,
        )
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = envpool.make(
        args.env_id,
        env_type="gym",
        num_envs=args.num_envs,
        episodic_life=True,
        reward_clip=True,
        gray_scale=True,
        stack_num=4
    )
    envs.num_envs = args.num_envs
    envs.single_action_space = envs.action_space
    envs.single_observation_space = envs.observation_space
    envs = RecordEpisodeStatistics(envs)
    assert isinstance(envs.action_space, gym.spaces.Discrete), "only discrete action space is supported"

    agent = Agent(envs, args = args).to(device)
    if args.load_agent_name is not None:
        if args.load_agent_name == 'student_':
            agent = Agent(envs, args = args, hidden_dims = [16, 32, 32, 32*7*7, 8, 4, 3, 512]).to(device)
            load_agent_name = args.load_agent_name.split(',')
            agent.load_state_dict(torch.load('saved_models/' + args.env_id + '_' + load_agent_name[0]))
            agent.eval()
            agents = [agent]
        else:
            load_agent_name = args.load_agent_name.split(',')
            if len(load_agent_name) == 1:
                agent.load_state_dict(torch.load('saved_models/'+args.env_id+'_'+load_agent_name[0]))
                agent.eval()
                agents = [agent]
            else:
                agents = []
                for i in range(len(load_agent_name)):
                    agent_i = copy.deepcopy(agent)
                    agent_i.load_state_dict(torch.load('saved_models/'+args.env_id+'_'+load_agent_name[i]))
                    agent_i.eval()
                    agents.append(agent_i)
        print ('\n ***** Agent Model Loaded *******\n')


    class Get_action(nn.Module):
        def __init__(self):
            super(Get_action, self).__init__()
            self.tradeoff_gnn = torch.nn.parameter.Parameter(data=torch.tensor([3.0]), requires_grad=True)
            self.tradeoff_mlp = torch.nn.parameter.Parameter(data=torch.tensor([3.0]), requires_grad=True)

        def parameters(self):
            return iter([self.tradeoff_mlp, self.tradeoff_gnn])

        def cal(self, next_obs):
            actions_all = []

            logits_temp = None
            for agent_i in agents:
                if args.atk is not None:
                    action, logprob, _, value, logits = agent_i.get_action_and_value(next_obs)

                else:
                    action, logprob, _, value, logits = agent_i.get_action_and_value(next_obs)

                if (args.save_dataset_name is not None) or (args.fusion is not None):
                    if logits_temp is None:
                        logits_temp = logits.unsqueeze(1)
                    else:
                        logits_temp = torch.cat((logits_temp, logits.unsqueeze(1)), 1)
                actions_all.append(action)
            if args.fusion is None:
                out = logits
                action = actions_all[0]
            elif args.fusion == 'vote':
                out = torch.mean(logits_temp, dim = 1)
                action = agg_vote(actions_all)
            elif args.fusion == 'mlp':
                if args.env_id == 'RoadRunner-v5':
                    obs_hidden = VAE(next_obs)[-1][0]
                    data = torch.cat((logits_temp[0][1], logits_temp[0][2], obs_hidden))
                    out_mlp = MLP(data).unsqueeze(0)
                    out_mlp = normalize(out_mlp, p=2.0, dim=1)
                    out = out_mlp + self.tradeoff_mlp*normalize(logits_temp[:, 0], p=2.0, dim=1)
                    _, action = torch.max(out, 1)
                elif args.env_id in ['Riverraid-v5', 'SpaceInvaders-v5']:
                    obs_hidden = VAE(next_obs)[-1][0]
                    obs_hidden = torch.cat((obs_hidden,torch.zeros(taskid_digits).to(device)), dim=0)

                    actions_fusion = []
                    actions_fusion_n = []
                    logits_fusion = []
                    for t in range(2, len(logits_temp[0])):
                        feature_1 = torch.cat((logits_temp[0][1], (torch.ones(taskid_digits)*(t/len(logits_temp[0]))).to(device)), dim=0)
                        feature_2 = torch.cat((logits_temp[0][t], (torch.ones(taskid_digits)*(t/len(logits_temp[0]))).to(device)),
                                              dim=0)
                        data = torch.cat((feature_1, feature_2, obs_hidden))
                        out_mlp = MLP(data).unsqueeze(0)
                        out_mlp = normalize(out_mlp, p=2.0, dim=1)
                        out = out_mlp + self.tradeoff_mlp * normalize(logits_temp[:, 0], p=2.0, dim=1)
                        _, action = torch.max(out, 1)
                        actions_fusion.append(action)
                        _, action_n = torch.min(out, 1)
                        actions_fusion_n.append(action_n)
                        logits_fusion.append(out)
                    if args.agg is None:
                        action = agg_vote(actions_fusion) # vote
                    elif args.agg == 'pn':
                        action = agg_pn(actions_fusion, actions_fusion_n)
                else:
                    raise TypeError('env-id not recognized')
            elif args.fusion == 'gnn':
                if args.env_id == 'RoadRunner-v5':
                    obs_hidden = VAE(next_obs)[-1][0]
                    x = torch.stack(
                        (logits_temp[0][1], logits_temp[0][2], obs_hidden, and_feature))
                    edge_index = torch.tensor([[0, 1, 2], [3,3,3]])
                    data = MyData(x=x, edge_index=edge_index.detach()).to(device)
                    out_gnn = GNN(data)[-1].unsqueeze(0)

                    out_gnn = normalize(out_gnn, p=2.0, dim=1)
                    out = out_gnn + self.tradeoff_gnn * normalize(logits_temp[:, 0], p=2.0, dim=1)
                    _, action = torch.max(out, 1)
                elif args.env_id in ['Riverraid-v5', 'SpaceInvaders-v5']:
                    obs_hidden = VAE(next_obs)[-1][0]
                    obs_hidden = torch.cat((obs_hidden, torch.zeros(taskid_digits).to(device)), dim=0)

                    actions_fusion = []
                    actions_fusion_n = []
                    logits_fusion = []
                    for t in range(2, len(logits_temp[0])):
                        feature_1 = torch.cat((logits_temp[0][1], (torch.ones(taskid_digits)*(t/len(logits_temp[0]))).to(device)),
                                              dim=0)
                        feature_2 = torch.cat((logits_temp[0][t], (torch.ones(taskid_digits)*(t/len(logits_temp[0]))).to(device)),
                                              dim=0)

                        x = torch.stack(
                            (feature_1, feature_2, obs_hidden, and_feature))
                        edge_index = torch.tensor([[0, 1, 2], [3, 3, 3]])
                        data = MyData(x=x, edge_index=edge_index.detach()).to(device)
                        out_gnn = GNN(data)[-1].unsqueeze(0)
                        out_gnn = normalize(out_gnn, p=2.0, dim=1)
                        out = out_gnn + self.tradeoff_gnn * normalize(logits_temp[:, 0], p=2.0, dim=1)
                        _, action = torch.max(out, 1)
                        actions_fusion.append(action)
                        _, action_n = torch.min(out, 1)
                        actions_fusion_n.append(action)
                        logits_fusion.append(out)
                    if args.agg is None:
                        action, out = agg_final(logits_fusion)
                    elif args.agg == 'pn':
                        action = agg_pn(actions_fusion,actions_fusion_n)
                else:
                    raise TypeError('env-id not recognized')

            return action, out, logits_temp

        def forward(self, next_obs):
            action, logits, logits_temp = self.cal(next_obs)
            return logits
    get_action = Get_action().to(device)

    if args.atk is not None:
        if args.atk == 'cw':
            atk = torchattacks.CW(get_action, c=1e-4, kappa=0, steps=5, lr=0.01)
        elif args.atk == 'fgsm':
            atk = torchattacks.FGSM(get_action, eps=args.eps)
        elif args.atk == 'pgd':
            atk = torchattacks.PGD(get_action, eps=args.eps, alpha=2 / 255, steps=5)
        elif args.atk == 'jitter':
            atk = torchattacks.Jitter(get_action, eps=args.eps, alpha=2 / 255, steps=5
                                      , scale=10, std=0.1, random_start=True)
        elif args.atk == 'square':
            atk = torchattacks.Square(get_action, eps=args.eps, n_queries=5, n_restarts=5, loss='ce') # blackbox
        else:
            raise TypeError("Attack method not recognized")



        atk_img_saved = False

    if args.env_id == 'RoadRunner-v5':
        GNN_input_dim = envs.single_action_space.n
        MLP_input_dim = envs.single_action_space.n*3
    elif args.env_id in ['Riverraid-v5', 'SpaceInvaders-v5']:
        gnn_taskid_digits = 10
        mlp_taskid_digits = 1
        GNN_input_dim = envs.single_action_space.n + gnn_taskid_digits
        MLP_input_dim = (envs.single_action_space.n + mlp_taskid_digits)*3
        if args.fusion == 'mlp':
            taskid_digits = mlp_taskid_digits
        elif args.fusion == 'gnn':
            taskid_digits = gnn_taskid_digits
    else:
        raise TypeError('env-id not recognized')


    if args.fusion =='gnn':
        if args.env_id == 'RoadRunner-v5':
            GNN = GCN_pooling_Net(GNN_input_dim, envs.single_action_space.n).to(device)
        elif args.env_id in ['Riverraid-v5', 'SpaceInvaders-v5']:
            GNN = GCN_pooling_Net_large(GNN_input_dim, envs.single_action_space.n).to(device)
        else:
            raise TypeError('env-id not recognized')
        GNN.load_state_dict(torch.load('saved_models/GNN_'+args.load_GNN_name))
        GNN.eval()
        and_feature = torch.ones(GNN_input_dim).to(device)
        global_feature = torch.zeros(GNN_input_dim).to(device)
        VAE = VanillaVAE(4, envs.single_action_space.n, hidden_dims=[32, 64, 128, 256, 512], out_dim_temp=3).to(device)
        VAE.load_state_dict(torch.load('saved_models/VAE_'+args.load_VAE_name))
        VAE.eval()
    elif args.fusion == 'mlp':
        MLP = MLP(MLP_input_dim, 64, envs.single_action_space.n).to(device)
        MLP.load_state_dict(torch.load('saved_models/MLP_'+args.load_MLP_name))
        MLP.eval()
        VAE = VanillaVAE(4, envs.single_action_space.n, hidden_dims=[32, 64, 128, 256, 512], out_dim_temp=3).to(device)
        VAE.load_state_dict(torch.load('saved_models/VAE_'+args.load_VAE_name))
        VAE.eval()

    # ALGO Logic: Storage setup
    obs = torch.zeros((args.num_steps, args.num_envs) + envs.single_observation_space.shape).to(device)
    actions = torch.zeros((args.num_steps, args.num_envs) + envs.single_action_space.shape).to(device)
    logprobs = torch.zeros((args.num_steps, args.num_envs)).to(device)
    rewards = torch.zeros((args.num_steps, args.num_envs)).to(device)
    dones = torch.zeros((args.num_steps, args.num_envs)).to(device)
    values = torch.zeros((args.num_steps, args.num_envs)).to(device)
    avg_returns = deque(maxlen=20)
    avg_returns_all = []

    # TRY NOT TO MODIFY: start the game
    global_step = 0
    start_time = time.time()
    next_obs = torch.Tensor(envs.reset()).to(device)
    next_done = torch.zeros(args.num_envs).to(device)
    num_updates = args.total_timesteps // args.batch_size

    prev_info = None

    if args.save_dataset_name is not None:
        dataset = {'obs':None, 'obs_adv':None, 'logits':None}

    for update in range(1, num_updates + 1):
        # Annealing the rate if instructed to do so.
        if args.anneal_lr:
            frac = 1.0 - (update - 1.0) / num_updates
            lrnow = frac * args.learning_rate

        for step in range(0, args.num_steps):
            global_step += 1 * args.num_envs
            obs[step] = next_obs
            dones[step] = next_done

            action, logits, logits_temp = get_action.cal(next_obs)

            if args.atk is not None:
                adv_next_obs = atk(next_obs / 255.0, action)
                action, logits, logits_temp = get_action.cal(adv_next_obs)
                if not atk_img_saved:
                    plt.imshow(next_obs.data.cpu()[0,0,:,:])
                    plt.savefig('figs/'+args.env_id+'_'+args.atk+'_ori')
                    plt.imshow(adv_next_obs.data.cpu()[0,0,:,:])
                    plt.savefig('figs/' + args.env_id + '_' + args.atk + '_attacked')
                    atk_img_saved = True

            if args.save_dataset_name is not None:
                if dataset['obs'] is None:
                    dataset['obs'] = next_obs
                    dataset['logits'] = logits_temp
                else:
                    dataset['obs'] = torch.cat((dataset['obs'],next_obs),0)
                    dataset['logits'] = torch.cat((dataset['logits'], logits_temp), 0)

            next_obs, reward, done, info = envs.step(action.data.cpu().numpy())
            rewards[step] = torch.tensor(reward).to(device).view(-1)
            next_obs, next_done = torch.Tensor(next_obs).to(device), torch.Tensor(done).to(device)

            for idx, d in enumerate(done):
                if d and info["lives"][idx] == 0:
                    print(f"global_step={global_step}, episodic_return={info['r'][idx]}")
                    avg_returns.append(info["r"][idx])
                    avg_returns_all.append(info["r"][idx])


                    pk.dump(avg_returns_all,open('saved_data/' + args.env_id+'_'+args.return_save_name, 'wb'))
            if len(avg_returns_all) > args.test_len:
                break
        if len(avg_returns_all) > args.test_len:
            break


        if (args.save_dataset_name is not None) and (len(dataset['obs'])>args.dataset_length):
            torch.save(dataset, 'dataset/'+ args.env_id+'_'+args.save_dataset_name)
            print ('Dataset Saved: '+'dataset/'+ args.env_id+'_'+args.save_dataset_name)
            break



    envs.close()