from org.orekit.bodies import OneAxisEllipsoid, CelestialBodyFactory
from org.orekit.forces.gravity import ThirdBodyAttraction
from org.orekit.forces.radiation import SolarRadiationPressure, IsotropicRadiationSingleCoefficient
from org.orekit.models.earth.atmosphere import NRLMSISE00, data as atmosphere_data
from org.orekit.forces.drag import IsotropicDrag, DragForce as Drag
from org.orekit.forces import FixedPanel, BoxAndSolarArraySpacecraft
from org.hipparchus.geometry.euclidean.threed import Vector3D

from constants import MOON_EQUATORIAL_RADIUS

class ThirdBodyForce(ThirdBodyAttraction):
    def __init__(self, name: str):
        allowed_bodies = {'SOLAR_SYSTEM_BARYCENTER', 'SUN', 'MERCURY', 'VENUS', 'EARTH_MOON', 'EARTH', 'MOON', 'MARS', 'JUPITER', 'SATURN', 'URANUS', 'NEPTUNE', 'PLUTO'}
        if name not in allowed_bodies:
            raise ValueError(f"Invalid third body with name '{name}'. Allowed names are: {', '.join(allowed_bodies)}")
        third_body = CelestialBodyFactory.getBody(name)
        super().__init__(third_body)

class SolarRadiationForce(SolarRadiationPressure):
    def __init__(self, earth: OneAxisEllipsoid, surface_area: float, reflection_coefficient: float = 0.8, is_starlink: bool = False):
        if is_starlink:
            # solar_panel = FixedPanel(Vector3D.MINUS_K, 8.0 * 2.8, True, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            # print(f'{solar_panel.getArea()}, {solar_panel.getDrag()}, {solar_panel.getAbsorption()}, {solar_panel.getReflection()}')
            # panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 2.8, 0.005, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            # panels = BoxAndSolarArraySpacecraft.buildBox(0.005, 2.8, 8.0, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            # panels = BoxAndSolarArraySpacecraft.buildBox(0.005, 8.0, 2.8, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            # panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 0.005, 2.8, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 1.0, 2.8, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            # panels = BoxAndSolarArraySpacecraft.buildPanels(2.8, 0.2, 1.4, CelestialBodyFactory.getSun(), 8.0 * 2.8, Vector3D.MINUS_K, 0.0, 0.0, 1 - reflection_coefficient, reflection_coefficient)
            body = BoxAndSolarArraySpacecraft(panels)
        else:
            body = IsotropicRadiationSingleCoefficient(surface_area, reflection_coefficient)
        super().__init__(CelestialBodyFactory.getSun(), earth, body)
        self.addOccultingBody(CelestialBodyFactory.getMoon(), MOON_EQUATORIAL_RADIUS)

class DragForce(Drag):
    def __init__(self, earth: OneAxisEllipsoid, surface_area: float, drag_coefficient: float, is_starlink: bool = False):
        weather_data = atmosphere_data.CssiSpaceWeatherData(atmosphere_data.CssiSpaceWeatherData.DEFAULT_SUPPORTED_NAMES)
        atmosphere = NRLMSISE00(weather_data, CelestialBodyFactory.getSun(), earth)
        if is_starlink:
            # panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 2.8, 0.005, drag_coefficient, 0.0, 0.0, 0.0)
            # panels = BoxAndSolarArraySpacecraft.buildBox(0.005, 2.8, 8.0, drag_coefficient, 0.0, 0.0, 0.0)
            # panels = BoxAndSolarArraySpacecraft.buildBox(0.005, 8.0, 2.8, drag_coefficient, 0.0, 0.0, 0.0)
            # panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 0.005, 2.8, drag_coefficient, 0.0, 0.0, 0.0)
            panels = BoxAndSolarArraySpacecraft.buildBox(8.0, 1.0, 2.8, drag_coefficient, 0.0, 0.0, 0.0)
            # panels = BoxAndSolarArraySpacecraft.buildPanels(2.8, 0.2, 1.4, CelestialBodyFactory.getSun(), 8.0 * 2.8, Vector3D.MINUS_K, drag_coefficient, 0.0, 0.0, 0.0)
            body = BoxAndSolarArraySpacecraft(panels)
        else:
            body = IsotropicDrag(surface_area, drag_coefficient)
        super().__init__(atmosphere, body)