import numpy as np
from env import OrbitZoo
from bodies import Satellite
from rl_algorithms.ppo import PPO
from constants import EARTH_RADIUS, INERTIAL_FRAME, MU
from math import radians
import random

from org.orekit.time import AbsoluteDate
from org.orekit.orbits import KeplerianOrbit, PositionAngleType

params = {
        "satellites": [
            {"name": "agent_1",
             "initial_state": {"x": 32299497.899668593, "y": 27102496.774823245, "z": 0.0, "x_dot": -1976.3573913582284, "y_dot": 2355.3310214012895, "z_dot": 0.0},
             "initial_state_uncertainty": {"x": 0.000001, "y": 0.000001, "z": 0.000001, "x_dot": 0.000001, "y_dot": 0.000001, "z_dot": 0.000001},
             "initial_mass": 200.0,
             "fuel_mass": 50.0,
             "isp": 3100.0,
             "radius": 5.0,
             "save_steps_info": False,
             "agent": {
                "lr_actor": 0.0001,
                "lr_critic": 0.001,
                "gae_lambda": 0.95,
                "epochs": 10,
                "gamma": 0.99,
                "clip": 0.2,
                "action_std_init": 0.05,
                "state_dim_actor": 8,
                "state_dim_critic": 8,
                "action_space": [5, 2*np.pi],
             }},
            {"name": "agent_2",
             "initial_state": {"x": 32299497.899668593, "y": 27102496.774823245, "z": 0.0, "x_dot": -1976.3573913582284, "y_dot": 2355.3310214012895, "z_dot": 0.0},
             "initial_state_uncertainty": {"x": 0.000001, "y": 0.000001, "z": 0.000001, "x_dot": 0.000001, "y_dot": 0.000001, "z_dot": 0.000001},
             "initial_mass": 200.0,
             "fuel_mass": 50.0,
             "isp": 3100.0,
             "radius": 5.0,
             "save_steps_info": False,
             "agent": {
                "lr_actor": 0.0001,
                "lr_critic": 0.001,
                "gae_lambda": 0.95,
                "epochs": 10,
                "gamma": 0.99,
                "clip": 0.2,
                "action_std_init": 0.05,
                "state_dim_actor": 8,
                "state_dim_critic": 8,
                "action_space": [5, 2*np.pi],
             }},
            {"name": "agent_3",
             "initial_state": {"x": 32299497.899668593, "y": 27102496.774823245, "z": 0.0, "x_dot": -1976.3573913582284, "y_dot": 2355.3310214012895, "z_dot": 0.0},
             "initial_state_uncertainty": {"x": 0.000001, "y": 0.000001, "z": 0.000001, "x_dot": 0.000001, "y_dot": 0.000001, "z_dot": 0.000001},
             "initial_mass": 200.0,
             "fuel_mass": 50.0,
             "isp": 3100.0,
             "radius": 5.0,
             "save_steps_info": False,
             "agent": {
                "lr_actor": 0.0001,
                "lr_critic": 0.001,
                "gae_lambda": 0.95,
                "epochs": 10,
                "gamma": 0.99,
                "clip": 0.2,
                "action_std_init": 0.05,
                "state_dim_actor": 8,
                "state_dim_critic": 8,
                "action_space": [5, 2*np.pi],
             }},
            {"name": "agent_4",
             "initial_state": {"x": 32299497.899668593, "y": 27102496.774823245, "z": 0.0, "x_dot": -1976.3573913582284, "y_dot": 2355.3310214012895, "z_dot": 0.0},
             "initial_state_uncertainty": {"x": 0.000001, "y": 0.000001, "z": 0.000001, "x_dot": 0.000001, "y_dot": 0.000001, "z_dot": 0.000001},
             "initial_mass": 200.0,
             "fuel_mass": 50.0,
             "isp": 3100.0,
             "radius": 5.0,
             "save_steps_info": False,
             "agent": {
                "lr_actor": 0.0001,
                "lr_critic": 0.001,
                "gae_lambda": 0.95,
                "epochs": 10,
                "gamma": 0.99,
                "clip": 0.2,
                "action_std_init": 0.05,
                "state_dim_actor": 8,
                "state_dim_critic": 8,
                "action_space": [5, 2*np.pi],
             }},
        ],
        "delta_t": 360.0,
        "forces": {
            "gravity_model":  "HolmesFeatherstone",
            "third_bodies": {
                "active": True,
                "bodies": ["SUN", "MOON"],
            },
            "solar_radiation_pressure": {
                "active": True,
                "reflection_coefficients": {
                    "agent_1": 1.0,
                    "agent_2": 1.0,
                    "agent_3": 1.0,
                    "agent_4": 1.0,
                }
            },
            "drag": {
                "active": True,
                "drag_coefficients": {
                    "agent_1": 1.0,
                    "agent_2": 1.0,
                    "agent_3": 1.0,
                    "agent_4": 1.0,
                }
            }
        },
        "interface": {
            "show": True,
            "delay_ms": 0,
            "zoom": 1.0,
            "drifters": {
                "show": True,
                "show_label": True,
                "show_velocity": False,
                "show_trail": True,
                "trail_last_steps": 50,
                "color_body": (255, 255, 255),
                "color_label": (255, 255, 255),
                "color_velocity": (255, 255, 255),
                "color_trail": (255, 255, 255),
            },
            "satellites": {
                "show": True,
                "show_label": True,
                "show_velocity": False,
                "show_thrust": True,
                "show_trail": True,
                "trail_last_steps": 50,
                "color_body": (255, 0, 0),
                "color_label": (255, 255, 255),
                "color_velocity": (255, 255, 255),
                "color_thrust": (0, 255, 0),
                "color_trail": (255, 0, 0),
            },
            "earth": {
                "show": True,
                "color": (0, 0, 255),
                "resolution": 30,
            },
            "equator_grid": {
                "show": False,
                "color": (30, 140, 200),
                "resolution": 10,
            },
            "timestamp": {
                "show": True,
            },
            "orbits": [
                {"a": 42164e3 - 6378e3, "e": 0.0001, "i": 0.0001, "pa": 20.0, "raan": 20.0, "color": (255, 0, 255)},
            ],
        }
    }

class Agent(Satellite):

    def __init__(self, params):
        super().__init__(params)
        self.ppo = PPO(params["agent"]["device"],
                       params["agent"]["state_dim_actor"],
                       params["agent"]["state_dim_critic"],
                       params["agent"]["action_dim"],
                       params["agent"]["lr_actor"],
                       params["agent"]["lr_critic"],
                       params["agent"]["gae_lambda"],
                       params["agent"]["gamma"],
                       params["agent"]["epochs"],
                       params["agent"]["clip"],
                       True,
                       params["agent"]["action_std_init"])
        
    def get_state(self):
        pass
    
    def __get_random_orbit__(self, seed = None):
        orbit = super().__get_random_orbit__()
        orbit = KeplerianOrbit(orbit.getPVCoordinates(), orbit.getFrame(), orbit.getDate(), orbit.getMu())
        orbit = KeplerianOrbit(orbit.getA(), orbit.getE(), orbit.getI(), orbit.getPerigeeArgument(), orbit.getRightAscensionOfAscendingNode(), radians(float(random.randint(0, 360))), PositionAngleType.MEAN, orbit.getFrame(), orbit.getDate(), orbit.getMu())
        return orbit
        
    def print_networks(self):
        print(self.ppo.policy.actor)
        print(self.ppo.policy.critic)

class MARLGEOEnv(OrbitZoo):

    def __init__(self, params):
        super().__init__(params)
        # geostationary orbit
        target = [42_164_000 - EARTH_RADIUS, 0.01, 0.01, 20.0, 20.0, 0.0]
        orbit = KeplerianOrbit(target[0] + EARTH_RADIUS, target[1], radians(target[2]), radians(target[3]), radians(target[4]), radians(target[5]), 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, PositionAngleType.TRUE, INERTIAL_FRAME, AbsoluteDate(), MU)
        self.target_orbit = np.array([orbit.getA(), orbit.getEquinoctialEx(), orbit.getEquinoctialEy()])
        
        target_anomalies = np.array([-np.pi, -1/2*np.pi, 0.0, 1/2*np.pi])
        tolerance_anomaly = 1/8*np.pi
        self.tolerances = [[anomaly - tolerance_anomaly, anomaly + tolerance_anomaly] for anomaly in target_anomalies]

    def create_bodies(self, params):
        self.drifters = []
        self.satellites = [Agent(body_params) for body_params in params["satellites"]]  if "satellites" in params else []

    def reset(self, seed=None, options=None):
        
        super().reset(seed)

        env.data = []

        observations = {}
        anomalies = [satellite.get_equinoctial_position()[5] + np.pi for satellite in self.satellites]
        for satellite in self.satellites:
            observations[satellite.name] = satellite.get_equinoctial_position()[:3] + [satellite.get_fuel()] + anomalies

        infos = {satellite.name: {} for satellite in self.satellites}

        return observations, infos

    def step(self, actions=None):

        # print(actions)
        clipped_actions = {str(satellite): np.clip(actions[str(satellite)], [-1,-1], [1,1]) for satellite in env.satellites}
        scaled_actions = {str(satellite): list(((clipped_actions[satellite.name] + 1) / 2) * high) + [0] for satellite in env.satellites}

        # print(scaled_actions)

        # states_before = {str(satellite): satellite.get_state() for satellite in env.satellites}

        states = {satellite.name: satellite.step(scaled_actions[satellite.name]) for satellite in env.satellites}

        observations = {}
        anomalies = [satellite.get_equinoctial_position()[5] + np.pi for satellite in self.satellites]
        for satellite in self.satellites:
            observations[satellite.name] = satellite.get_equinoctial_position()[:3] + [satellite.get_fuel()] + anomalies

        #rewards = {satellite.name: 0 for satellite in env.satellites}
        #terminations = {satellite.name: False for satellite in env.satellites}
        rewards, terminations = self.rewards(observations, scaled_actions)
        # print(rewards)
        truncations = {satellite.name: False for satellite in env.satellites}
        infos = {satellite.name: {} for satellite in env.satellites}
        return observations, rewards, terminations, truncations, infos
    
    def rewards(self, observations, actions):

        rewards = {}
        truncations = {}

        target_altitude = 42164e3
        target_distance = np.pi / 4

        # calculate anomaly separation penalty (same penalty for all agents)
        anomalies = observations[env.satellites[0].name][-4:]
        anomaly_penalty = 0
        total_pairs = 0
        for i, anomaly_i in enumerate(anomalies):
            for j, anomaly_j in enumerate(anomalies):
                if i < j:
                    angular_difference = np.abs(anomaly_i - anomaly_j) % (2 * np.pi)
                    angular_difference = min(angular_difference, 2 * np.pi - angular_difference)
                    # print(f"Angular difference between {i} and {j}: {angular_difference}")
                    if angular_difference < target_distance:
                        anomaly_penalty += (target_distance - angular_difference) / target_distance
                    total_pairs += 1
        avg_anomaly_penalty = anomaly_penalty / total_pairs if total_pairs > 0 else 0

        self.data.append(avg_anomaly_penalty)

        for satellite in self.satellites:
            # calculate altitude penalty (individual penalty for each agent)
            altitude = satellite.get_altitude()
            altitude_penalty = np.abs(altitude - target_altitude)
            # calculate action penalty
            action_penalty = actions[satellite.name][0]

            reward = - (1e-6*altitude_penalty + 1e-5*avg_anomaly_penalty + 100*action_penalty)
            # print(f'- (1*{altitude_penalty} + 1*{avg_anomaly_penalty}) = {reward}')
            rewards[satellite.name] = reward
            truncations[satellite.name] = False

        return rewards, truncations
    
    def rewards_old(self, observations):

        rewards = {}
        truncations = {}

        target_orbit = self.target_orbit

        for satellite in self.satellites:
            observation = observations[satellite.name]
            orbit = observation[:3]
            orbit_penalty = np.ones(3) @ ((orbit - target_orbit)**2 / target_orbit)
            reward = - (orbit_penalty / 1e7)
            rewards[satellite.name] = reward
            truncations[satellite.name] = False
        
        return rewards, truncations

env = MARLGEOEnv(params)

time_step = 1
action_std_decay_freq = 40000
action_std_decay_rate = 0.05
update_freq = 1024
min_action_std = 0.05
episodes = 10
steps_per_episode = 1000
low = np.array([0.0, 0.0])
high = np.array(params["satellites"][0]["agent"]["action_space"])

# load models
for i in range(len(env.satellites)):
    env.satellites[i].ppo.load(f".\\missions\\geo\\model_geo_ind_agent_{i}.pth")

best_score = -1e5
start_episode = 1

data = []

for episode in range(start_episode, start_episode + episodes):
    start_step = time_step
    current_ep_rewards = {str(satellite): 0 for satellite in env.satellites}
    observations, _ = env.reset()
    initial_pos = {satellite.name: satellite.get_cartesian_position() + satellite.get_cartesian_velocity() for satellite in env.satellites}
    # print({satellite.name: satellite.get_cartesian_position() + satellite.get_cartesian_velocity() for satellite in env.satellites})
    observations, _, _, _, _ = env.step({satellite.name: np.array([-1.0, 0.0]) for satellite in env.satellites})
    # positions = {satellite.name: satellite.get_}
    # data.append()

    for t in range(1, steps_per_episode + 1):
        try:

            if any(not satellite.has_fuel() for satellite in env.satellites):
                break

            # select actions with policies
            actions = {str(satellite): satellite.ppo.select_action(observations[str(satellite)]) for satellite in env.satellites}
            # actions = {satellite.name: np.array([-1.0, 0.5]) for satellite in env.satellites}

            # apply step
            observations, rewards, terminations, _, _ = env.step(actions)
            
            current_ep_rewards = {str(satellite): current_ep_rewards[str(satellite)] + rewards[str(satellite)] for satellite in env.satellites}

            if any(terminations[satellite.name] for satellite in env.satellites):
                break
            
        except Exception as e:
            break

        if params["interface"]["show"]:
            env.render()

    for satellite in env.satellites:
        satellite.ppo.buffer.clear()

    # show scores at the end of episode
    final_score = 0
    print(f"episode: {episode}")
