import numpy as np
from env_copy import OrbitZoo
from bodies import Satellite
from rl_algorithms.ppo_std import PPO

params = {
        "satellites": [
            # {"name": "agent",
            #  "initial_state": {"x": 6928137.0, "y": 0.0, "z": 0.0, "x_dot": 0.0, "y_dot": 7585.088535158763, "z_dot": 0.0},
            #  "initial_state_uncertainty": {"x": 1e-15, "y": 1e-15, "z": 1e-15, "x_dot": 1e-15, "y_dot": 1e-15, "z_dot": 1e-15},
            #  "initial_mass": 25.0,
            #  "fuel_mass": 75.0,
            #  "isp": 0.0067,
            #  "radius": 16.8,
            #  "save_steps_info": False,
            #  "agent": {
            #     "lr_actor": 0.0001,
            #     "lr_critic": 0.001,
            #     "gae_lambda": 0.95,
            #     "epochs": 5,
            #     "gamma": 0.99,
            #     "clip": 0.03,
            #     "action_std_init": 0.01,
            #     "state_dim_actor": 8,
            #     "state_dim_critic": 8,
            #     "action_space": [0.04 / 50, 2 * np.pi / 6],
            #  }},
        ],
        "delta_t": 1.0,
        "forces": {
            "gravity_model":  "Newtonian",
            "third_bodies": {
                "active": False,
                "bodies": ["SUN", "MOON"],
            },
            "solar_radiation_pressure": {
                "active": False,
                "reflection_coefficients": {
                    "agent": 0.5,
                }
            },
            "drag": {
                "active": True,
                "drag_coefficients": {
                    # "agent": 2.123,
                }
            }
        },
        "interface": {
            "show": False,
            "delay_ms": 0,
            "zoom": 1.0,
            "drifters": {
                "show": True,
                "show_label": True,
                "show_velocity": False,
                "show_trail": True,
                "trail_last_steps": 50,
                "color_body": (255, 255, 255),
                "color_label": (255, 255, 255),
                "color_velocity": (255, 255, 255),
                "color_trail": (255, 255, 255),
            },
            "satellites": {
                "show": True,
                "show_label": True,
                "show_velocity": False,
                "show_thrust": True,
                "show_trail": True,
                "trail_last_steps": 5000,
                "color_body": (255, 0, 0),
                "color_label": (255, 255, 255),
                "color_velocity": (255, 255, 255),
                "color_thrust": (0, 255, 0),
                "color_trail": (255, 0, 0),
            },
            "earth": {
                "show": True,
                "color": (0, 0, 255),
                "resolution": 70,
            },
            "equator_grid": {
                "show": False,
                "color": (30, 140, 200),
                "resolution": 10,
            },
            "timestamp": {
                "show": True,
            },
            "orbits": [
                {"a": 550.0e3, "e": 0.00001, "i": 0.00001, "pa": 0.0, "raan": 0.0, "color": (0, 255, 0)},
            ],
        }
    }

class Agent(Satellite):

    def __init__(self, params):
        super().__init__(params)
        self.ppo = PPO(params["agent"]["device"],
                       params["agent"]["state_dim_actor"],
                       params["agent"]["state_dim_critic"],
                       params["agent"]["action_dim"],
                       params["agent"]["lr_actor"],
                       params["agent"]["lr_critic"],
                       params["agent"]["gae_lambda"],
                       params["agent"]["gamma"],
                       params["agent"]["epochs"],
                       params["agent"]["clip"],
                       True,
                       params["agent"]["action_std_init"])
        
        self.thrust_mag = 0.0
        self.thrust_theta = 0.0

    def reset(self, seed=None):
        super().reset(seed)
        self.thrust_mag = 0.0
        self.thrust_theta = 0.0
        
    def get_state(self):
        position = np.array(self.get_cartesian_position())
        velocity = np.array(self.get_cartesian_velocity())
        pos_diff = np.abs(np.linalg.norm(position) - 6928137.0)
        vel_diff = np.abs(np.linalg.norm(velocity) - 7585.088535158763)
        return list(position[:2] / 6928137.0) + list(velocity[:2] / 7585.088535158763) + [pos_diff, vel_diff, self.thrust_theta, self.thrust_mag]

class HerreraEnvironment(OrbitZoo):

    def __init__(self, params):
        super().__init__(params)

    def create_bodies(self, params):
        self.drifters = []
        self.satellites = [Agent(body_params) for body_params in params["satellites"]]  if "satellites" in params else []

    def reset(self, seed=None, options=None):
        
        super().reset(seed)

        self.step_counter = 0

        observations = {satellite.name: satellite.get_state() for satellite in self.satellites}
        infos = {satellite.name: {} for satellite in self.satellites}

        return observations, infos

    def step(self, actions=None):
        self.step_counter += 1

        # Process only active agents
        clipped_actions = {
            str(satellite): np.clip(actions[str(satellite)], [-1, -1], [1, 1])
            for satellite in self.satellites
            if str(satellite) in actions
        }

        scaled_actions = {
            str(satellite): list(low + ((clipped_actions[str(satellite)] + 1) / 2) * (high - low)) + [0]
            for satellite in self.satellites
            if str(satellite) in clipped_actions
        }

        for satellite in self.satellites:
            name = str(satellite)
            if name in scaled_actions:
                action = scaled_actions[name]
                satellite.thrust_mag += action[0] * 0.04 - 0.02
                satellite.thrust_mag = np.clip(satellite.thrust_mag, 0.0, 1.0)
                satellite.thrust_theta += action[1] * np.pi / 3 - np.pi / 6
                satellite.thrust_theta -= (2 * np.pi) * np.floor((satellite.thrust_theta + np.pi) * (1 / (2 * np.pi)))

        observations = {
            sat.name: sat.step(np.array([sat.thrust_mag, sat.thrust_theta, 0.0]))
            for sat in self.satellites
            if str(sat) in actions
        }

        rewards, terminations = self.rewards(observations)

        truncations = {sat: False for sat in observations}
        infos = {sat: {} for sat in observations}

        return observations, rewards, terminations, truncations, infos


    def rewards(self, observations):
        rewards = {}
        terminations = {}

        for name, observation in observations.items():
            sat = next(s for s in self.satellites if str(s) == name)
            termination = observation[4] > 1.0 or not sat.has_fuel()
            reward = 0 if termination else (self.step_counter / 800) + 0.5
            rewards[name] = reward
            terminations[name] = termination

        return rewards, terminations


for i in range(100):
    name = f"agent_{i}"
    params["satellites"].append(
            {"name": name,
             "initial_state": {"x": 6928137.0, "y": 0.0, "z": 0.0, "x_dot": 0.0, "y_dot": 7585.088535158763, "z_dot": 0.0},
             "initial_state_uncertainty": {"x": 1e-15, "y": 1e-15, "z": 1e-15, "x_dot": 1e-15, "y_dot": 1e-15, "z_dot": 1e-15},
             "initial_mass": 25.0,
             "fuel_mass": 75.0,
             "isp": 0.0067,
             "radius": 16.8,
             "save_steps_info": False,
             "agent": {
                "lr_actor": 0.0001,
                "lr_critic": 0.001,
                "gae_lambda": 0.95,
                "epochs": 5,
                "gamma": 0.99,
                "clip": 0.03,
                "action_std_init": 0.01,
                "state_dim_actor": 8,
                "state_dim_critic": 8,
                "action_space": [0.04 / 50, 2 * np.pi / 6],
             }})
    params["forces"]["drag"]["drag_coefficients"][name] = 2.123

env = HerreraEnvironment(params)

for satellite in env.satellites:
    satellite.ppo.load("model_checkpoint_eval_herrera3.pth")

episodes = 9000
steps_per_episode = 800
low = np.array([0.0, 0.0])
high = np.array([1.0, 1.0])
best_score = 1e8

for episode in range(episodes):
    observations, _ = env.reset(42)
    scores_episode = {str(satellite): 0 for satellite in env.satellites}
    done = {str(satellite): False for satellite in env.satellites}
    
    for _ in range(1, steps_per_episode + 1):
        actions = {
            str(satellite): satellite.ppo.select_action(observations[str(satellite)])
            for satellite in env.satellites
            if not done[str(satellite)]
        }

        # print(len(actions))

        if not actions:  # All agents are done
            break

        observations, rewards, terminations, _, _ = env.step(actions)

        # print(any(terminations[str(satellite)] for satellite in env.satellites))

        for satellite in env.satellites:
            sat_id = str(satellite)
            if not done[sat_id]:
                scores_episode[sat_id] += rewards[sat_id]
                if terminations[sat_id]:
                    done[sat_id] = True

    
    scores = []
    for satellite in env.satellites:
        satellite.ppo.buffer.clear()
        scores.append(scores_episode[str(satellite)])
    print(scores)

# print(f'EVAL >>> mean: {np.mean(scores)}, std: {np.std(scores)}')